"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LazyRole = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const role_1 = require("./role");
/**
 * An IAM role that only gets attached to the construct tree once it gets used, not before
 *
 * This construct can be used to simplify logic in other constructs
 * which need to create a role but only if certain configurations occur
 * (such as when AutoScaling is configured). The role can be configured in one
 * place, but if it never gets used it doesn't get instantiated and will
 * not be synthesized or deployed.
 *
 * @resource AWS::IAM::Role
 */
class LazyRole extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.grantPrincipal = this;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.statements = new Array();
        this.policies = new Array();
        this.managedPolicies = new Array();
    }
    /**
     * Adds a permission to the role's default policy document.
     * If there is no default policy attached to this role, it will be created.
     * @param statement The permission statement to add to the policy document
     */
    addToPrincipalPolicy(statement) {
        if (this.role) {
            return this.role.addToPrincipalPolicy(statement);
        }
        else {
            this.statements.push(statement);
            return { statementAdded: true, policyDependable: this };
        }
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    /**
     * Attaches a policy to this role.
     * @param policy The policy to attach
     */
    attachInlinePolicy(policy) {
        if (this.role) {
            this.role.attachInlinePolicy(policy);
        }
        else {
            this.policies.push(policy);
        }
    }
    /**
     * Attaches a managed policy to this role.
     * @param policy The managed policy to attach.
     */
    addManagedPolicy(policy) {
        if (this.role) {
            this.role.addManagedPolicy(policy);
        }
        else {
            this.managedPolicies.push(policy);
        }
    }
    /**
     * Returns the ARN of this role.
     */
    get roleArn() {
        return this.instantiate().roleArn;
    }
    /**
     * Returns the stable and unique string identifying the role (i.e. AIDAJQABLZS4A3QDU576Q)
     *
     * @attribute
     */
    get roleId() {
        return this.instantiate().roleId;
    }
    get roleName() {
        return this.instantiate().roleName;
    }
    get policyFragment() {
        return this.instantiate().policyFragment;
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(identity, ...actions) {
        return this.instantiate().grant(identity, ...actions);
    }
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(identity) {
        return this.instantiate().grantPassRole(identity);
    }
    instantiate() {
        if (!this.role) {
            const role = new role_1.Role(this, 'Default', this.props);
            this.statements.forEach(role.addToPolicy.bind(role));
            this.policies.forEach(role.attachInlinePolicy.bind(role));
            this.managedPolicies.forEach(role.addManagedPolicy.bind(role));
            this.role = role;
        }
        return this.role;
    }
}
exports.LazyRole = LazyRole;
//# sourceMappingURL=data:application/json;base64,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