"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsTask = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const ecs = require("../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const util_1 = require("./util");
/**
 * Start a task on an ECS cluster
 */
class EcsTask {
    constructor(props) {
        this.props = props;
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        this.cluster = props.cluster;
        this.taskDefinition = props.taskDefinition;
        this.taskCount = props.taskCount !== undefined ? props.taskCount : 1;
        // Security groups are only configurable with the "awsvpc" network mode.
        if (this.taskDefinition.networkMode !== ecs.NetworkMode.AWS_VPC) {
            if (props.securityGroup !== undefined || props.securityGroups !== undefined) {
                this.taskDefinition.node.addWarning('security groups are ignored when network mode is not awsvpc');
            }
            return;
        }
        if (props.securityGroups) {
            this.securityGroups = props.securityGroups;
            return;
        }
        let securityGroup = props.securityGroup || this.taskDefinition.node.tryFindChild('SecurityGroup');
        securityGroup = securityGroup || new ec2.SecurityGroup(this.taskDefinition, 'SecurityGroup', { vpc: this.props.cluster.vpc });
        this.securityGroup = securityGroup; // Maintain backwards-compatibility for customers that read the generated security group.
        this.securityGroups = [securityGroup];
    }
    /**
     * Allows using tasks as target of EventBridge events
     */
    bind(_rule, _id) {
        const policyStatements = [new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.taskDefinition.taskDefinitionArn],
                conditions: {
                    ArnEquals: { 'ecs:cluster': this.cluster.clusterArn },
                },
            })];
        // If it so happens that a Task Execution Role was created for the TaskDefinition,
        // then the EventBridge Role must have permissions to pass it (otherwise it doesn't).
        if (this.taskDefinition.executionRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.executionRole.roleArn],
            }));
        }
        // For Fargate task we need permission to pass the task role.
        if (this.taskDefinition.isFargateCompatible) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.taskRole.roleArn],
            }));
        }
        const arn = this.cluster.clusterArn;
        const role = util_1.singletonEventRole(this.taskDefinition, policyStatements);
        const containerOverrides = this.props.containerOverrides && this.props.containerOverrides
            .map(({ containerName, ...overrides }) => ({ name: containerName, ...overrides }));
        const input = { containerOverrides };
        const taskCount = this.taskCount;
        const taskDefinitionArn = this.taskDefinition.taskDefinitionArn;
        const subnetSelection = this.props.subnetSelection || { subnetType: ec2.SubnetType.PRIVATE };
        const assignPublicIp = subnetSelection.subnetType === ec2.SubnetType.PUBLIC ? 'ENABLED' : 'DISABLED';
        const baseEcsParameters = { taskCount, taskDefinitionArn };
        const ecsParameters = this.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC
            ? {
                ...baseEcsParameters,
                launchType: this.taskDefinition.isEc2Compatible ? 'EC2' : 'FARGATE',
                networkConfiguration: {
                    awsVpcConfiguration: {
                        subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                        assignPublicIp,
                        securityGroups: this.securityGroups && this.securityGroups.map(sg => sg.securityGroupId),
                    },
                },
            }
            : baseEcsParameters;
        return {
            id: '',
            arn,
            role,
            ecsParameters,
            input: events.RuleTargetInput.fromObject(input),
            targetResource: this.taskDefinition,
        };
    }
}
exports.EcsTask = EcsTask;
//# sourceMappingURL=data:application/json;base64,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