"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Nodegroup = exports.NodegroupAmiType = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const eks_generated_1 = require("./eks.generated");
/**
 * The AMI type for your node group. GPU instance types should use the `AL2_x86_64_GPU` AMI type, which uses the
 * Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the `AL2_x86_64` AMI type, which
 * uses the Amazon EKS-optimized Linux AMI.
 */
var NodegroupAmiType;
(function (NodegroupAmiType) {
    /**
     * Amazon Linux 2
     */
    NodegroupAmiType["AL2_X86_64"] = "AL2_x86_64";
    /**
     * Amazon Linux 2 with GPU support
     */
    NodegroupAmiType["AL2_X86_64_GPU"] = "AL2_x86_64_GPU";
})(NodegroupAmiType = exports.NodegroupAmiType || (exports.NodegroupAmiType = {}));
/**
 * The Nodegroup resource class
 */
class Nodegroup extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.nodegroupName,
        });
        this.cluster = props.cluster;
        this.desiredSize = (_b = (_a = props.desiredSize) !== null && _a !== void 0 ? _a : props.minSize) !== null && _b !== void 0 ? _b : 2;
        this.maxSize = (_c = props.maxSize) !== null && _c !== void 0 ? _c : this.desiredSize;
        this.minSize = (_d = props.minSize) !== null && _d !== void 0 ? _d : 1;
        if (this.desiredSize > this.maxSize) {
            throw new Error(`Desired capacity ${this.desiredSize} can't be greater than max size ${this.maxSize}`);
        }
        if (this.desiredSize < this.minSize) {
            throw new Error(`Minimum capacity ${this.minSize} can't be greater than desired size ${this.desiredSize}`);
        }
        if (!props.nodeRole) {
            const ngRole = new aws_iam_1.Role(this, 'NodeGroupRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            });
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
            this.role = ngRole;
        }
        else {
            this.role = props.nodeRole;
        }
        const resource = new eks_generated_1.CfnNodegroup(this, 'Resource', {
            clusterName: this.cluster.clusterName,
            nodegroupName: props.nodegroupName,
            nodeRole: this.role.roleArn,
            subnets: this.cluster.vpc.selectSubnets(props.subnets).subnetIds,
            amiType: props.amiType,
            diskSize: props.diskSize,
            forceUpdateEnabled: (_e = props.forceUpdate) !== null && _e !== void 0 ? _e : true,
            instanceTypes: props.instanceType ? [props.instanceType.toString()] : undefined,
            labels: props.labels,
            releaseVersion: props.releaseVersion,
            remoteAccess: props.remoteAccess ? {
                ec2SshKey: props.remoteAccess.sshKeyName,
                sourceSecurityGroups: props.remoteAccess.sourceSecurityGroups ?
                    props.remoteAccess.sourceSecurityGroups.map(m => m.securityGroupId) : undefined,
            } : undefined,
            scalingConfig: {
                desiredSize: this.desiredSize,
                maxSize: this.maxSize,
                minSize: this.minSize,
            },
            tags: props.tags,
        });
        // As managed nodegroup will auto map the instance role to RBAC behind the scene and users don't have to manually
        // do it anymore. We don't need to print out the instance role arn now.
        this.nodegroupArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'eks',
            resource: 'nodegroup',
            resourceName: this.physicalName,
        });
        this.nodegroupName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import the Nodegroup from attributes
     */
    static fromNodegroupName(scope, id, nodegroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.nodegroupName = nodegroupName;
            }
        }
        return new Import(scope, id);
    }
}
exports.Nodegroup = Nodegroup;
//# sourceMappingURL=data:application/json;base64,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