"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pipeline = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("./action");
const codepipeline_generated_1 = require("./codepipeline.generated");
const cross_region_support_stack_1 = require("./cross-region-support-stack");
const full_action_descriptor_1 = require("./full-action-descriptor");
const stage_1 = require("./stage");
const validation_1 = require("./validation");
class PipelineBase extends core_1.Resource {
    /**
     * Defines an event rule triggered by this CodePipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codepipeline'],
            resources: [this.pipelineArn],
        });
        return rule;
    }
    /**
     * Defines an event rule triggered by the "CodePipeline Pipeline Execution
     * State Change" event emitted from this pipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodePipeline Pipeline Execution State Change'],
        });
        return rule;
    }
}
/**
 * An AWS CodePipeline pipeline with its associated IAM role and S3 bucket.
 *
 * @example
 * // create a pipeline
 * const pipeline = new Pipeline(this, 'Pipeline');
 *
 * // add a stage
 * const sourceStage = pipeline.addStage({ name: 'Source' });
 *
 * // add a source action to the stage
 * sourceStage.addAction(new codepipeline_actions.CodeCommitSourceAction({
 *   actionName: 'Source',
 *   outputArtifactName: 'SourceArtifact',
 *   repository: repo,
 * }));
 *
 * // ... add more stages
 */
class Pipeline extends PipelineBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.pipelineName,
        });
        this._stages = new Array();
        this._crossRegionSupport = {};
        this._crossAccountSupport = {};
        validation_1.validateName('Pipeline', this.physicalName);
        // only one of artifactBucket and crossRegionReplicationBuckets can be supplied
        if (props.artifactBucket && props.crossRegionReplicationBuckets) {
            throw new Error('Only one of artifactBucket and crossRegionReplicationBuckets can be specified!');
        }
        // If a bucket has been provided, use it - otherwise, create a bucket.
        let propsBucket = this.getArtifactBucketFromProps(props);
        if (!propsBucket) {
            const encryptionKey = new kms.Key(this, 'ArtifactsBucketEncryptionKey', {
                // remove the key - there is a grace period of a few days before it's gone for good,
                // that should be enough for any emergency access to the bucket artifacts
                removalPolicy: core_1.RemovalPolicy.DESTROY,
            });
            propsBucket = new s3.Bucket(this, 'ArtifactsBucket', {
                bucketName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                encryptionKey,
                encryption: s3.BucketEncryption.KMS,
                blockPublicAccess: new s3.BlockPublicAccess(s3.BlockPublicAccess.BLOCK_ALL),
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
            // add an alias to make finding the key in the console easier
            new kms.Alias(this, 'ArtifactsBucketEncryptionKeyAlias', {
                aliasName: this.generateNameForDefaultBucketKeyAlias(),
                targetKey: encryptionKey,
                removalPolicy: core_1.RemovalPolicy.DESTROY,
            });
        }
        this.artifactBucket = propsBucket;
        // If a role has been provided, use it - otherwise, create a role.
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com'),
        });
        const codePipeline = new codepipeline_generated_1.CfnPipeline(this, 'Resource', {
            artifactStore: core_1.Lazy.anyValue({ produce: () => this.renderArtifactStoreProperty() }),
            artifactStores: core_1.Lazy.anyValue({ produce: () => this.renderArtifactStoresProperty() }),
            stages: core_1.Lazy.anyValue({ produce: () => this.renderStages() }),
            roleArn: this.role.roleArn,
            restartExecutionOnUpdate: props && props.restartExecutionOnUpdate,
            name: this.physicalName,
        });
        // this will produce a DependsOn for both the role and the policy resources.
        codePipeline.node.addDependency(this.role);
        this.artifactBucket.grantReadWrite(this.role);
        this.pipelineName = this.getResourceNameAttribute(codePipeline.ref);
        this.pipelineVersion = codePipeline.attrVersion;
        this.crossRegionBucketsPassed = !!props.crossRegionReplicationBuckets;
        for (const [region, replicationBucket] of Object.entries(props.crossRegionReplicationBuckets || {})) {
            this._crossRegionSupport[region] = {
                replicationBucket,
                stack: core_1.Stack.of(replicationBucket),
            };
        }
        // Does not expose a Fn::GetAtt for the ARN so we'll have to make it ourselves
        this.pipelineArn = core_1.Stack.of(this).formatArn({
            service: 'codepipeline',
            resource: this.pipelineName,
        });
        for (const stage of props.stages || []) {
            this.addStage(stage);
        }
    }
    /**
     * Import a pipeline into this app.
     *
     * @param scope the scope into which to import this pipeline
     * @param id the logical ID of the returned pipeline construct
     * @param pipelineArn The ARN of the pipeline (e.g. `arn:aws:codepipeline:us-east-1:123456789012:MyDemoPipeline`)
     */
    static fromPipelineArn(scope, id, pipelineArn) {
        class Import extends PipelineBase {
            constructor() {
                super(...arguments);
                this.pipelineName = core_1.Stack.of(scope).parseArn(pipelineArn).resource;
                this.pipelineArn = pipelineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Creates a new Stage, and adds it to this Pipeline.
     *
     * @param props the creation properties of the new Stage
     * @returns the newly created Stage
     */
    addStage(props) {
        // check for duplicate Stages and names
        if (this._stages.find(s => s.stageName === props.stageName)) {
            throw new Error(`Stage with duplicate name '${props.stageName}' added to the Pipeline`);
        }
        const stage = new stage_1.Stage(props, this);
        const index = props.placement
            ? this.calculateInsertIndexFromPlacement(props.placement)
            : this.stageCount;
        this._stages.splice(index, 0, stage);
        return stage;
    }
    /**
     * Adds a statement to the pipeline role.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Get the number of Stages in this Pipeline.
     */
    get stageCount() {
        return this._stages.length;
    }
    /**
     * Returns the stages that comprise the pipeline.
     *
     * **Note**: the returned array is a defensive copy,
     * so adding elements to it has no effect.
     * Instead, use the {@link addStage} method if you want to add more stages
     * to the pipeline.
     */
    get stages() {
        return this._stages.slice();
    }
    /**
     * Returns all of the {@link CrossRegionSupportStack}s that were generated automatically
     * when dealing with Actions that reside in a different region than the Pipeline itself.
     *
     * @experimental
     */
    get crossRegionSupport() {
        const ret = {};
        Object.keys(this._crossRegionSupport).forEach((key) => {
            ret[key] = this._crossRegionSupport[key];
        });
        return ret;
    }
    /** @internal */
    _attachActionToPipeline(stage, action, actionScope) {
        // handle cross-region actions here
        const crossRegionInfo = this.ensureReplicationResourcesExistFor(action);
        // get the role for the given action
        const actionRole = this.getRoleForAction(stage, action, actionScope);
        // // CodePipeline Variables
        validation_1.validateNamespaceName(action.actionProperties.variablesNamespace);
        // bind the Action
        const actionConfig = action.bind(actionScope, stage, {
            role: actionRole ? actionRole : this.role,
            bucket: crossRegionInfo.artifactBucket,
        });
        return new full_action_descriptor_1.FullActionDescriptor({
            action,
            actionConfig,
            actionRole,
            actionRegion: crossRegionInfo.region,
        });
    }
    /**
     * Validate the pipeline structure
     *
     * Validation happens according to the rules documented at
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements
     * @override
     */
    validate() {
        return [
            ...this.validateSourceActionLocations(),
            ...this.validateHasStages(),
            ...this.validateStages(),
            ...this.validateArtifacts(),
        ];
    }
    ensureReplicationResourcesExistFor(action) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRegion;
        let otherStack;
        const actionResource = action.actionProperties.resource;
        if (actionResource) {
            const actionResourceStack = core_1.Stack.of(actionResource);
            if (pipelineStack.region !== actionResourceStack.region) {
                actionRegion = actionResourceStack.region;
                // if the resource is from a different stack in another region but the same account,
                // use that stack as home for the cross-region support resources
                if (pipelineStack.account === actionResourceStack.account) {
                    otherStack = actionResourceStack;
                }
            }
        }
        else {
            actionRegion = action.actionProperties.region;
        }
        // if actionRegion is undefined,
        // it means the action is in the same region as the pipeline -
        // so, just return the artifactBucket
        if (!actionRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // get the region the Pipeline itself is in
        const pipelineRegion = this.requireRegion();
        // if the action is in the same region as the pipeline, nothing to do
        if (actionRegion === pipelineRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // source actions have to be in the same region as the pipeline
        if (action.actionProperties.category === action_1.ActionCategory.SOURCE) {
            throw new Error(`Source action '${action.actionProperties.actionName}' must be in the same region as the pipeline`);
        }
        // check whether we already have a bucket in that region,
        // either passed from the outside or previously created
        let crossRegionSupport = this._crossRegionSupport[actionRegion];
        if (!crossRegionSupport) {
            // we need to create scaffolding resources for this region
            crossRegionSupport = this.createSupportResourcesForRegion(otherStack, actionRegion);
            this._crossRegionSupport[actionRegion] = crossRegionSupport;
        }
        // the stack containing the replication bucket must be deployed before the pipeline
        pipelineStack.addDependency(crossRegionSupport.stack);
        crossRegionSupport.replicationBucket.grantReadWrite(this.role);
        return {
            artifactBucket: crossRegionSupport.replicationBucket,
            region: actionRegion,
        };
    }
    createSupportResourcesForRegion(otherStack, actionRegion) {
        // if we have a stack from the resource passed - use that!
        if (otherStack) {
            // check if the stack doesn't have this magic construct already
            const id = `CrossRegionReplicationSupport-d823f1d8-a990-4e5c-be18-4ac698532e65-${actionRegion}`;
            let crossRegionSupportConstruct = otherStack.node.tryFindChild(id);
            if (!crossRegionSupportConstruct) {
                crossRegionSupportConstruct = new cross_region_support_stack_1.CrossRegionSupportConstruct(otherStack, id);
            }
            return {
                replicationBucket: crossRegionSupportConstruct.replicationBucket,
                stack: otherStack,
            };
        }
        // otherwise - create a stack with the resources needed for replication across regions
        const pipelineStack = core_1.Stack.of(this);
        const pipelineAccount = pipelineStack.account;
        if (core_1.Token.isUnresolved(pipelineAccount)) {
            throw new Error("You need to specify an explicit account when using CodePipeline's cross-region support");
        }
        const app = this.requireApp();
        const supportStackId = `cross-region-stack-${pipelineAccount}:${actionRegion}`;
        let supportStack = app.node.tryFindChild(supportStackId);
        if (!supportStack) {
            supportStack = new cross_region_support_stack_1.CrossRegionSupportStack(app, supportStackId, {
                pipelineStackName: pipelineStack.stackName,
                region: actionRegion,
                account: pipelineAccount,
                synthesizer: this.getCrossRegionSupportSynthesizer(),
            });
        }
        return {
            stack: supportStack,
            replicationBucket: supportStack.replicationBucket,
        };
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the pipeline account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer.cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
    generateNameForDefaultBucketKeyAlias() {
        const prefix = 'alias/codepipeline-';
        const maxAliasLength = 256;
        const uniqueId = this.node.uniqueId;
        // take the last 256 - (prefix length) characters of uniqueId
        const startIndex = Math.max(0, uniqueId.length - (maxAliasLength - prefix.length));
        return prefix + uniqueId.substring(startIndex).toLowerCase();
    }
    /**
     * Gets the role used for this action,
     * including handling the case when the action is supposed to be cross-account.
     *
     * @param stage the stage the action belongs to
     * @param action the action to return/create a role for
     * @param actionScope the scope, unique to the action, to create new resources in
     */
    getRoleForAction(stage, action, actionScope) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRole = this.getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action);
        if (!actionRole && this.isAwsOwned(action)) {
            // generate a Role for this specific Action
            actionRole = new iam.Role(actionScope, 'CodePipelineActionRole', {
                assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            });
        }
        // the pipeline role needs assumeRole permissions to the action role
        if (actionRole) {
            this.role.addToPolicy(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                resources: [actionRole.roleArn],
            }));
        }
        return actionRole;
    }
    getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action) {
        const pipelineStack = core_1.Stack.of(this);
        // if a Role has been passed explicitly, always use it
        // (even if the backing resource is from a different account -
        // this is how the user can override our default support logic)
        if (action.actionProperties.role) {
            if (this.isAwsOwned(action)) {
                // the role has to be deployed before the pipeline
                // (our magical cross-stack dependencies will not work,
                // because the role might be from a different environment),
                // but _only_ if it's a new Role -
                // an imported Role should not add the dependency
                if (action.actionProperties.role instanceof iam.Role) {
                    const roleStack = core_1.Stack.of(action.actionProperties.role);
                    pipelineStack.addDependency(roleStack);
                }
                return action.actionProperties.role;
            }
            else {
                // ...except if the Action is not owned by 'AWS',
                // as that would be rejected by CodePipeline at deploy time
                throw new Error("Specifying a Role is not supported for actions with an owner different than 'AWS' - " +
                    `got '${action.actionProperties.owner}' (Action: '${action.actionProperties.actionName}' in Stage: '${stage.stageName}')`);
            }
        }
        // if we don't have a Role passed,
        // and the action is cross-account,
        // generate a Role in that other account stack
        const otherAccountStack = this.getOtherStackIfActionIsCrossAccount(action);
        if (!otherAccountStack) {
            return undefined;
        }
        // if we have a cross-account action, the pipeline's bucket must have a KMS key
        if (!this.artifactBucket.encryptionKey) {
            throw new Error('The Pipeline is being used in a cross-account manner, ' +
                'but its artifact bucket does not have a KMS key defined. ' +
                'A KMS key is required for a cross-account Pipeline. ' +
                'Make sure to pass a Bucket with a Key when creating the Pipeline');
        }
        // generate a role in the other stack, that the Pipeline will assume for executing this action
        const ret = new iam.Role(otherAccountStack, `${this.node.uniqueId}-${stage.stageName}-${action.actionProperties.actionName}-ActionRole`, {
            assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
        });
        // the other stack with the role has to be deployed before the pipeline stack
        // (CodePipeline verifies you can assume the action Role on creation)
        pipelineStack.addDependency(otherAccountStack);
        return ret;
    }
    /**
     * Returns the Stack this Action belongs to if this is a cross-account Action.
     * If this Action is not cross-account (i.e., it lives in the same account as the Pipeline),
     * it returns undefined.
     *
     * @param action the Action to return the Stack for
     */
    getOtherStackIfActionIsCrossAccount(action) {
        const pipelineStack = core_1.Stack.of(this);
        if (action.actionProperties.resource) {
            const resourceStack = core_1.Stack.of(action.actionProperties.resource);
            // check if resource is from a different account
            if (pipelineStack.account === resourceStack.account) {
                return undefined;
            }
            else {
                this._crossAccountSupport[resourceStack.account] = resourceStack;
                return resourceStack;
            }
        }
        if (!action.actionProperties.account) {
            return undefined;
        }
        const targetAccount = action.actionProperties.account;
        // check whether the account is a static string
        if (core_1.Token.isUnresolved(targetAccount)) {
            throw new Error(`The 'account' property must be a concrete value (action: '${action.actionProperties.actionName}')`);
        }
        // check whether the pipeline account is a static string
        if (core_1.Token.isUnresolved(pipelineStack.account)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set account');
        }
        if (pipelineStack.account === targetAccount) {
            return undefined;
        }
        let targetAccountStack = this._crossAccountSupport[targetAccount];
        if (!targetAccountStack) {
            const stackId = `cross-account-support-stack-${targetAccount}`;
            const app = this.requireApp();
            targetAccountStack = app.node.tryFindChild(stackId);
            if (!targetAccountStack) {
                targetAccountStack = new core_1.Stack(app, stackId, {
                    stackName: `${pipelineStack.stackName}-support-${targetAccount}`,
                    env: {
                        account: targetAccount,
                        region: action.actionProperties.region ? action.actionProperties.region : pipelineStack.region,
                    },
                });
            }
            this._crossAccountSupport[targetAccount] = targetAccountStack;
        }
        return targetAccountStack;
    }
    isAwsOwned(action) {
        const owner = action.actionProperties.owner;
        return !owner || owner === 'AWS';
    }
    getArtifactBucketFromProps(props) {
        if (props.artifactBucket) {
            return props.artifactBucket;
        }
        if (props.crossRegionReplicationBuckets) {
            const pipelineRegion = this.requireRegion();
            return props.crossRegionReplicationBuckets[pipelineRegion];
        }
        return undefined;
    }
    calculateInsertIndexFromPlacement(placement) {
        // check if at most one placement property was provided
        const providedPlacementProps = ['rightBefore', 'justAfter', 'atIndex']
            .filter((prop) => placement[prop] !== undefined);
        if (providedPlacementProps.length > 1) {
            throw new Error('Error adding Stage to the Pipeline: ' +
                'you can only provide at most one placement property, but ' +
                `'${providedPlacementProps.join(', ')}' were given`);
        }
        if (placement.rightBefore !== undefined) {
            const targetIndex = this.findStageIndex(placement.rightBefore);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it before, '${placement.rightBefore.stageName}', was not found`);
            }
            return targetIndex;
        }
        if (placement.justAfter !== undefined) {
            const targetIndex = this.findStageIndex(placement.justAfter);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it after, '${placement.justAfter.stageName}', was not found`);
            }
            return targetIndex + 1;
        }
        return this.stageCount;
    }
    findStageIndex(targetStage) {
        return this._stages.findIndex(stage => stage === targetStage);
    }
    validateSourceActionLocations() {
        const errors = new Array();
        let firstStage = true;
        for (const stage of this._stages) {
            const onlySourceActionsPermitted = firstStage;
            for (const action of stage.actionDescriptors) {
                errors.push(...validation_1.validateSourceAction(onlySourceActionsPermitted, action.category, action.actionName, stage.stageName));
            }
            firstStage = false;
        }
        return errors;
    }
    validateHasStages() {
        if (this.stageCount < 2) {
            return ['Pipeline must have at least two stages'];
        }
        return [];
    }
    validateStages() {
        const ret = new Array();
        for (const stage of this._stages) {
            ret.push(...stage.validate());
        }
        return ret;
    }
    validateArtifacts() {
        const ret = new Array();
        const producers = {};
        const firstConsumers = {};
        for (const [stageIndex, stage] of enumerate(this._stages)) {
            // For every output artifact, get the producer
            for (const action of stage.actionDescriptors) {
                const actionLoc = new PipelineLocation(stageIndex, stage, action);
                for (const outputArtifact of action.outputs) {
                    // output Artifacts always have a name set
                    const name = outputArtifact.artifactName;
                    if (producers[name]) {
                        ret.push(`Both Actions '${producers[name].actionName}' and '${action.actionName}' are producting Artifact '${name}'. Every artifact can only be produced once.`);
                        continue;
                    }
                    producers[name] = actionLoc;
                }
                // For every input artifact, get the first consumer
                for (const inputArtifact of action.inputs) {
                    const name = inputArtifact.artifactName;
                    if (!name) {
                        ret.push(`Action '${action.actionName}' is using an unnamed input Artifact, which is not being produced in this pipeline`);
                        continue;
                    }
                    firstConsumers[name] = firstConsumers[name] ? firstConsumers[name].first(actionLoc) : actionLoc;
                }
            }
        }
        // Now validate that every input artifact is produced before it's
        // being consumed.
        for (const [artifactName, consumerLoc] of Object.entries(firstConsumers)) {
            const producerLoc = producers[artifactName];
            if (!producerLoc) {
                ret.push(`Action '${consumerLoc.actionName}' is using input Artifact '${artifactName}', which is not being produced in this pipeline`);
                continue;
            }
            if (consumerLoc.beforeOrEqual(producerLoc)) {
                ret.push(`${consumerLoc} is consuming input Artifact '${artifactName}' before it is being produced at ${producerLoc}`);
            }
        }
        return ret;
    }
    renderArtifactStoresProperty() {
        if (!this.crossRegion) {
            return undefined;
        }
        // add the Pipeline's artifact store
        const primaryRegion = this.requireRegion();
        this._crossRegionSupport[primaryRegion] = {
            replicationBucket: this.artifactBucket,
            stack: core_1.Stack.of(this),
        };
        return Object.entries(this._crossRegionSupport).map(([region, support]) => ({
            region,
            artifactStore: this.renderArtifactStore(support.replicationBucket),
        }));
    }
    renderArtifactStoreProperty() {
        if (this.crossRegion) {
            return undefined;
        }
        return this.renderPrimaryArtifactStore();
    }
    renderPrimaryArtifactStore() {
        return this.renderArtifactStore(this.artifactBucket);
    }
    renderArtifactStore(bucket) {
        let encryptionKey;
        const bucketKey = bucket.encryptionKey;
        if (bucketKey) {
            encryptionKey = {
                type: 'KMS',
                id: bucketKey.keyArn,
            };
        }
        return {
            type: 'S3',
            location: bucket.bucketName,
            encryptionKey,
        };
    }
    get crossRegion() {
        if (this.crossRegionBucketsPassed) {
            return true;
        }
        return this._stages.some(stage => stage.actionDescriptors.some(action => action.region !== undefined));
    }
    renderStages() {
        return this._stages.map(stage => stage.render());
    }
    requireRegion() {
        const region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set region');
        }
        return region;
    }
    requireApp() {
        const app = this.node.root;
        if (!app || !core_1.App.isApp(app)) {
            throw new Error('Pipeline stack which uses cross-environment actions must be part of a CDK app');
        }
        return app;
    }
}
exports.Pipeline = Pipeline;
function enumerate(xs) {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        ret.push([i, xs[i]]);
    }
    return ret;
}
class PipelineLocation {
    constructor(stageIndex, stage, action) {
        this.stageIndex = stageIndex;
        this.stage = stage;
        this.action = action;
    }
    get stageName() {
        return this.stage.stageName;
    }
    get actionName() {
        return this.action.actionName;
    }
    /**
     * Returns whether a is before or the same order as b
     */
    beforeOrEqual(rhs) {
        if (this.stageIndex !== rhs.stageIndex) {
            return rhs.stageIndex < rhs.stageIndex;
        }
        return this.action.runOrder <= rhs.action.runOrder;
    }
    /**
     * Returns the first location between this and the other one
     */
    first(rhs) {
        return this.beforeOrEqual(rhs) ? this : rhs;
    }
    toString() {
        // runOrders are 1-based, so make the stageIndex also 1-based otherwise it's going to be confusing.
        return `Stage ${this.stageIndex + 1} Action ${this.action.runOrder} ('${this.stageName}'/'${this.actionName}')`;
    }
}
//# sourceMappingURL=data:application/json;base64,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