"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeDeployEcsDeployAction = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const action_1 = require("../action");
class CodeDeployEcsDeployAction extends action_1.Action {
    constructor(props) {
        const inputs = [];
        inputs.push(determineTaskDefinitionArtifact(props));
        inputs.push(determineAppSpecArtifact(props));
        if (props.containerImageInputs) {
            if (props.containerImageInputs.length > 4) {
                throw new Error(`Action cannot have more than 4 container image inputs, got: ${props.containerImageInputs.length}`);
            }
            for (const imageInput of props.containerImageInputs) {
                inputs.push(imageInput.input);
            }
        }
        super({
            ...props,
            resource: props.deploymentGroup,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'CodeDeployToECS',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 0 },
            inputs,
        });
        this.actionProps = props;
    }
    bound(_scope, _stage, options) {
        // permissions, based on:
        // https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.application.applicationArn],
            actions: ['codedeploy:GetApplication', 'codedeploy:GetApplicationRevision', 'codedeploy:RegisterApplicationRevision'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.deploymentGroupArn],
            actions: ['codedeploy:CreateDeployment', 'codedeploy:GetDeployment'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.deploymentConfig.deploymentConfigArn],
            actions: ['codedeploy:GetDeploymentConfig'],
        }));
        // Allow action to register the task definition template file with ECS
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['ecs:RegisterTaskDefinition'],
        }));
        // Allow passing any roles specified in the task definition template file to ECS
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: ['*'],
            conditions: {
                StringEqualsIfExists: {
                    'iam:PassedToService': [
                        'ecs-tasks.amazonaws.com',
                    ],
                },
            },
        }));
        // the Action's Role needs to read from the Bucket to get artifacts
        options.bucket.grantRead(options.role);
        const actionConfig = {
            configuration: {
                ApplicationName: this.actionProps.deploymentGroup.application.applicationName,
                DeploymentGroupName: this.actionProps.deploymentGroup.deploymentGroupName,
                TaskDefinitionTemplateArtifact: determineTaskDefinitionArtifact(this.actionProps).artifactName,
                TaskDefinitionTemplatePath: this.actionProps.taskDefinitionTemplateFile
                    ? this.actionProps.taskDefinitionTemplateFile.fileName
                    : 'taskdef.json',
                AppSpecTemplateArtifact: determineAppSpecArtifact(this.actionProps).artifactName,
                AppSpecTemplatePath: this.actionProps.appSpecTemplateFile
                    ? this.actionProps.appSpecTemplateFile.fileName
                    : 'appspec.yaml',
            },
        };
        if (this.actionProps.containerImageInputs) {
            for (let i = 1; i <= this.actionProps.containerImageInputs.length; i++) {
                const imageInput = this.actionProps.containerImageInputs[i - 1];
                actionConfig.configuration[`Image${i}ArtifactName`] = imageInput.input.artifactName;
                actionConfig.configuration[`Image${i}ContainerName`] = imageInput.taskDefinitionPlaceholder
                    ? imageInput.taskDefinitionPlaceholder
                    : 'IMAGE';
            }
        }
        return actionConfig;
    }
}
exports.CodeDeployEcsDeployAction = CodeDeployEcsDeployAction;
function determineTaskDefinitionArtifact(props) {
    if (props.taskDefinitionTemplateFile && props.taskDefinitionTemplateInput) {
        throw new Error("Exactly one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' can be provided in the ECS CodeDeploy Action");
    }
    if (props.taskDefinitionTemplateFile) {
        return props.taskDefinitionTemplateFile.artifact;
    }
    if (props.taskDefinitionTemplateInput) {
        return props.taskDefinitionTemplateInput;
    }
    throw new Error("Specifying one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' is required for the ECS CodeDeploy Action");
}
function determineAppSpecArtifact(props) {
    if (props.appSpecTemplateFile && props.appSpecTemplateInput) {
        throw new Error("Exactly one of 'appSpecTemplateInput' or 'appSpecTemplateFile' can be provided in the ECS CodeDeploy Action");
    }
    if (props.appSpecTemplateFile) {
        return props.appSpecTemplateFile.artifact;
    }
    if (props.appSpecTemplateInput) {
        return props.appSpecTemplateInput;
    }
    throw new Error("Specifying one of 'appSpecTemplateInput' or 'appSpecTemplateFile' is required for the ECS CodeDeploy Action");
}
//# sourceMappingURL=data:application/json;base64,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