"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlarmRule = exports.AlarmState = void 0;
/**
 * Enumeration indicates state of Alarm used in building Alarm Rule.
 */
var AlarmState;
(function (AlarmState) {
    /**
     * State indicates resource is in ALARM
     */
    AlarmState["ALARM"] = "ALARM";
    /**
     * State indicates resource is not in ALARM
     */
    AlarmState["OK"] = "OK";
    /**
     * State indicates there is not enough data to determine is resource is in ALARM
     */
    AlarmState["INSUFFICIENT_DATA"] = "INSUFFICIENT_DATA";
})(AlarmState = exports.AlarmState || (exports.AlarmState = {}));
/**
 * Enumeration of supported Composite Alarms operators.
 */
var Operator;
(function (Operator) {
    Operator["AND"] = "AND";
    Operator["OR"] = "OR";
    Operator["NOT"] = "NOT";
})(Operator || (Operator = {}));
/**
 * Class with static functions to build AlarmRule for Composite Alarms.
 */
class AlarmRule {
    /**
     * function to join all provided AlarmRules with AND operator.
     *
     * @param operands IAlarmRules to be joined with AND operator.
     */
    static allOf(...operands) {
        return this.concat(Operator.AND, ...operands);
    }
    /**
     * function to join all provided AlarmRules with OR operator.
     *
     * @param operands IAlarmRules to be joined with OR operator.
     */
    static anyOf(...operands) {
        return this.concat(Operator.OR, ...operands);
    }
    /**
     * function to wrap provided AlarmRule in NOT operator.
     *
     * @param operand IAlarmRule to be wrapped in NOT operator.
     */
    static not(operand) {
        // tslint:disable-next-line:new-parens
        return new class {
            renderAlarmRule() {
                return `(NOT (${operand.renderAlarmRule()}))`;
            }
        };
    }
    /**
     * function to build TRUE/FALSE intent for Rule Expression.
     *
     * @param value boolean value to be used in rule expression.
     */
    static fromBoolean(value) {
        // tslint:disable-next-line:new-parens
        return new class {
            renderAlarmRule() {
                return `${String(value).toUpperCase()}`;
            }
        };
    }
    /**
     * function to build Rule Expression for given IAlarm and AlarmState.
     *
     * @param alarm IAlarm to be used in Rule Expression.
     * @param alarmState AlarmState to be used in Rule Expression.
     */
    static fromAlarm(alarm, alarmState) {
        // tslint:disable-next-line:new-parens
        return new class {
            renderAlarmRule() {
                return `${alarmState}(${alarm.alarmArn})`;
            }
        };
    }
    /**
     * function to build Rule Expression for given Alarm Rule string.
     *
     * @param alarmRule string to be used in Rule Expression.
     */
    static fromString(alarmRule) {
        // tslint:disable-next-line:new-parens
        return new class {
            renderAlarmRule() {
                return alarmRule;
            }
        };
    }
    static concat(operator, ...operands) {
        // tslint:disable-next-line:new-parens
        return new class {
            renderAlarmRule() {
                const expression = operands
                    .map(operand => `${operand.renderAlarmRule()}`)
                    .join(` ${operator} `);
                return `(${expression})`;
            }
        };
    }
}
exports.AlarmRule = AlarmRule;
//# sourceMappingURL=data:application/json;base64,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