"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EbsDeviceVolumeType = exports.BlockDeviceVolume = void 0;
/**
 * Describes a block device mapping for an EC2 instance or Auto Scaling group.
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info
     * @param virtualName Virtual device name
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
    }
    /**
     * Creates a new Elastic Block Storage device
     *
     * @param volumeSize The volume size, in Gibibytes (GiB)
     * @param options additional device options
     */
    static ebs(volumeSize, options = {}) {
        return new this({ ...options, volumeSize });
    }
    /**
     * Creates a new Elastic Block Storage device from an existing snapshot
     *
     * @param snapshotId The snapshot ID of the volume to use
     * @param options additional device options
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        return new this({ ...options, snapshotId });
    }
    /**
     * Creates a virtual, ephemeral device.
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index. Must be equal or greater than 0
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
    /**
     * Supresses a volume mapping
     */
    static noDevice() {
        return this._NO_DEVICE;
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
/**
 * @internal
 */
BlockDeviceVolume._NO_DEVICE = new BlockDeviceVolume();
/**
 * Supported EBS volume types for blockDevices
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    /**
     * Magnetic
     */
    EbsDeviceVolumeType["STANDARD"] = "standard";
    /**
     *  Provisioned IOPS SSD
     */
    EbsDeviceVolumeType["IO1"] = "io1";
    /**
     * General Purpose SSD
     */
    EbsDeviceVolumeType["GP2"] = "gp2";
    /**
     * Throughput Optimized HDD
     */
    EbsDeviceVolumeType["ST1"] = "st1";
    /**
     * Cold HDD
     */
    EbsDeviceVolumeType["SC1"] = "sc1";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
//# sourceMappingURL=data:application/json;base64,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