import * as cloudwatch from "../../aws-cloudwatch";
import * as ec2 from "../../aws-ec2";
import * as elb from "../../aws-elasticloadbalancing";
import * as elbv2 from "../../aws-elasticloadbalancingv2";
import * as iam from "../../aws-iam";
import * as sns from "../../aws-sns";
import { Construct, Duration, IResource, Resource } from "../../core";
import { BasicLifecycleHookProps, LifecycleHook } from './lifecycle-hook';
import { BasicScheduledActionProps, ScheduledAction } from './scheduled-action';
import { BasicStepScalingPolicyProps, StepScalingPolicy } from './step-scaling-policy';
import { BaseTargetTrackingProps, TargetTrackingScalingPolicy } from './target-tracking-scaling-policy';
import { BlockDevice } from './volume';
/**
 * The monitoring mode for instances launched in an autoscaling group
 */
export declare enum Monitoring {
    /**
     * Generates metrics every 5 minutes
     */
    BASIC = 0,
    /**
     * Generates metrics every minute
     */
    DETAILED = 1
}
/**
 * Basic properties of an AutoScalingGroup, except the exact machines to run and where they should run
 *
 * Constructs that want to create AutoScalingGroups can inherit
 * this interface and specialize the essential parts in various ways.
 */
export interface CommonAutoScalingGroupProps {
    /**
     * Minimum number of instances in the fleet
     *
     * @default 1
     */
    readonly minCapacity?: number;
    /**
     * Maximum number of instances in the fleet
     *
     * @default desiredCapacity
     */
    readonly maxCapacity?: number;
    /**
     * Initial amount of instances in the fleet
     *
     * If this is set to a number, every deployment will reset the amount of
     * instances to this number. It is recommended to leave this value blank.
     *
     * @default minCapacity, and leave unchanged during deployment
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-desiredcapacity
     */
    readonly desiredCapacity?: number;
    /**
     * Name of SSH keypair to grant access to instances
     *
     * @default - No SSH access will be possible.
     */
    readonly keyName?: string;
    /**
     * Where to place instances within the VPC
     *
     * @default - All Private subnets.
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * SNS topic to send notifications about fleet changes
     *
     * @default - No fleet change notifications will be sent.
     * @deprecated use `notifications`
     */
    readonly notificationsTopic?: sns.ITopic;
    /**
     * Configure autoscaling group to send notifications about fleet changes to an SNS topic(s)
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-notificationconfigurations
     * @default - No fleet change notifications will be sent.
     */
    readonly notifications?: NotificationConfiguration[];
    /**
     * Whether the instances can initiate connections to anywhere by default
     *
     * @default true
     */
    readonly allowAllOutbound?: boolean;
    /**
     * What to do when an AutoScalingGroup's instance configuration is changed
     *
     * This is applied when any of the settings on the ASG are changed that
     * affect how the instances should be created (VPC, instance type, startup
     * scripts, etc.). It indicates how the existing instances should be
     * replaced with new instances matching the new config. By default, nothing
     * is done and only new instances are launched with the new config.
     *
     * @default UpdateType.None
     */
    readonly updateType?: UpdateType;
    /**
     * Configuration for rolling updates
     *
     * Only used if updateType == UpdateType.RollingUpdate.
     *
     * @default - RollingUpdateConfiguration with defaults.
     */
    readonly rollingUpdateConfiguration?: RollingUpdateConfiguration;
    /**
     * Configuration for replacing updates.
     *
     * Only used if updateType == UpdateType.ReplacingUpdate. Specifies how
     * many instances must signal success for the update to succeed.
     *
     * @default minSuccessfulInstancesPercent
     */
    readonly replacingUpdateMinSuccessfulInstancesPercent?: number;
    /**
     * If the ASG has scheduled actions, don't reset unchanged group sizes
     *
     * Only used if the ASG has scheduled actions (which may scale your ASG up
     * or down regardless of cdk deployments). If true, the size of the group
     * will only be reset if it has been changed in the CDK app. If false, the
     * sizes will always be changed back to what they were in the CDK app
     * on deployment.
     *
     * @default true
     */
    readonly ignoreUnmodifiedSizeProperties?: boolean;
    /**
     * How many ResourceSignal calls CloudFormation expects before the resource is considered created
     *
     * @default 1
     */
    readonly resourceSignalCount?: number;
    /**
     * The length of time to wait for the resourceSignalCount
     *
     * The maximum value is 43200 (12 hours).
     *
     * @default Duration.minutes(5)
     */
    readonly resourceSignalTimeout?: Duration;
    /**
     * Default scaling cooldown for this AutoScalingGroup
     *
     * @default Duration.minutes(5)
     */
    readonly cooldown?: Duration;
    /**
     * Whether instances in the Auto Scaling Group should have public
     * IP addresses associated with them.
     *
     * @default - Use subnet setting.
     */
    readonly associatePublicIpAddress?: boolean;
    /**
     * The maximum hourly price (in USD) to be paid for any Spot Instance launched to fulfill the request. Spot Instances are
     * launched when the price you specify exceeds the current Spot market price.
     *
     * @default none
     */
    readonly spotPrice?: string;
    /**
     * Configuration for health checks
     *
     * @default - HealthCheck.ec2 with no grace period
     */
    readonly healthCheck?: HealthCheck;
    /**
     * Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes.
     *
     * Each instance that is launched has an associated root device volume,
     * either an Amazon EBS volume or an instance store volume.
     * You can use block device mappings to specify additional EBS volumes or
     * instance store volumes to attach to an instance when it is launched.
     *
     * @see https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html
     *
     * @default - Uses the block device mapping of the AMI
     */
    readonly blockDevices?: BlockDevice[];
    /**
     * The maximum amount of time that an instance can be in service. The maximum duration applies
     * to all current and future instances in the group. As an instance approaches its maximum duration,
     * it is terminated and replaced, and cannot be used again.
     *
     * You must specify a value of at least 604,800 seconds (7 days). To clear a previously set value,
     * simply leave this property undefinied.
     *
     * @see https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-max-instance-lifetime.html
     *
     * @default none
     */
    readonly maxInstanceLifetime?: Duration;
    /**
     * Controls whether instances in this group are launched with detailed or basic monitoring.
     *
     * When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account
     * is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes.
     *
     * @see https://docs.aws.amazon.com/autoscaling/latest/userguide/as-instance-monitoring.html#enable-as-instance-metrics
     *
     * @default - Monitoring.DETAILED
     */
    readonly instanceMonitoring?: Monitoring;
}
/**
 * Properties of a Fleet
 */
export interface AutoScalingGroupProps extends CommonAutoScalingGroupProps {
    /**
     * VPC to launch these instances in.
     */
    readonly vpc: ec2.IVpc;
    /**
     * Type of instance to launch
     */
    readonly instanceType: ec2.InstanceType;
    /**
     * AMI to launch
     */
    readonly machineImage: ec2.IMachineImage;
    /**
     * Security group to launch the instances in.
     *
     * @default - A SecurityGroup will be created if none is specified.
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * Specific UserData to use
     *
     * The UserData may still be mutated after creation.
     *
     * @default - A UserData object appropriate for the MachineImage's
     * Operating System is created.
     */
    readonly userData?: ec2.UserData;
    /**
     * An IAM role to associate with the instance profile assigned to this Auto Scaling Group.
     *
     * The role must be assumable by the service principal `ec2.amazonaws.com`:
     *
     * @example
     *
     *    const role = new iam.Role(this, 'MyRole', {
     *      assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com')
     *    });
     *
     * @default A role will automatically be created, it can be accessed via the `role` property
     */
    readonly role?: iam.IRole;
}
declare abstract class AutoScalingGroupBase extends Resource implements IAutoScalingGroup {
    abstract autoScalingGroupName: string;
    abstract autoScalingGroupArn: string;
    protected albTargetGroup?: elbv2.ApplicationTargetGroup;
    /**
     * Send a message to either an SQS queue or SNS topic when instances launch or terminate
     */
    addLifecycleHook(id: string, props: BasicLifecycleHookProps): LifecycleHook;
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id: string, props: BasicScheduledActionProps): ScheduledAction;
    /**
     * Scale out or in to achieve a target CPU utilization
     */
    scaleOnCpuUtilization(id: string, props: CpuUtilizationScalingProps): TargetTrackingScalingPolicy;
    /**
     * Scale out or in to achieve a target network ingress rate
     */
    scaleOnIncomingBytes(id: string, props: NetworkUtilizationScalingProps): TargetTrackingScalingPolicy;
    /**
     * Scale out or in to achieve a target network egress rate
     */
    scaleOnOutgoingBytes(id: string, props: NetworkUtilizationScalingProps): TargetTrackingScalingPolicy;
    /**
     * Scale out or in to achieve a target request handling rate
     *
     * The AutoScalingGroup must have been attached to an Application Load Balancer
     * in order to be able to call this.
     */
    scaleOnRequestCount(id: string, props: RequestCountScalingProps): TargetTrackingScalingPolicy;
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id: string, props: MetricTargetTrackingProps): TargetTrackingScalingPolicy;
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id: string, props: BasicStepScalingPolicyProps): StepScalingPolicy;
}
/**
 * A Fleet represents a managed set of EC2 instances
 *
 * The Fleet models a number of AutoScalingGroups, a launch configuration, a
 * security group and an instance role.
 *
 * It allows adding arbitrary commands to the startup scripts of the instances
 * in the fleet.
 *
 * The ASG spans the availability zones specified by vpcSubnets, falling back to
 * the Vpc default strategy if not specified.
 */
export declare class AutoScalingGroup extends AutoScalingGroupBase implements elb.ILoadBalancerTarget, ec2.IConnectable, elbv2.IApplicationLoadBalancerTarget, elbv2.INetworkLoadBalancerTarget, iam.IGrantable {
    static fromAutoScalingGroupName(scope: Construct, id: string, autoScalingGroupName: string): IAutoScalingGroup;
    /**
     * The type of OS instances of this fleet are running.
     */
    readonly osType: ec2.OperatingSystemType;
    /**
     * Allows specify security group connections for instances of this fleet.
     */
    readonly connections: ec2.Connections;
    /**
     * The IAM role assumed by instances of this fleet.
     */
    readonly role: iam.IRole;
    /**
     * The principal to grant permissions to
     */
    readonly grantPrincipal: iam.IPrincipal;
    /**
     * Name of the AutoScalingGroup
     */
    readonly autoScalingGroupName: string;
    /**
     * Arn of the AutoScalingGroup
     */
    readonly autoScalingGroupArn: string;
    /**
     * UserData for the instances
     */
    readonly userData: ec2.UserData;
    /**
     * The maximum spot price configured for thie autoscaling group. `undefined`
     * indicates that this group uses on-demand capacity.
     */
    readonly spotPrice?: string;
    /**
     * The maximum amount of time that an instance can be in service.
     */
    readonly maxInstanceLifetime?: Duration;
    private readonly autoScalingGroup;
    private readonly securityGroup;
    private readonly securityGroups;
    private readonly loadBalancerNames;
    private readonly targetGroupArns;
    private readonly notifications;
    constructor(scope: Construct, id: string, props: AutoScalingGroupProps);
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup: ec2.ISecurityGroup): void;
    /**
     * Attach to a classic load balancer
     */
    attachToClassicLB(loadBalancer: elb.LoadBalancer): void;
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToApplicationTargetGroup(targetGroup: elbv2.IApplicationTargetGroup): elbv2.LoadBalancerTargetProps;
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToNetworkTargetGroup(targetGroup: elbv2.INetworkTargetGroup): elbv2.LoadBalancerTargetProps;
    /**
     * Add command to the startup script of fleet instances.
     * The command must be in the scripting language supported by the fleet's OS (i.e. Linux/Windows).
     */
    addUserData(...commands: string[]): void;
    /**
     * Adds a statement to the IAM role assumed by instances of this fleet.
     */
    addToRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Apply CloudFormation update policies for the AutoScalingGroup
     */
    private applyUpdatePolicies;
    private renderNotificationConfiguration;
}
/**
 * The type of update to perform on instances in this AutoScalingGroup
 */
export declare enum UpdateType {
    /**
     * Don't do anything
     */
    NONE = "None",
    /**
     * Replace the entire AutoScalingGroup
     *
     * Builds a new AutoScalingGroup first, then delete the old one.
     */
    REPLACING_UPDATE = "Replace",
    /**
     * Replace the instances in the AutoScalingGroup.
     */
    ROLLING_UPDATE = "RollingUpdate"
}
/**
 * AutoScalingGroup fleet change notifications configurations.
 * You can configure AutoScaling to send an SNS notification whenever your Auto Scaling group scales.
 */
export interface NotificationConfiguration {
    /**
     * SNS topic to send notifications about fleet scaling events
     */
    readonly topic: sns.ITopic;
    /**
     * Which fleet scaling events triggers a notification
     * @default ScalingEvents.ALL
     */
    readonly scalingEvents?: ScalingEvents;
}
/**
 * Fleet scaling events
 */
export declare enum ScalingEvent {
    /**
     * Notify when an instance was launced
     */
    INSTANCE_LAUNCH = "autoscaling:EC2_INSTANCE_LAUNCH",
    /**
     * Notify when an instance was terminated
     */
    INSTANCE_TERMINATE = "autoscaling:EC2_INSTANCE_TERMINATE",
    /**
     * Notify when an instance failed to terminate
     */
    INSTANCE_TERMINATE_ERROR = "autoscaling:EC2_INSTANCE_TERMINATE_ERROR",
    /**
     * Notify when an instance failed to launch
     */
    INSTANCE_LAUNCH_ERROR = "autoscaling:EC2_INSTANCE_LAUNCH_ERROR",
    /**
     * Send a test notification to the topic
     */
    TEST_NOTIFICATION = "autoscaling:TEST_NOTIFICATION"
}
/**
 * Additional settings when a rolling update is selected
 */
export interface RollingUpdateConfiguration {
    /**
     * The maximum number of instances that AWS CloudFormation updates at once.
     *
     * @default 1
     */
    readonly maxBatchSize?: number;
    /**
     * The minimum number of instances that must be in service before more instances are replaced.
     *
     * This number affects the speed of the replacement.
     *
     * @default 0
     */
    readonly minInstancesInService?: number;
    /**
     * The percentage of instances that must signal success for an update to succeed.
     *
     * If an instance doesn't send a signal within the time specified in the
     * pauseTime property, AWS CloudFormation assumes that the instance wasn't
     * updated.
     *
     * This number affects the success of the replacement.
     *
     * If you specify this property, you must also enable the
     * waitOnResourceSignals and pauseTime properties.
     *
     * @default 100
     */
    readonly minSuccessfulInstancesPercent?: number;
    /**
     * The pause time after making a change to a batch of instances.
     *
     * This is intended to give those instances time to start software applications.
     *
     * Specify PauseTime in the ISO8601 duration format (in the format
     * PT#H#M#S, where each # is the number of hours, minutes, and seconds,
     * respectively). The maximum PauseTime is one hour (PT1H).
     *
     * @default Duration.minutes(5) if the waitOnResourceSignals property is true, otherwise 0
     */
    readonly pauseTime?: Duration;
    /**
     * Specifies whether the Auto Scaling group waits on signals from new instances during an update.
     *
     * AWS CloudFormation must receive a signal from each new instance within
     * the specified PauseTime before continuing the update.
     *
     * To have instances wait for an Elastic Load Balancing health check before
     * they signal success, add a health-check verification by using the
     * cfn-init helper script. For an example, see the verify_instance_health
     * command in the Auto Scaling rolling updates sample template.
     *
     * @default true if you specified the minSuccessfulInstancesPercent property, false otherwise
     */
    readonly waitOnResourceSignals?: boolean;
    /**
     * Specifies the Auto Scaling processes to suspend during a stack update.
     *
     * Suspending processes prevents Auto Scaling from interfering with a stack
     * update.
     *
     * @default HealthCheck, ReplaceUnhealthy, AZRebalance, AlarmNotification, ScheduledActions.
     */
    readonly suspendProcesses?: ScalingProcess[];
}
/**
 * A list of ScalingEvents, you can use one of the predefined lists, such as ScalingEvents.ERRORS
 * or create a custome group by instantiating a `NotificationTypes` object, e.g: `new NotificationTypes(`NotificationType.INSTANCE_LAUNCH`)`.
 */
export declare class ScalingEvents {
    /**
     * Fleet scaling errors
     */
    static readonly ERRORS: ScalingEvents;
    /**
     * All fleet scaling events
     */
    static readonly ALL: ScalingEvents;
    /**
     * Fleet scaling launch events
     */
    static readonly LAUNCH_EVENTS: ScalingEvents;
    constructor(...types: ScalingEvent[]);
}
export declare enum ScalingProcess {
    LAUNCH = "Launch",
    TERMINATE = "Terminate",
    HEALTH_CHECK = "HealthCheck",
    REPLACE_UNHEALTHY = "ReplaceUnhealthy",
    AZ_REBALANCE = "AZRebalance",
    ALARM_NOTIFICATION = "AlarmNotification",
    SCHEDULED_ACTIONS = "ScheduledActions",
    ADD_TO_LOAD_BALANCER = "AddToLoadBalancer"
}
/**
 * EC2 Heath check options
 */
export interface Ec2HealthCheckOptions {
    /**
     * Specified the time Auto Scaling waits before checking the health status of an EC2 instance that has come into service
     *
     * @default Duration.seconds(0)
     */
    readonly grace?: Duration;
}
/**
 * ELB Heath check options
 */
export interface ElbHealthCheckOptions {
    /**
     * Specified the time Auto Scaling waits before checking the health status of an EC2 instance that has come into service
     *
     * This option is required for ELB health checks.
     */
    readonly grace: Duration;
}
/**
 * Health check settings
 */
export declare class HealthCheck {
    readonly type: string;
    readonly gracePeriod?: Duration | undefined;
    /**
     * Use EC2 for health checks
     *
     * @param options EC2 health check options
     */
    static ec2(options?: Ec2HealthCheckOptions): HealthCheck;
    /**
     * Use ELB for health checks.
     * It considers the instance unhealthy if it fails either the EC2 status checks or the load balancer health checks.
     *
     * @param options ELB health check options
     */
    static elb(options: ElbHealthCheckOptions): HealthCheck;
    private constructor();
}
/**
 * An AutoScalingGroup
 */
export interface IAutoScalingGroup extends IResource {
    /**
     * The name of the AutoScalingGroup
     * @attribute
     */
    readonly autoScalingGroupName: string;
    /**
     * The arn of the AutoScalingGroup
     * @attribute
     */
    readonly autoScalingGroupArn: string;
    /**
     * Send a message to either an SQS queue or SNS topic when instances launch or terminate
     */
    addLifecycleHook(id: string, props: BasicLifecycleHookProps): LifecycleHook;
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id: string, props: BasicScheduledActionProps): ScheduledAction;
    /**
     * Scale out or in to achieve a target CPU utilization
     */
    scaleOnCpuUtilization(id: string, props: CpuUtilizationScalingProps): TargetTrackingScalingPolicy;
    /**
     * Scale out or in to achieve a target network ingress rate
     */
    scaleOnIncomingBytes(id: string, props: NetworkUtilizationScalingProps): TargetTrackingScalingPolicy;
    /**
     * Scale out or in to achieve a target network egress rate
     */
    scaleOnOutgoingBytes(id: string, props: NetworkUtilizationScalingProps): TargetTrackingScalingPolicy;
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id: string, props: MetricTargetTrackingProps): TargetTrackingScalingPolicy;
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id: string, props: BasicStepScalingPolicyProps): StepScalingPolicy;
}
/**
 * Properties for enabling scaling based on CPU utilization
 */
export interface CpuUtilizationScalingProps extends BaseTargetTrackingProps {
    /**
     * Target average CPU utilization across the task
     */
    readonly targetUtilizationPercent: number;
}
/**
 * Properties for enabling scaling based on network utilization
 */
export interface NetworkUtilizationScalingProps extends BaseTargetTrackingProps {
    /**
     * Target average bytes/seconds on each instance
     */
    readonly targetBytesPerSecond: number;
}
/**
 * Properties for enabling scaling based on request/second
 */
export interface RequestCountScalingProps extends BaseTargetTrackingProps {
    /**
     * Target average requests/seconds on each instance
     */
    readonly targetRequestsPerSecond: number;
}
/**
 * Properties for enabling tracking of an arbitrary metric
 */
export interface MetricTargetTrackingProps extends BaseTargetTrackingProps {
    /**
     * Metric to track
     *
     * The metric must represent a utilization, so that if it's higher than the
     * target value, your ASG should scale out, and if it's lower it should
     * scale in.
     */
    readonly metric: cloudwatch.IMetric;
    /**
     * Value to keep the metric around
     */
    readonly targetValue: number;
}
export {};
