"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualRouter = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
const route_1 = require("./route");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualRouterBase extends cdk.Resource {
    /**
     * Add a single route to the router
     */
    addRoute(id, props) {
        const route = new route_1.Route(this, id, {
            ...props,
            routeName: id,
            mesh: this.mesh,
            virtualRouter: this,
        });
        return route;
    }
}
class VirtualRouter extends VirtualRouterBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualRouterName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.listeners = new Array();
        this.mesh = props.mesh;
        this.addListener(props.listener || { portMapping: { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP } });
        const router = new appmesh_generated_1.CfnVirtualRouter(this, 'Resource', {
            virtualRouterName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                listeners: this.listeners,
            },
        });
        this.virtualRouterName = this.getResourceNameAttribute(router.attrVirtualRouterName);
        this.virtualRouterArn = this.getResourceArnAttribute(router.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualRouter given an ARN
     */
    static fromVirtualRouterArn(scope, id, virtualRouterArn) {
        return new ImportedVirtualRouter(scope, id, { virtualRouterArn });
    }
    /**
     * Import an existing VirtualRouter given names
     */
    static fromVirtualRouterName(scope, id, meshName, virtualRouterName) {
        return new ImportedVirtualRouter(scope, id, { meshName, virtualRouterName });
    }
    /**
     * Import an existing virtual router given attributes
     */
    static fromVirtualRouterAttributes(scope, id, attrs) {
        return new ImportedVirtualRouter(scope, id, attrs);
    }
    /**
     * Add port mappings to the router
     */
    addListener(listener) {
        this.listeners.push({
            portMapping: listener.portMapping,
        });
    }
}
exports.VirtualRouter = VirtualRouter;
/**
 * Used to import a VirtualRouter and perform actions or read properties from
 */
class ImportedVirtualRouter extends VirtualRouterBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.mesh) {
            this._mesh = props.mesh;
        }
        if (props.meshName) {
            if (props.mesh) {
                throw new Error('Supply either \'mesh\' or \'meshName\', but not both');
            }
            this._mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', props.meshName);
        }
        if (props.virtualRouterArn) {
            this.virtualRouterArn = props.virtualRouterArn;
            this.virtualRouterName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualRouterArn).resourceName));
        }
        else if (props.virtualRouterName && props.meshName) {
            this.virtualRouterName = props.virtualRouterName;
            this.virtualRouterArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter`,
                resourceName: this.virtualRouterName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
    /**
     * The AppMesh mesh the VirtualRouter belongs to
     */
    get mesh() {
        if (!this._mesh) {
            throw new Error('Please supply either \'mesh\' or \'meshName\' when calling \'fromVirtualRouterAttributes\'');
        }
        return this._mesh;
    }
}
//# sourceMappingURL=data:application/json;base64,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