"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpRoute = exports.HttpRouteKey = exports.HttpMethod = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const integration_1 = require("./integration");
/**
 * Supported HTTP methods
 */
var HttpMethod;
(function (HttpMethod) {
    /** HTTP ANY */
    HttpMethod["ANY"] = "ANY";
    /** HTTP DELETE */
    HttpMethod["DELETE"] = "DELETE";
    /** HTTP GET */
    HttpMethod["GET"] = "GET";
    /** HTTP HEAD */
    HttpMethod["HEAD"] = "HEAD";
    /** HTTP OPTIONS */
    HttpMethod["OPTIONS"] = "OPTIONS";
    /** HTTP PATCH */
    HttpMethod["PATCH"] = "PATCH";
    /** HTTP POST */
    HttpMethod["POST"] = "POST";
    /** HTTP PUT */
    HttpMethod["PUT"] = "PUT";
})(HttpMethod = exports.HttpMethod || (exports.HttpMethod = {}));
/**
 * HTTP route in APIGateway is a combination of the HTTP method and the path component.
 * This class models that combination.
 */
class HttpRouteKey {
    constructor(key, path) {
        this.key = key;
        this.path = path;
    }
    /**
     * Create a route key with the combination of the path and the method.
     * @param method default is 'ANY'
     */
    static with(path, method) {
        if (path !== '/' && (!path.startsWith('/') || path.endsWith('/'))) {
            throw new Error('path must always start with a "/" and not end with a "/"');
        }
        return new HttpRouteKey(`${method !== null && method !== void 0 ? method : 'ANY'} ${path}`, path);
    }
}
exports.HttpRouteKey = HttpRouteKey;
/**
 * The catch-all route of the API, i.e., when no other routes match
 */
HttpRouteKey.DEFAULT = new HttpRouteKey('$default');
/**
 * Route class that creates the Route for API Gateway HTTP API
 * @resource AWS::ApiGatewayV2::Route
 */
class HttpRoute extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.httpApi = props.httpApi;
        this.path = props.routeKey.path;
        let integration;
        const config = props.integration.bind(this);
        integration = new integration_1.HttpIntegration(this, `${this.node.id}-Integration`, {
            httpApi: props.httpApi,
            integrationType: config.type,
            integrationUri: config.uri,
            method: config.method,
            payloadFormatVersion: config.payloadFormatVersion,
        });
        const routeProps = {
            apiId: props.httpApi.httpApiId,
            routeKey: props.routeKey.key,
            target: `integrations/${integration.integrationId}`,
        };
        const route = new apigatewayv2_generated_1.CfnRoute(this, 'Resource', routeProps);
        this.routeId = route.ref;
    }
}
exports.HttpRoute = HttpRoute;
//# sourceMappingURL=data:application/json;base64,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