"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaIntegration = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const aws_1 = require("./aws");
/**
 * Integrates an AWS Lambda function to an API Gateway method.
 *
 * @example
 *
 *    const handler = new lambda.Function(this, 'MyFunction', ...);
 *    api.addMethod('GET', new LambdaIntegration(handler));
 *
 */
class LambdaIntegration extends aws_1.AwsIntegration {
    constructor(handler, options = {}) {
        const proxy = options.proxy === undefined ? true : options.proxy;
        super({
            proxy,
            service: 'lambda',
            path: `2015-03-31/functions/${handler.functionArn}/invocations`,
            options,
        });
        this.handler = handler;
        this.enableTest = options.allowTestInvoke === undefined ? true : false;
    }
    bind(method) {
        super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        const desc = `${method.api.node.uniqueId}.${method.httpMethod}.${method.resource.path.replace(/\//g, '.')}`;
        this.handler.addPermission(`ApiPermission.${desc}`, {
            principal,
            scope: method,
            sourceArn: core_1.Lazy.stringValue({ produce: () => method.methodArn }),
        });
        // add permission to invoke from the console
        if (this.enableTest) {
            this.handler.addPermission(`ApiPermission.Test.${desc}`, {
                principal,
                scope: method,
                sourceArn: method.testMethodArn,
            });
        }
        const cfnFunction = this.handler.node.defaultChild;
        let deploymentToken;
        if (!core_1.Token.isUnresolved(cfnFunction.functionName)) {
            deploymentToken = JSON.stringify({ functionName: cfnFunction.functionName });
        }
        return {
            deploymentToken,
        };
    }
}
exports.LambdaIntegration = LambdaIntegration;
//# sourceMappingURL=data:application/json;base64,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