# Dataism Lab for Quantitative Finance (DLQF) Python Module

## Summary - The goal of this module provides a collection of methods that can be implemented by clients to test for linearity and non-linearity on univariate and multivariate data sets.

# 0. Abstract

In quantitative finance, accurately modeling the dynamics of asset prices is critical for risk management, portfolio optimization, and the development of trading strategies. Many financial time series exhibit complex, potentially nonlinear behavior that cannot be adequately captured by traditional linear models. Detecting nonlinearity in time series can signal the need for more advanced modeling techniques such as regime-switching models, nonlinear autoregressive models, or machine learning methods. This document describes a comprehensive Python module for nonlinearity testing that supports both univariate and multivariate nonlinearity tests with specific focus on financial time series (e.g., equities, indices, and market volumes). The module implements conventional tests such as those proposed by Ramsey, Keenan, Tsay, and BDS, along with modern approaches including kernel PCA and mutual information measures. Ultimately, the module aims to aid quantitative finance practitioners and researchers in determining the appropriateness of linear versus nonlinear modeling in financial applications.

# 1. Introduction

Many financial time series exhibit nonlinear dynamics that standard linear models do not capture. For quantitative finance practitioners, nonlinearity tests are crucial to decide if more advanced models such as nonlinear autoregressive models, regime-switching models, or machine learning techniques are necessary. The module fin_nln provides comprehensive testing capabilities for both univariate and multivariate nonlinearity detection, with integrated support for financial data analysis.

# 2. Module Structure

1. **fin_nln.univar**: 
    + Comprehensive univariate testing module with 16 statistical tests
    + Includes stationarity tests (ADF, KPSS, Phillips-Perron)
    + Normality tests (Jarque-Bera, Shapiro-Wilk)
    + Nonlinearity tests (BDS, Tsay, RESET, Keenan, ARCH-LM, White, Harvey-Collier)
    + Advanced measures (Hurst exponent, Lyapunov exponent, mutual information)

2. **fin_nln.multivar**: 
    + Multivariate testing module with 7 analytical functions
    + Linear relationship tests (VAR models, Granger causality, Johansen cointegration)
    + Nonlinear detection methods (multivariate BDS, mutual information, kernel PCA)

# 3. Available Tests

### Univariate Tests (16 functions)

**Stationarity & Pre-conditioning:**
- **Augmented Dickey-Fuller (ADF)** - Tests for unit roots/stationarity
- **KPSS Test** - Alternative stationarity test  
- **Phillips-Perron Test** - Unit root test robust to serial correlation

**Normality:**
- **Jarque-Bera Test** - Tests for normality based on skewness and kurtosis
- **Shapiro-Wilk Test** - Alternative normality test

**Nonlinearity Detection:**
- **BDS Test** - General nonlinearity test for time series
- **Tsay Test** - Tests for threshold nonlinearity
- **RESET Test** (Ramsey) - Tests for functional form misspecification
- **Keenan Test** - One-degree-of-freedom test for nonlinearity
- **ARCH-LM Test** - Tests for autoregressive conditional heteroskedasticity
- **White Test** - Tests for heteroskedasticity
- **Harvey-Collier Test** - Tests for linearity in regression

**Advanced Measures:**
- **Hurst Exponent** - Measures long-range dependence and self-similarity
- **Lyapunov Exponent** - Detects chaotic behavior
- **Mutual Information** - Measures nonlinear dependence
- **Ljung-Box Test** - Tests for autocorrelation in residuals

### Multivariate Tests (7 functions)

**Linear Relationships:**
- **VAR Model Analysis** - Vector autoregression for linear interdependencies
- **Granger Causality Test** - Tests if one series helps predict another
- **Johansen Cointegration** - Tests for long-run equilibrium relationships

**Nonlinear Analysis:**
- **Multivariate BDS (mBDS)** - Extension of BDS test to multivariate context
- **Mutual Information** - Measures nonlinear dependence between variables
- **Kernel PCA** - Nonlinear principal component analysis
- **Nonlinear PCA** - Alternative nonlinear dimensionality reduction

## 4. Installation & Quick Start

### Installation
```bash
pip install fin-nln
```

### Basic Usage Example

```python
import fin_nln
import yfinance as yf
import pandas as pd

# Download financial data
data = yf.download("SPY", start="2020-01-01", end="2023-01-01")
returns = data['Close'].pct_change().dropna()

# Run comprehensive univariate analysis
from fin_nln.univar.tests import run_all_univariate_tests
univariate_results = run_all_univariate_tests(returns.values)
print(univariate_results)

# Run multivariate analysis on multiple assets
tickers = ["SPY", "QQQ", "IWM"]
multi_data = yf.download(tickers, start="2020-01-01", end="2023-01-01")['Close']
multi_returns = multi_data.pct_change().dropna()

from fin_nln.multivar.tests import run_all_multivariate_tests
multivariate_results = run_all_multivariate_tests(multi_returns)
print(multivariate_results)

# Individual test examples
from fin_nln.univar.tests import bds_wrapper, tsay_test
from fin_nln.multivar.tests import mBDS, granger

# Test for nonlinearity with BDS test
bds_result = bds_wrapper(returns.values)
print(f"BDS Test p-value: {bds_result['p_value']}")

# Test for threshold nonlinearity
tsay_result = tsay_test(returns.values)
print(f"Tsay Test p-value: {tsay_result['p_value']}")

# Multivariate BDS test
mbds_result = mBDS(multi_returns)
print(f"Multivariate BDS p-value: {mbds_result['p_value']}")
```

## 5. Key Features

- **Comprehensive Testing**: 23 statistical tests covering linear and nonlinear relationships
- **Financial Focus**: Designed specifically for financial time series analysis  
- **Easy Integration**: Simple API with pandas/numpy compatibility
- **Production Ready**: Thoroughly tested with 49/49 test cases passing
- **Academic Rigor**: Implements peer-reviewed statistical methodologies

# 2. Surface-Level

1. **fin_nln.univar**: 
    + This module builds on the existing nlntstuniv functionality. 
    + It implements tests such as Ramsey’s RESET, Keenan’s test, Tsay’s test, and Terasvirta’s test for univariate time series.
    + Enhancements will be provided to support financial time series, including flexibility in selecting lag structures.

2. **fin_nln.multivar**: 
    + This module extends the functionality of the nlntstmultv module for multivariate nonlinearity testing. 
    + It is designed to accept and analyze multiple related financial time series. 
    + Applications include testing for nonlinear interactions among equity returns or asset classes.

# 3. Deeper Breakdown

### Univariate Linearity Tests

- **Ljung-Box Test** *from univariate.linearity.ljung_box_test import ljung_box_test*
    - Tests for autocorrelation in residuals (linear dependence).

- **Augmented Dickey-Fuller (ADF)**	*from univariate.linearity.adf_test import adf_test*
    - Stationarity check — helps determine linear time series behavior.

- **AR Model Fit & Residual Analysis** *from univariate.linearity.fit_ar_resid import fit_ar_resid*
    - Fit AR(p) and inspect residuals for linear structure.

### Univariate **Non**-Linearity Tests

- **Hurst Exponent (R/S Analysis)**	*from univariate.non-linearity.hurst_exponent import hurst_exponent*
    - Measures long memory (persistence or anti-persistence).

- **BDS Test** *from univariate.non-linearity.bds_test import bds_test*
    - Detects general nonlinearity in residuals of a time series.

- **Largest Lyapunov Exponent**	*from univariate.non-linearity.lyapunov_exponent import lyapunov_exponent*
    - Detects chaotic behavior via sensitivity to initial conditions.

- **Tsay Test**	*from univariate.non-linearity.tsay_test import tsay_test*
    - A test for threshold nonlinearity in autoregressive processes.

### Multivariate Linearity Tests

- **VAR Model Fit**	*from multivariate.linearity.var_model_fit*
    - Tests for linear interdependencies among variables.

- **Granger Causality Test** *from multivariate.linearity.granger_test import granger_test*
    - Determines if one time series linearly predicts another.

- **Cointegration Tests (Johansen)** *from multivariate.linearity.johansen_cointegration import johansen_cointegration*
    - Tests for linear long-run equilibrium relationships.

### Multivariate **Non**-Linearity Tests

- **Multivariate BDS Test (mBDS)** *from multivariate.non-linearity.mBDS_test import mBDS_test*
    - Extension of BDS to multivariate context.

- **Mutual Information** *from multivariate.non-linearity.mutual_info import mutual_info*
    - Measures nonlinear dependence among variables.

- **Nonlinear PCA / Kernel PCA** *from multivariate.non-linearity.nonlinear_pca.py import nonlinear_pca / from multivariate.non-linearity.kernel_pca import kernel_pca*
    - Uncovers nonlinear structures in multivariate data.

- **Cross-Lyapunov Exponents**	*from multivariate.non-linearity.lyapunov_exponent import lyapunov_exponent*
    - Tests chaotic synchronization across variables.
