# Copyright (c) 2021 University of Illinois and others. All rights reserved.
#
# This program and the accompanying materials are made available under the
# terms of the Mozilla Public License v2.0 which accompanies this distribution,
# and is available at https://www.mozilla.org/en-US/MPL/2.0/

import pandas as pd
from pyincore import BaseAnalysis


class SocialVulnerability(BaseAnalysis):
    """This analysis computes a social vulnerability score for per associated zone in census data.

    The computation extracts zoning and a social vulnerability score obtained by computing demographic features of
    interest against national average values.

    The output of the computation is a dataset CSV format.

    Contributors
        | Science: Elaina Sutley, Amin Enderami
        | Implementation:  Amin Enderami, Santiago Núñez-Corrales, and NCSA IN-CORE Dev Team

    Related publications

    Args:
        incore_client (IncoreClient): Service authentication.

    """

    def __init__(self, incore_client):
        super(SocialVulnerability, self).__init__(incore_client)

    def run(self):
        """Execute the social vulnerability analysis using known parameters."""
        year = int(self.get_parameter('year'))
        state = self.get_parameter('state')
        county = self.get_parameter('county')
        census_geo_level = self.get_parameter('census_geo_level')

        df_navs = pd.DataFrame(self.get_input_dataset('national_vulnerability_feature_averages').get_csv_reader())

        df_dem = pd.DataFrame(self.get_input_dataset('social_vulnerability_demographic_factors').get_csv_reader())

        # Make sure data types match
        df_dem["Factor 1: White, nonHispanic"] = df_dem["Factor 1: White, nonHispanic"].astype(float)
        df_dem["Factor 2: Home Owners"] = df_dem["Factor 2: Home Owners"].astype(float)
        df_dem["Factor 3: earning higher than national poverty rate"] =\
            df_dem["Factor 3: earning higher than national poverty rate"].astype(float)
        df_dem["Factor 4: over 25 with high school diploma or higher"] =\
            df_dem["Factor 4: over 25 with high school diploma or higher"].astype(float)
        df_dem["Factor 5: without disability age 18 to 65"] =\
            df_dem["Factor 5: without disability age 18 to 65"].astype(float)

        self.social_vulnerability_model(year, state, county, census_geo_level, df_navs, df_dem)

    def social_vulnerability_model(self, year, state, county, census_geo_level, df_navs, df_dem):
        """

        Args:
            year (int): string indicating the year of the analysis
            state (str): string indicating the state containing relevant zones
            county (str): string indicating the county containing relevant zones
            census_geo_level (str): string indicating the relevant census geographic level
            df_navs (pd.DataFrame): dataframe containing the national average values for vulnerability factors
            df_dem (pd.DataFrame): dataframe containing demographic factors required for the vulnerability score

        Returns:

        """

        # Compute the social vulnerability index
        df_sv = self.compute_svs(df_dem, df_navs)

        # Save into a CSV file
        result_name = self.get_parameter("result_name")
        self.set_result_csv_data("sv_result", df_sv,
                                 name=result_name + f"_{year}_{state}_{county}_{census_geo_level}",
                                 source="dataframe")

    @staticmethod
    def compute_svs(df, df_navs):
        """ Computation of the social vulnerability score and corresponding zoning

        Args:
            df (pd.DataFrame): dataframe for the census geographic unit of interest
            df_navs (pd.DataFrame): dataframe containing national average values

        Returns:
            pd.DataFrame: Social vulnerability score and corresponding zoning data
        """
        navs = df_navs['average'].astype(float).array

        df['R1'] = df['Factor 1: White, nonHispanic'] / navs[0]
        df['R2'] = df['Factor 2: Home Owners'] / navs[1]
        df['R3'] = df['Factor 3: earning higher than national poverty rate'] / navs[2]
        df['R4'] = df['Factor 4: over 25 with high school diploma or higher'] / navs[3]
        df['R5'] = df['Factor 5: without disability age 18 to 65'] / navs[4]
        df['SVS'] = df.apply(lambda row: (row['R1'] + row['R2'] + row['R3'] + row['R4'] + row['R5']) / 5, axis=1)

        maximum_nav = 1/navs
        std = abs(1 - (sum(maximum_nav) / len(maximum_nav))) / 3

        lb_2 = 1 - 1.5*std
        lb_1 = 1 - 0.5*std
        ub_1 = 1 + 0.5*std
        ub_2 = 1 + 1.5*std

        zones = []

        for svs in df['SVS'].tolist():
            if svs < lb_2:
                new_zone = 'High Vulnerable (zone5)'
            elif svs < lb_1:
                new_zone = 'Medium to High Vulnerable (zone4)'
            elif svs < ub_1:
                new_zone = 'Medium Vulnerable (zone3)'
            elif svs < ub_2:
                new_zone = 'Medium to Low Vulnerable (zone2)'
            elif svs > ub_2:
                new_zone = 'Low Vulnerable (zone1)'
            else:
                new_zone = 'No Data'
            zones.append(new_zone)

        df['zone'] = zones

        df = df.rename(columns={'': 'FIPS'}).sort_values(by='FIPS')

        return df

    def get_spec(self):
        """Get specifications of the housing serial recovery model.

        Returns:
            obj: A JSON object of specifications of the social vulnerability model.

        """
        return {
            'name': 'social-vulnerability',
            'description': 'Social vulnerability score model',
            'input_parameters': [
                {
                    'id': 'result_name',
                    'required': True,
                    'description': 'Result CSV dataset name',
                    'type': str
                },
                {
                    'id': 'year',
                    'required': True,
                    'description': 'Year for which the analysis is computed',
                    'type': int
                },
                {
                    'id': 'state',
                    'required': True,
                    'description': 'State for which the analysis is computed',
                    'type': str
                },
                {
                    'id': 'county',
                    'required': True,
                    'description': 'Country for which the analysis is computed',
                    'type': str
                },
                {
                    'id': 'census_geo_level',
                    'required': True,
                    'description': 'Census geographic level for which the analysis is computed',
                    'type': str
                }
            ],
            'input_datasets': [
                {
                    'id': 'national_vulnerability_feature_averages',
                    'required': True,
                    'description': 'A csv file with national vulnerability feature averages',
                    'type': 'incore:socialVulnerabilityFeatureAverages'
                },
                {
                    'id': 'social_vulnerability_demographic_factors',
                    'required': True,
                    'description': 'A csv file with social vulnerability demographic factors for a given geographic '
                                   'type',
                    'type': 'incore:socialVulnerabilityDemFactors'
                }
            ],
            'output_datasets': [
                {
                    'id': 'sv_result',
                    'parent_type': 'social_vulnerability_score',
                    'description': 'A csv file with zones containing demographic factors'
                                   'qualified by a social vulnerability score',
                    'type': 'incore:socialVulnerabilityScore'
                }
            ]
        }
