"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.GitHubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const decamelize = require("decamelize");
const YAML = require("yaml");
const aws_credentials_1 = require("./private/aws-credentials");
const github = require("./workflows-model");
const CDKOUT_ARTIFACT = 'cdk.out';
const ASSET_HASH_NAME = 'asset-hash';
/**
 * CDK Pipelines for GitHub workflows.
 */
class GitHubWorkflow extends pipelines_1.PipelineBase {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l;
        super(scope, id, props);
        this.jobOutputs = {};
        this.assetHashMap = {};
        this.stackProperties = {};
        this.cdkCliVersion = props.cdkCliVersion;
        this.preSynthed = (_b = props.preSynthed) !== null && _b !== void 0 ? _b : false;
        this.buildContainer = props.buildContainer;
        this.preBuildSteps = (_c = props.preBuildSteps) !== null && _c !== void 0 ? _c : [];
        this.postBuildSteps = (_d = props.postBuildSteps) !== null && _d !== void 0 ? _d : [];
        this.gitHubActionRoleArn = props.gitHubActionRoleArn;
        this.useGitHubActionRole = this.gitHubActionRoleArn ? true : false;
        this.awsCredentials = (_e = props.awsCredentials) !== null && _e !== void 0 ? _e : {
            accessKeyId: 'AWS_ACCESS_KEY_ID',
            secretAccessKey: 'AWS_SECRET_ACCESS_KEY',
        };
        this.dockerCredentials = (_f = props.dockerCredentials) !== null && _f !== void 0 ? _f : [];
        this.workflowPath = (_g = props.workflowPath) !== null && _g !== void 0 ? _g : '.github/workflows/deploy.yml';
        if (!this.workflowPath.endsWith('.yml') && !this.workflowPath.endsWith('.yaml')) {
            throw new Error('workflow file is expected to be a yaml file');
        }
        if (!this.workflowPath.includes('.github/workflows/')) {
            throw new Error('workflow files must be stored in the \'.github/workflows\' directory of your repository');
        }
        this.workflowName = (_h = props.workflowName) !== null && _h !== void 0 ? _h : 'deploy';
        this.workflowTriggers = (_j = props.workflowTriggers) !== null && _j !== void 0 ? _j : {
            push: { branches: ['main'] },
            workflowDispatch: {},
        };
        this.runner = (_k = props.runner) !== null && _k !== void 0 ? _k : github.Runner.UBUNTU_LATEST;
        this.publishAssetsAuthRegion = (_l = props.publishAssetsAuthRegion) !== null && _l !== void 0 ? _l : 'us-west-2';
    }
    /**
     * Deploy a single Stage by itself with options for further GitHub configuration.
     *
     * Add a Stage to the pipeline, to be deployed in sequence with other Stages added to the pipeline.
     * All Stacks in the stage will be deployed in an order automatically determined by their relative dependencies.
     */
    addStageWithGitHubOptions(stage, options) {
        const stageDeployment = this.addStage(stage, options);
        // keep track of GitHub specific options
        const stacks = stageDeployment.stacks;
        if (options === null || options === void 0 ? void 0 : options.gitHubEnvironment) {
            this.addStackProps(stacks, 'environment', options.gitHubEnvironment);
        }
        if (options === null || options === void 0 ? void 0 : options.stackCapabilities) {
            this.addStackProps(stacks, 'capabilities', options.stackCapabilities);
        }
        return stageDeployment;
    }
    addStackProps(stacks, key, value) {
        for (const stack of stacks) {
            this.stackProperties[stack.stackArtifactId] = {
                ...this.stackProperties[stack.stackArtifactId],
                [key]: value,
            };
        }
    }
    doBuildPipeline() {
        var _b;
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error('The GitHub Workflow must be defined in the scope of an App');
        }
        const cdkoutDir = app.outdir;
        const jobs = new Array();
        const structure = new helpers_internal_1.PipelineGraph(this, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false,
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!helpers_internal_1.isGraph(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    const job = this.jobForNode(node, {
                        assemblyDir: cdkoutDir,
                        structure,
                    });
                    if (job) {
                        jobs.push(job);
                    }
                }
            }
        }
        // convert jobs to a map and make sure there are no duplicates
        const jobmap = {};
        for (const job of jobs) {
            if (job.id in jobmap) {
                throw new Error(`duplicate job id ${job.id}`);
            }
            jobmap[job.id] = snakeCaseKeys(job.definition);
        }
        // Update jobs with late-bound output requests
        this.insertJobOutputs(jobmap);
        const workflow = {
            name: this.workflowName,
            on: snakeCaseKeys(this.workflowTriggers, '_'),
            jobs: jobmap,
        };
        // write as a yaml file
        const yaml = YAML.stringify(workflow, {
            indent: 2,
        });
        // create directory if it does not exist
        fs_1.mkdirSync(path.dirname(this.workflowPath), { recursive: true });
        // GITHUB_WORKFLOW is set when GitHub Actions is running the workflow.
        // see: https://docs.github.com/en/actions/learn-github-actions/environment-variables#default-environment-variables
        const diffProtection = (_b = this.node.tryGetContext('cdk-pipelines-github:diffProtection')) !== null && _b !== void 0 ? _b : true;
        if (diffProtection && process.env.GITHUB_WORKFLOW === this.workflowName) {
            // check if workflow file has changed
            if (!fs_1.existsSync(this.workflowPath) || yaml !== fs_1.readFileSync(this.workflowPath, 'utf8')) {
                throw new Error(`Please commit the updated workflow file ${path.relative(__dirname, this.workflowPath)} when you change your pipeline definition.`);
            }
        }
        fs_1.writeFileSync(this.workflowPath, yaml);
    }
    insertJobOutputs(jobmap) {
        for (const [jobId, jobOutputs] of Object.entries(this.jobOutputs)) {
            jobmap[jobId] = {
                ...jobmap[jobId],
                outputs: {
                    ...jobmap[jobId].outputs,
                    ...this.renderJobOutputs(jobOutputs),
                },
            };
        }
    }
    renderJobOutputs(outputs) {
        const renderedOutputs = {};
        for (const output of outputs) {
            renderedOutputs[output.outputName] = `\${{ steps.${output.stepId}.outputs.${output.outputName} }}`;
        }
        return renderedOutputs;
    }
    /**
     * Make an action from the given node and/or step
     */
    jobForNode(node, options) {
        var _b, _c;
        switch ((_b = node.data) === null || _b === void 0 ? void 0 : _b.type) {
            // Nothing for these, they are groupings (shouldn't even have popped up here)
            case 'group':
            case 'stack-group':
            case undefined:
                throw new Error(`jobForNode: did not expect to get group nodes: ${(_c = node.data) === null || _c === void 0 ? void 0 : _c.type}`);
            case 'self-update':
                throw new Error('GitHub Workflows does not support self mutation');
            case 'publish-assets':
                return this.jobForAssetPublish(node, node.data.assets, options);
            case 'prepare':
                throw new Error('"prepare" is not supported by GitHub Worflows');
            case 'execute':
                return this.jobForDeploy(node, node.data.stack, node.data.captureOutputs);
            case 'step':
                if (node.data.isBuildStep) {
                    return this.jobForBuildStep(node, node.data.step);
                }
                else if (node.data.step instanceof pipelines_1.ShellStep) {
                    return this.jobForScriptStep(node, node.data.step);
                }
                else {
                    throw new Error(`unsupported step type: ${node.data.step.constructor.name}`);
                }
        }
    }
    jobForAssetPublish(node, assets, options) {
        if (assets.length === 0) {
            throw new Error('Asset Publish step must have at least 1 asset');
        }
        const installSuffix = this.cdkCliVersion ? `@${this.cdkCliVersion}` : '';
        const cdkoutDir = options.assemblyDir;
        const jobId = node.uniqueId;
        const assetId = assets[0].assetId;
        // check if asset is docker asset and if we have docker credentials
        const dockerLoginSteps = [];
        if (node.uniqueId.includes('DockerAsset') && this.dockerCredentials.length > 0) {
            for (const creds of this.dockerCredentials) {
                dockerLoginSteps.push(...this.stepsToConfigureDocker(creds));
            }
        }
        // create one file and make one step
        const relativeToAssembly = (p) => path.posix.join(cdkoutDir, path.relative(path.resolve(cdkoutDir), p));
        const fileContents = ['set -x'].concat(assets.map((asset) => {
            return `npx cdk-assets --path "${relativeToAssembly(asset.assetManifestPath)}" --verbose publish "${asset.assetSelector}"`;
        }));
        // we need the jobId to reference the outputs later
        this.assetHashMap[assetId] = jobId;
        fileContents.push(`echo '::set-output name=${ASSET_HASH_NAME}::${assetId}'`);
        const publishStepFile = path.join(cdkoutDir, `publish-${jobId}-step.sh`);
        fs_1.mkdirSync(path.dirname(publishStepFile), { recursive: true });
        fs_1.writeFileSync(publishStepFile, fileContents.join('\n'), { encoding: 'utf-8' });
        const publishStep = {
            id: 'Publish',
            name: `Publish ${jobId}`,
            run: `/bin/bash ./cdk.out/${path.relative(cdkoutDir, publishStepFile)}`,
        };
        return {
            id: jobId,
            definition: {
                name: `Publish Assets ${jobId}`,
                needs: this.renderDependencies(node),
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                runsOn: this.runner.runsOn,
                outputs: {
                    [ASSET_HASH_NAME]: `\${{ steps.Publish.outputs.${ASSET_HASH_NAME} }}`,
                },
                steps: [
                    ...this.stepsToDownloadAssembly(cdkoutDir),
                    {
                        name: 'Install',
                        run: `npm install --no-save cdk-assets${installSuffix}`,
                    },
                    ...this.stepsToConfigureAws(this.useGitHubActionRole, { region: this.publishAssetsAuthRegion }),
                    ...dockerLoginSteps,
                    publishStep,
                ],
            },
        };
    }
    jobForDeploy(node, stack, _captureOutputs) {
        var _b, _c;
        const region = stack.region;
        const account = stack.account;
        if (!region || !account) {
            throw new Error('"account" and "region" are required');
        }
        if (!stack.templateUrl) {
            throw new Error(`unable to determine template URL for stack ${stack.stackArtifactId}`);
        }
        const resolve = (s) => {
            return cx_api_1.EnvironmentPlaceholders.replace(s, {
                accountId: account,
                region: region,
                partition: 'aws',
            });
        };
        const replaceAssetHash = (template) => {
            var _b;
            const hash = path.parse((_b = template.split('/').pop()) !== null && _b !== void 0 ? _b : '').name;
            if (this.assetHashMap[hash] === undefined) {
                throw new Error(`Template asset hash ${hash} not found.`);
            }
            return template.replace(hash, `\${{ needs.${this.assetHashMap[hash]}.outputs.${ASSET_HASH_NAME} }}`);
        };
        const params = {
            'name': stack.stackName,
            'template': replaceAssetHash(resolve(stack.templateUrl)),
            'no-fail-on-empty-changeset': '1',
        };
        const capabilities = (_b = this.stackProperties[stack.stackArtifactId]) === null || _b === void 0 ? void 0 : _b.capabilities;
        if (capabilities) {
            params.capabilities = Array(capabilities).join(',');
        }
        if (stack.executionRoleArn) {
            params['role-arn'] = resolve(stack.executionRoleArn);
        }
        const assumeRoleArn = stack.assumeRoleArn ? resolve(stack.assumeRoleArn) : undefined;
        return {
            id: node.uniqueId,
            definition: {
                name: `Deploy ${stack.stackArtifactId}`,
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                ...(((_c = this.stackProperties[stack.stackArtifactId]) === null || _c === void 0 ? void 0 : _c.environment) ? {
                    environment: this.stackProperties[stack.stackArtifactId].environment,
                } : {}),
                needs: this.renderDependencies(node),
                runsOn: this.runner.runsOn,
                steps: [
                    ...this.stepsToConfigureAws(this.useGitHubActionRole, { region, assumeRoleArn }),
                    {
                        id: 'Deploy',
                        uses: 'aws-actions/aws-cloudformation-github-deploy@v1',
                        with: params,
                    },
                ],
            },
        };
    }
    jobForBuildStep(node, step) {
        if (!(step instanceof pipelines_1.ShellStep)) {
            throw new Error('synthStep must be a ScriptStep');
        }
        if (step.inputs.length > 0) {
            throw new Error('synthStep cannot have inputs');
        }
        if (step.outputs.length > 1) {
            throw new Error('synthStep must have a single output');
        }
        if (!step.primaryOutput) {
            throw new Error('synthStep requires a primaryOutput which contains cdk.out');
        }
        const cdkOut = step.outputs[0];
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: 'Synthesize',
                permissions: {
                    contents: github.JobPermission.READ,
                    // The Synthesize job does not use the GitHub Action Role on its own, but it's possible
                    // that it is being used in the preBuildSteps.
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                container: this.buildContainer,
                steps: [
                    ...this.stepsToCheckout(),
                    ...this.preBuildSteps,
                    ...installSteps,
                    {
                        name: 'Build',
                        run: step.commands.join('\n'),
                    },
                    ...this.postBuildSteps,
                    ...this.stepsToUploadAssembly(cdkOut.directory),
                ],
            },
        };
    }
    /**
     * Searches for the stack that produced the output via the current
     * job's dependencies.
     *
     * This function should always find a stack, since it is guaranteed
     * that a CfnOutput comes from a referenced stack.
     */
    findStackOfOutput(ref, node) {
        var _b;
        for (const dep of node.allDeps) {
            if (((_b = dep.data) === null || _b === void 0 ? void 0 : _b.type) === 'execute' && ref.isProducedBy(dep.data.stack)) {
                return dep.uniqueId;
            }
        }
        // Should never happen
        throw new Error(`The output ${ref.outputName} is not referenced by any of the dependent stacks!`);
    }
    addJobOutput(jobId, output) {
        if (this.jobOutputs[jobId] === undefined) {
            this.jobOutputs[jobId] = [output];
        }
        else {
            this.jobOutputs[jobId].push(output);
        }
    }
    jobForScriptStep(node, step) {
        const envVariables = {};
        for (const [envName, ref] of Object.entries(step.envFromCfnOutputs)) {
            const jobId = this.findStackOfOutput(ref, node);
            this.addJobOutput(jobId, {
                outputName: ref.outputName,
                stepId: 'Deploy',
            });
            envVariables[envName] = `\${{ needs.${jobId}.outputs.${ref.outputName} }}`;
        }
        const downloadInputs = new Array();
        const uploadOutputs = new Array();
        for (const input of step.inputs) {
            downloadInputs.push({
                uses: 'actions/download-artifact@v2',
                with: {
                    name: input.fileSet.id,
                    path: input.directory,
                },
            });
        }
        for (const output of step.outputs) {
            uploadOutputs.push({
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: output.fileSet.id,
                    path: output.directory,
                },
            });
        }
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: {
                    ...step.env,
                    ...envVariables,
                },
                steps: [
                    ...downloadInputs,
                    ...installSteps,
                    { run: step.commands.join('\n') },
                    ...uploadOutputs,
                ],
            },
        };
    }
    stepsToConfigureAws(openId, { region, assumeRoleArn }) {
        function getDeployRole(arn) {
            return arn.replace('cfn-exec', 'deploy');
        }
        let steps = [];
        if (openId) {
            steps.push(aws_credentials_1.awsCredentialStep('Authenticate Via OIDC Role', {
                region,
                gitHubActionRoleArn: this.gitHubActionRoleArn,
            }));
            if (assumeRoleArn) {
                // Result of initial credentials with GitHub Action role are these environment variables
                steps.push(aws_credentials_1.awsCredentialStep('Assume CDK Deploy Role', {
                    region,
                    accessKeyId: '${{ env.AWS_ACCESS_KEY_ID }}',
                    secretAccessKey: '${{ env.AWS_SECRET_ACCESS_KEY }}',
                    sessionToken: '${{ env.AWS_SESSION_TOKEN }}',
                    roleToAssume: getDeployRole(assumeRoleArn),
                }));
            }
        }
        else {
            steps.push(aws_credentials_1.awsCredentialStep('Authenticate Via GitHub Secrets', {
                region,
                accessKeyId: `\${{ secrets.${this.awsCredentials.accessKeyId} }}`,
                secretAccessKey: `\${{ secrets.${this.awsCredentials.secretAccessKey} }}`,
                ...(this.awsCredentials.sessionToken ? {
                    sessionToken: `\${{ secrets.${this.awsCredentials.sessionToken} }}`,
                } : undefined),
                roleToAssume: assumeRoleArn,
            }));
        }
        return steps;
    }
    stepsToConfigureDocker(dockerCredential) {
        let params;
        if (dockerCredential.name === 'docker') {
            params = {
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        else if (dockerCredential.name === 'ecr') {
            params = {
                registry: dockerCredential.registry,
            };
        }
        else {
            params = {
                registry: dockerCredential.registry,
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        return [
            {
                uses: 'docker/login-action@v1',
                with: params,
            },
        ];
    }
    stepsToDownloadAssembly(targetDir) {
        if (this.preSynthed) {
            return this.stepsToCheckout();
        }
        return [{
                name: `Download ${CDKOUT_ARTIFACT}`,
                uses: 'actions/download-artifact@v2',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: targetDir,
                },
            }];
    }
    stepsToCheckout() {
        return [{
                name: 'Checkout',
                uses: 'actions/checkout@v2',
            }];
    }
    stepsToUploadAssembly(dir) {
        if (this.preSynthed) {
            return [];
        }
        return [{
                name: `Upload ${CDKOUT_ARTIFACT}`,
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: dir,
                },
            }];
    }
    renderDependencies(node) {
        const deps = new Array();
        for (const d of node.allDeps) {
            if (d instanceof helpers_internal_1.Graph) {
                deps.push(...d.allLeaves().nodes);
            }
            else {
                deps.push(d);
            }
        }
        return deps.map(x => x.uniqueId);
    }
}
exports.GitHubWorkflow = GitHubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GitHubWorkflow[_a] = { fqn: "cdk-pipelines-github.GitHubWorkflow", version: "0.2.92" };
function snakeCaseKeys(obj, sep = '-') {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(o => snakeCaseKeys(o, sep));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        // we don't want to snake case environment variables
        if (k !== 'env' && typeof v === 'object' && v != null) {
            v = snakeCaseKeys(v);
        }
        result[decamelize(k, { separator: sep })] = v;
    }
    return result;
}
function* flatten(xs) {
    for (const x of xs) {
        for (const y of x) {
            yield y;
        }
    }
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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