/*
 * Decompiled with CFR 0.152.
 */
package de.siegmar.fastcsv.reader;

import de.siegmar.fastcsv.reader.BomHeader;
import de.siegmar.fastcsv.reader.BomUtil;
import de.siegmar.fastcsv.reader.CommentStrategy;
import de.siegmar.fastcsv.reader.CsvCallbackHandler;
import de.siegmar.fastcsv.reader.CsvIndex;
import de.siegmar.fastcsv.reader.CsvParser;
import de.siegmar.fastcsv.reader.CsvRecord;
import de.siegmar.fastcsv.reader.CsvRecordHandler;
import de.siegmar.fastcsv.reader.CsvScanner;
import de.siegmar.fastcsv.reader.RandomAccessFileInputStream;
import de.siegmar.fastcsv.reader.RecordWrapper;
import de.siegmar.fastcsv.reader.StatusListener;
import de.siegmar.fastcsv.util.Preconditions;
import de.siegmar.fastcsv.util.Util;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.RandomAccessFile;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.StringJoiner;
import java.util.concurrent.atomic.AtomicLong;

public final class IndexedCsvReader<T>
implements Closeable {
    private final Path file;
    private final Charset charset;
    private final char fieldSeparator;
    private final char quoteCharacter;
    private final CommentStrategy commentStrategy;
    private final char commentCharacter;
    private final boolean acceptCharsAfterQuotes;
    private final int pageSize;
    private final RandomAccessFile raf;
    private final CsvCallbackHandler<T> csvRecordHandler;
    private final CsvParser csvParser;
    private final CsvIndex csvIndex;

    IndexedCsvReader(Path file, Charset defaultCharset, char fieldSeparator, char quoteCharacter, CommentStrategy commentStrategy, char commentCharacter, boolean acceptCharsAfterQuotes, int pageSize, CsvCallbackHandler<T> csvRecordHandler, CsvIndex csvIndex, StatusListener statusListener) throws IOException {
        int bomHeaderLength;
        Preconditions.checkArgument(!Util.containsDupe(fieldSeparator, quoteCharacter, commentCharacter), "Control characters must differ (fieldSeparator=%s, quoteCharacter=%s, commentCharacter=%s)", Character.valueOf(fieldSeparator), Character.valueOf(quoteCharacter), Character.valueOf(commentCharacter));
        this.file = file;
        this.fieldSeparator = fieldSeparator;
        this.quoteCharacter = quoteCharacter;
        this.commentStrategy = commentStrategy;
        this.commentCharacter = commentCharacter;
        this.acceptCharsAfterQuotes = acceptCharsAfterQuotes;
        this.pageSize = pageSize;
        this.csvRecordHandler = csvRecordHandler;
        Optional<BomHeader> optionalBomHeader = IndexedCsvReader.detectBom(file, statusListener);
        if (optionalBomHeader.isEmpty()) {
            this.charset = defaultCharset;
            bomHeaderLength = 0;
        } else {
            BomHeader bomHeader = optionalBomHeader.get();
            this.charset = bomHeader.getCharset();
            bomHeaderLength = bomHeader.getLength();
        }
        this.csvIndex = csvIndex != null ? IndexedCsvReader.validatePrebuiltIndex(file, bomHeaderLength, (byte)fieldSeparator, (byte)quoteCharacter, commentStrategy, (byte)commentCharacter, csvIndex) : this.buildIndex(bomHeaderLength, statusListener);
        this.raf = new RandomAccessFile(file.toFile(), "r");
        this.csvParser = new CsvParser(fieldSeparator, quoteCharacter, commentStrategy, commentCharacter, acceptCharsAfterQuotes, csvRecordHandler, new InputStreamReader((InputStream)new RandomAccessFileInputStream(this.raf), this.charset));
    }

    private static Optional<BomHeader> detectBom(Path file, StatusListener statusListener) throws IOException {
        try {
            return BomUtil.detectCharset(file);
        }
        catch (IOException e) {
            statusListener.onError(e);
            throw e;
        }
    }

    private static CsvIndex validatePrebuiltIndex(Path file, int bomHeaderLength, byte fieldSeparator, byte quoteCharacter, CommentStrategy commentStrategy, byte commentCharacter, CsvIndex csvIndex) throws IOException {
        String expectedSignature = new StringJoiner(", ").add("bomHeaderLength=" + bomHeaderLength).add("fileSize=" + Files.size(file)).add("fieldSeparator=" + fieldSeparator).add("quoteCharacter=" + quoteCharacter).add("commentStrategy=" + String.valueOf((Object)commentStrategy)).add("commentCharacter=" + commentCharacter).toString();
        String actualSignature = new StringJoiner(", ").add("bomHeaderLength=" + csvIndex.getBomHeaderLength()).add("fileSize=" + csvIndex.getFileSize()).add("fieldSeparator=" + csvIndex.getFieldSeparator()).add("quoteCharacter=" + csvIndex.getQuoteCharacter()).add("commentStrategy=" + String.valueOf((Object)csvIndex.getCommentStrategy())).add("commentCharacter=" + csvIndex.getCommentCharacter()).toString();
        Preconditions.checkArgument(expectedSignature.equals(actualSignature), "Index does not match! Expected: %s; Actual: %s", expectedSignature, actualSignature);
        return csvIndex;
    }

    private CsvIndex buildIndex(int bomHeaderLength, StatusListener statusListener) throws IOException {
        CsvIndex csvIndex;
        block8: {
            ScannerListener listener = new ScannerListener(statusListener);
            SeekableByteChannel channel = Files.newByteChannel(this.file, StandardOpenOption.READ);
            try {
                statusListener.onInit(channel.size());
                new CsvScanner(channel, bomHeaderLength, (byte)this.fieldSeparator, (byte)this.quoteCharacter, this.commentStrategy, (byte)this.commentCharacter, listener).scan();
                CsvIndex idx = new CsvIndex(bomHeaderLength, channel.size(), (byte)this.fieldSeparator, (byte)this.quoteCharacter, this.commentStrategy, (byte)this.commentCharacter, listener.recordCounter.get(), listener.pageOffsets);
                statusListener.onComplete();
                csvIndex = idx;
                if (channel == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (channel != null) {
                        try {
                            channel.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (Throwable t) {
                    statusListener.onError(t);
                    throw t;
                }
            }
            channel.close();
        }
        return csvIndex;
    }

    public static IndexedCsvReaderBuilder builder() {
        return new IndexedCsvReaderBuilder();
    }

    public CsvIndex getIndex() {
        return this.csvIndex;
    }

    public List<T> readPage(int page) throws IOException {
        Preconditions.checkArgument(page >= 0, "page must be >= 0");
        return this.readPage(this.csvIndex.getPage(page));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<T> readPage(CsvIndex.CsvPage page) throws IOException {
        ArrayList<T> ret = new ArrayList<T>(this.pageSize);
        RandomAccessFile randomAccessFile = this.raf;
        synchronized (randomAccessFile) {
            this.raf.seek(page.getOffset());
            this.csvParser.reset(page.getStartingLineNumber() - 1L);
            for (int i = 0; i < this.pageSize && this.csvParser.parse(); ++i) {
                RecordWrapper<T> rec = this.csvRecordHandler.buildRecord();
                if (rec == null) continue;
                ret.add(rec.getWrappedRecord());
            }
            return ret;
        }
    }

    @Override
    public void close() throws IOException {
        this.csvParser.close();
    }

    public String toString() {
        return new StringJoiner(", ", IndexedCsvReader.class.getSimpleName() + "[", "]").add("file=" + String.valueOf(this.file)).add("charset=" + String.valueOf(this.charset)).add("fieldSeparator=" + this.fieldSeparator).add("quoteCharacter=" + this.quoteCharacter).add("commentStrategy=" + String.valueOf((Object)this.commentStrategy)).add("commentCharacter=" + this.commentCharacter).add("acceptCharsAfterQuotes=" + this.acceptCharsAfterQuotes).add("pageSize=" + this.pageSize).add("index=" + String.valueOf(this.csvIndex)).toString();
    }

    private final class ScannerListener
    implements CsvScanner.CsvListener {
        private final StatusListener statusListener;
        private final List<CsvIndex.CsvPage> pageOffsets = new ArrayList<CsvIndex.CsvPage>();
        private final AtomicLong recordCounter = new AtomicLong();
        private long startingLineNumber = 1L;

        private ScannerListener(StatusListener statusListener) {
            this.statusListener = statusListener;
        }

        @Override
        public void onReadBytes(int readCnt) {
            this.statusListener.onReadBytes(readCnt);
        }

        @Override
        public void startOffset(long offset) {
            if (this.recordCounter.getAndIncrement() % (long)IndexedCsvReader.this.pageSize == 0L) {
                this.pageOffsets.add(new CsvIndex.CsvPage(offset, this.startingLineNumber));
            }
        }

        @Override
        public void onReadRecord() {
            ++this.startingLineNumber;
            this.statusListener.onReadRecord();
        }

        @Override
        public void additionalLine() {
            ++this.startingLineNumber;
        }
    }

    public static final class IndexedCsvReaderBuilder {
        private static final int MAX_BASE_ASCII = 127;
        private static final int DEFAULT_PAGE_SIZE = 100;
        private static final int MIN_PAGE_SIZE = 1;
        private char fieldSeparator = (char)44;
        private char quoteCharacter = (char)34;
        private CommentStrategy commentStrategy = CommentStrategy.NONE;
        private char commentCharacter = (char)35;
        private boolean acceptCharsAfterQuotes = true;
        private StatusListener statusListener;
        private int pageSize = 100;
        private CsvIndex csvIndex;

        private IndexedCsvReaderBuilder() {
        }

        public IndexedCsvReaderBuilder fieldSeparator(char fieldSeparator) {
            IndexedCsvReaderBuilder.checkControlCharacter(fieldSeparator);
            this.fieldSeparator = fieldSeparator;
            return this;
        }

        public IndexedCsvReaderBuilder quoteCharacter(char quoteCharacter) {
            IndexedCsvReaderBuilder.checkControlCharacter(quoteCharacter);
            this.quoteCharacter = quoteCharacter;
            return this;
        }

        public IndexedCsvReaderBuilder commentStrategy(CommentStrategy commentStrategy) {
            Preconditions.checkArgument(commentStrategy != CommentStrategy.SKIP, "CommentStrategy SKIP is not supported in IndexedCsvReader");
            this.commentStrategy = commentStrategy;
            return this;
        }

        public IndexedCsvReaderBuilder commentCharacter(char commentCharacter) {
            IndexedCsvReaderBuilder.checkControlCharacter(commentCharacter);
            this.commentCharacter = commentCharacter;
            return this;
        }

        public IndexedCsvReaderBuilder acceptCharsAfterQuotes(boolean acceptCharsAfterQuotes) {
            this.acceptCharsAfterQuotes = acceptCharsAfterQuotes;
            return this;
        }

        public IndexedCsvReaderBuilder statusListener(StatusListener statusListener) {
            this.statusListener = statusListener;
            return this;
        }

        public IndexedCsvReaderBuilder index(CsvIndex csvIndex) {
            this.csvIndex = csvIndex;
            return this;
        }

        public IndexedCsvReaderBuilder pageSize(int pageSize) {
            Preconditions.checkArgument(pageSize >= 1, "pageSize must be >= %d", 1);
            this.pageSize = pageSize;
            return this;
        }

        private static void checkControlCharacter(char controlChar) {
            Preconditions.checkArgument(!Util.isNewline(controlChar), "A newline character must not be used as control character");
            Preconditions.checkArgument(controlChar <= '\u007f', "Multibyte control characters are not supported in IndexedCsvReader: '%s' (value: %d)", Character.valueOf(controlChar), controlChar);
        }

        public IndexedCsvReader<CsvRecord> ofCsvRecord(Path file) throws IOException {
            return this.build(new CsvRecordHandler(), file, StandardCharsets.UTF_8);
        }

        public IndexedCsvReader<CsvRecord> ofCsvRecord(Path file, Charset charset) throws IOException {
            return this.build(new CsvRecordHandler(), file, charset);
        }

        public <T> IndexedCsvReader<T> build(CsvCallbackHandler<T> callbackHandler, Path file) throws IOException {
            return this.build(callbackHandler, file, StandardCharsets.UTF_8);
        }

        public <T> IndexedCsvReader<T> build(CsvCallbackHandler<T> callbackHandler, Path file, Charset charset) throws IOException {
            Objects.requireNonNull(callbackHandler, "callbackHandler must not be null");
            Objects.requireNonNull(file, "file must not be null");
            Objects.requireNonNull(charset, "charset must not be null");
            StatusListener sl = this.statusListener != null ? this.statusListener : new StatusListener(){};
            return new IndexedCsvReader<T>(file, charset, this.fieldSeparator, this.quoteCharacter, this.commentStrategy, this.commentCharacter, this.acceptCharsAfterQuotes, this.pageSize, callbackHandler, this.csvIndex, sl);
        }
    }
}

