"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib/");
test('Cloudfront Distribution Included', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "CustomErrorResponses": [
                {
                    "ErrorCode": 403,
                    "ResponseCode": 200,
                    "ResponsePagePath": "/index.html"
                },
                {
                    "ErrorCode": 404,
                    "ResponseCode": 200,
                    "ResponsePagePath": "/index.html"
                }
            ],
            "DefaultCacheBehavior": {
                "ViewerProtocolPolicy": "redirect-to-https"
            },
            "DefaultRootObject": "index.html",
            "HttpVersion": "http2",
            "IPV6Enabled": true,
            "PriceClass": "PriceClass_100",
            "ViewerCertificate": {
                "CloudFrontDefaultCertificate": true
            }
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Cloudfront Distribution Included - with non-default error-doc cfg set', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "CustomErrorResponses": [
                {
                    "ErrorCode": 403,
                    "ResponseCode": 200,
                    "ResponsePagePath": "/error.html"
                },
                {
                    "ErrorCode": 404,
                    "ResponseCode": 200,
                    "ResponsePagePath": "/error.html"
                }
            ],
            "DefaultCacheBehavior": {
                "ViewerProtocolPolicy": "redirect-to-https"
            },
            "DefaultRootObject": "index.html",
            "HttpVersion": "http2",
            "IPV6Enabled": true,
            "PriceClass": "PriceClass_100",
            "ViewerCertificate": {
                "CloudFrontDefaultCertificate": true
            }
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Cloudfront With Custom Cert and Aliases', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com']
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "Aliases": [
                "www.test.com"
            ],
            "ViewerCertificate": {
                "AcmCertificateArn": "arn:1234",
                "SslSupportMethod": "sni-only"
            }
        }
    }));
});
test('Basic Site Setup', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Basic Site Setup with Error Doc set', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Basic Site Setup, Encrypted Bucket', () => {
    let stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Cloudfront With Encrypted Bucket', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true });
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com']
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "Aliases": [
                "www.test.com"
            ],
            "ViewerCertificate": {
                "AcmCertificateArn": "arn:1234",
                "SslSupportMethod": "sni-only"
            }
        }
    }));
});
test('Basic Site Setup, IP Filter', () => {
    let stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true, ipFilter: true, ipList: ['1.1.1.1'] })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Condition": {
                        "IpAddress": {
                            "aws:SourceIp": [
                                "1.1.1.1"
                            ]
                        }
                    },
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
});
test('Cloudfront With IP Filter', () => {
    let stack = new core_1.Stack();
    // WHEN
    let deploy = new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true, ipFilter: true, ipList: ['1.1.1.1'] });
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com']
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    "Action": "s3:GetObject",
                    "Condition": {
                        "IpAddress": {
                            "aws:SourceIp": [
                                "1.1.1.1"
                            ]
                        }
                    },
                    "Effect": "Allow",
                    "Principal": "*"
                }
            ]
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "Aliases": [
                "www.test.com"
            ],
            "ViewerCertificate": {
                "AcmCertificateArn": "arn:1234",
                "SslSupportMethod": "sni-only"
            }
        }
    }));
});
test('Create From Hosted Zone', () => {
    let app = new core_1.App();
    let stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234'
        }
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "Aliases": [
                "www.cdkspadeploy.com"
            ],
            "ViewerCertificate": {
                "SslSupportMethod": "sni-only"
            }
        }
    }));
});
test('Create From Hosted Zone with Error Bucket', () => {
    let app = new core_1.App();
    let stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234'
        }
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website'
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html'
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        "DistributionConfig": {
            "Aliases": [
                "www.cdkspadeploy.com"
            ],
            "ViewerCertificate": {
                "SslSupportMethod": "sni-only"
            }
        }
    }));
});
test('Basic Site Setup, URL Output Enabled With Name', () => {
    let stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true,
        exportWebsiteUrlName: exportName
    });
    // THEN  
    assert_1.expect(stack).to(assert_1.haveOutput({
        exportName: exportName
    }));
});
test('Basic Site Setup, URL Output Enabled With No Name', () => {
    let stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true
    });
    // THEN  
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName: exportName
    }));
});
test('Basic Site Setup, URL Output Not Enabled', () => {
    let stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: false
    });
    // THEN  
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName: exportName
    }));
});
//# sourceMappingURL=data:application/json;base64,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