"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.configureMongo = exports.MongoDbConfigure = void 0;
/* eslint-disable no-console */
const child_process_1 = require("child_process");
const util_1 = require("util");
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
const custom_resource_1 = require("../lib/custom-resource");
const filesystem_1 = require("../lib/filesystem");
const secrets_manager_1 = require("../lib/secrets-manager");
const types_1 = require("./types");
const exec = util_1.promisify(child_process_1.exec);
class MongoDbConfigure extends custom_resource_1.SimpleCustomResource {
    constructor(secretsManagerClient) {
        super();
        this.secretsManagerClient = secretsManagerClient;
    }
    /**
     * @inheritdoc
     */
    /* istanbul ignore next */ // @ts-ignore
    validateInput(data) {
        return types_1.implementsIMongoDbConfigureResource(data);
    }
    /**
     * @inheritdoc
     */
    // @ts-ignore  -- we do not use the physicalId
    async doCreate(physicalId, resourceProperties) {
        const mongoDbDriver = this.installMongoDbDriver();
        const mongoClient = await this.mongoLogin(mongoDbDriver, resourceProperties.Connection);
        try {
            if (resourceProperties.PasswordAuthUsers) {
                const adminDb = mongoClient.db('admin');
                for (const userArn of resourceProperties.PasswordAuthUsers) {
                    if (!await this.createPasswordAuthUser(adminDb, userArn)) {
                        // Note: Intentionally not including the data as part of this message. It may contain secrets, and including it will leak those secrets.
                        console.log(`User in '${userArn}' already exists in the MongoDB. No action performed for this user.`);
                    }
                }
            }
            if (resourceProperties.X509AuthUsers) {
                const externalDb = mongoClient.db('$external');
                for (const x509User of resourceProperties.X509AuthUsers) {
                    if (!await this.createX509AuthUser(externalDb, x509User)) {
                        // Note: Intentionally not including the data as part of this message. It may contain secrets, and including it will leak those secrets.
                        console.log(`User in '${x509User.Certificate}' already exists in the MongoDB. No action performed for this user.`);
                    }
                }
            }
        }
        finally {
            console.log('Closing Mongo connection');
            await mongoClient.close();
        }
        return undefined;
    }
    /**
     * @inheritdoc
     */
    /* istanbul ignore next */ // @ts-ignore
    async doDelete(physicalId, resourceProperties) {
        // Nothing to do -- we don't modify any existing DB contents.
        return;
    }
    /**
     * Installs the official NodeJS MongoDB driver into /tmp, and returns the module object for it.
     */
    /* istanbul ignore next */
    installMongoDbDriver() {
        console.log('Installing latest MongoDB Driver for NodeJS from npmjs.org');
        // Both HOME and --prefix are needed here because /tmp is the only writable location
        child_process_1.execSync('HOME=/tmp npm install mongodb@3 --production --no-package-lock --no-save --prefix /tmp');
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require('/tmp/node_modules/mongodb');
    }
    /**
     * Login to MongoDB and return the MongoClient connection object.
     * @param mongoDbDriver
     * @param options
     */
    async mongoLogin(mongoDbDriver, options) {
        // Get the CA cert.
        const caData = await this.readCertificateData(options.CaCertificate);
        await filesystem_1.writeAsciiFile('/tmp/ca.crt', caData);
        // Next, the login credentials
        const credentials = await this.readLoginCredentials(options.Credentials);
        // Login to MongoDB
        const mongoUri = `mongodb://${options.Hostname}:${options.Port}`;
        console.log(`Connecting to: ${mongoUri}`);
        // Reference: http://mongodb.github.io/node-mongodb-native/3.5/api/MongoClient.html#.connect
        return await mongoDbDriver.MongoClient.connect(mongoUri, {
            tls: true,
            tlsInsecure: false,
            tlsCAFile: '/tmp/ca.crt',
            auth: {
                user: credentials.username,
                password: credentials.password,
            },
            useUnifiedTopology: true,
        });
    }
    /**
     * Retrieve CA certificate data from the Secret with the given ARN.
     * @param certificateArn
     */
    async readCertificateData(certificateArn) {
        return await secrets_manager_1.readCertificateData(certificateArn, this.secretsManagerClient);
    }
    /**
     * Use openssl to retrieve the subject, in RFC2253 format, of the given certificate.
     * @param certificateData
     */
    async retrieveRfc2253Subject(certificateData) {
        await filesystem_1.writeAsciiFile('/tmp/client.crt', certificateData);
        const subject = await exec('openssl x509 -in /tmp/client.crt -noout -subject -nameopt RFC2253');
        return subject.stdout.replace(/subject= /, '').trim();
    }
    /**
     * Retrieve the credentials of the user that we're to login to the DB with.
     * @param credentialsArn
     */
    async readLoginCredentials(credentialsArn) {
        const data = await secrets_manager_1.Secret.fromArn(credentialsArn, this.secretsManagerClient).getValue();
        if (Buffer.isBuffer(data) || !data) {
            throw new Error(`Login credentials, in Secret ${credentialsArn}, for MongoDB must be a JSON encoded string`);
        }
        let credentials;
        try {
            credentials = JSON.parse(data);
        }
        catch (e) {
            // Note: Intentionally not including the data as part of this error message. It may contain secrets, and including it will leak those secrets.
            throw new Error(`Failed to parse JSON in MongoDB login credentials Secret (${credentialsArn}). Please ensure that the Secret contains properly formatted JSON.`);
        }
        for (const key of ['username', 'password']) {
            if (!(key in credentials)) {
                throw new Error(`Login credentials Secret (${credentialsArn}) is missing: ${key}`);
            }
        }
        return credentials;
    }
    /**
     * Read, from the given Secret, the information for a password-authenticated user to be created
     * in the DB.
     * @param userArn
     */
    async readPasswordAuthUserInfo(userArn) {
        const data = await secrets_manager_1.Secret.fromArn(userArn, this.secretsManagerClient).getValue();
        if (Buffer.isBuffer(data) || !data) {
            throw new Error(`Password-auth user credentials, in Secret ${userArn}, for MongoDB must be a JSON encoded string`);
        }
        let userCreds;
        try {
            userCreds = JSON.parse(data);
        }
        catch (e) {
            // Note: Intentionally not including the data as part of this error message. It may contain secrets, and including it will leak those secrets.
            throw new Error(`Failed to parse JSON for password-auth user Secret (${userArn}). Please ensure that the Secret contains properly formatted JSON.`);
        }
        for (const key of ['username', 'password', 'roles']) {
            if (!(key in userCreds)) {
                // Note: Intentionally not including the data as part of this error message. It may contain secrets, and including it will leak those secrets.
                throw new Error(`User credentials Secret '${userArn}' is missing: ${key}`);
            }
        }
        return userCreds;
    }
    /**
     * Query the given DB to determine whether or not there is a user with the given username.
     * @param db
     * @param username
     */
    async userExists(db, username) {
        const result = await db.command({ usersInfo: username });
        if (result.ok !== 1) {
            throw new Error(`MongoDB error checking whether user exists \'${username}\' -- ${JSON.stringify(result)}`);
        }
        return result.users.length > 0;
    }
    /**
     * Add a user to the database. This must only be called if you know that the user does not
     * already exist.
     * @param db
     * @param credentials
     */
    async createUser(db, credentials) {
        console.log(`Creating user: ${credentials.username}`);
        const request = {
            createUser: credentials.username,
            roles: credentials.roles,
        };
        // It is an error to include a pwd field with undefined value, and our password
        // will be absent/undefined for x.509 authenticated users in the $external DB.
        if (credentials.password) {
            request.pwd = credentials.password;
        }
        const result = await db.command(request);
        if (result.ok !== 1) {
            throw new Error(`MongoDB error when adding user \'${credentials.username}\' -- ${JSON.stringify(result)}`);
        }
    }
    /**
     * Create a user in the admin DB if it does not already exist. If it does exist, then do nothing.
     * @param db
     * @param userArn
     */
    async createPasswordAuthUser(db, userArn) {
        const credentials = await this.readPasswordAuthUserInfo(userArn);
        if (await this.userExists(db, credentials.username)) {
            return false;
        }
        await this.createUser(db, credentials);
        return true;
    }
    /**
     * Create a user in the $external DB if it does not already exist. If it does exist, then do nothing.
     * @param db
     * @param user
     */
    async createX509AuthUser(db, user) {
        const userCertData = await this.readCertificateData(user.Certificate);
        const username = await this.retrieveRfc2253Subject(userCertData);
        if (await this.userExists(db, username)) {
            return false;
        }
        const credentials = {
            username,
            // Note: No need to check for parse-errors. It's already been vetted twice. Once by the typescript code, and once by the
            // input verifier of this handler.
            roles: JSON.parse(user.Roles),
        };
        await this.createUser(db, credentials);
        return true;
    }
}
exports.MongoDbConfigure = MongoDbConfigure;
/**
 * The lambda handler that is used to log in to MongoDB and perform some configuration actions.
 */
/* istanbul ignore next */
async function configureMongo(event, context) {
    const handler = new MongoDbConfigure(new aws_sdk_1.SecretsManager());
    return await handler.handler(event, context);
}
exports.configureMongo = configureMongo;
//# sourceMappingURL=data:application/json;base64,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