"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Certificate = void 0;
/* eslint-disable no-console */
const child_process = require("child_process");
const fs = require("fs");
const os = require("os");
const path = require("path");
const util_1 = require("util");
const filesystem_1 = require("../filesystem");
const exec = util_1.promisify(child_process.exec);
class Certificate {
    constructor(cert, key, passphrase, certChain) {
        this.cert = cert;
        this.key = key;
        this.passphrase = passphrase;
        this.certChain = certChain;
    }
    static async fromGenerated(subject, passphrase, certValidFor, signingCertificate) {
        const tmpDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'tmp.'));
        try {
            let cert;
            let key;
            let certChain;
            if (!signingCertificate) {
                [cert, key] = await Certificate.generateSelfSigned(tmpDir, subject, passphrase, certValidFor !== null && certValidFor !== void 0 ? certValidFor : 1095);
                // certChain cannot be left undefined. CFN expects that attributes will *always* have values.
                certChain = '';
            }
            else {
                [cert, key, certChain] = await Certificate.generateSigned(tmpDir, subject, passphrase, certValidFor !== null && certValidFor !== void 0 ? certValidFor : 1095, signingCertificate);
            }
            return new Certificate(cert, key, passphrase, certChain);
        }
        finally {
            const unlinks = [];
            const filenames = await fs.promises.readdir(tmpDir);
            for (const file of filenames) {
                unlinks.push(fs.promises.unlink(path.join(tmpDir, file)));
            }
            await Promise.all(unlinks);
            await fs.promises.rmdir(tmpDir);
        }
    }
    static async decryptKey(key, passphrase) {
        const tmpDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'tmp.'));
        try {
            const encrypedKeyFile = path.join(tmpDir, 'encrypted.key');
            await filesystem_1.writeAsciiFile(encrypedKeyFile, key);
            const decryptedKeyFile = path.join(tmpDir, 'decrypted.key');
            const cmd = 'openssl rsa ' +
                `-in ${encrypedKeyFile} ` +
                '-passin env:CERT_PASSPHRASE ' +
                `-out ${decryptedKeyFile}`;
            console.debug(`Running: ${cmd}`);
            await exec(cmd, { env: { CERT_PASSPHRASE: passphrase, PATH: process.env.PATH } });
            const keyDecrypted = await filesystem_1.readAsciiFile(decryptedKeyFile);
            return keyDecrypted;
        }
        finally {
            const unlinks = [];
            const filenames = await fs.promises.readdir(tmpDir);
            for (const file of filenames) {
                unlinks.push(fs.promises.unlink(path.join(tmpDir, file)));
            }
            await Promise.all(unlinks);
            await fs.promises.rmdir(tmpDir);
        }
    }
    static async generateSelfSigned(tmpDir, subject, passphrase, certValidFor) {
        const crtFile = path.join(tmpDir, 'crt');
        const keyFile = path.join(tmpDir, 'key');
        const cmd = 'openssl req -x509 ' +
            '-passout env:CERT_PASSPHRASE ' +
            '-newkey rsa:2048 ' +
            `-days ${certValidFor} ` +
            '-extensions v3_ca ' +
            `-keyout ${keyFile} -out ${crtFile} ` +
            `-subj ${subject.toString()}`;
        console.debug(`Running: ${cmd}`);
        await exec(cmd, { env: { CERT_PASSPHRASE: passphrase, PATH: process.env.PATH } });
        const cert = await filesystem_1.readAsciiFile(crtFile);
        const key = await filesystem_1.readAsciiFile(keyFile);
        return [cert, key];
    }
    static async generateSigned(tmpDir, subject, passphrase, certValidFor, signingCertificate) {
        const signingCertFile = path.join(tmpDir, 'signing.crt');
        const signingKeyFile = path.join(tmpDir, 'signing.key');
        const caChain = signingCertificate.cert + signingCertificate.certChain;
        await filesystem_1.writeAsciiFile(signingCertFile, caChain);
        await filesystem_1.writeAsciiFile(signingKeyFile, signingCertificate.key);
        const csrFile = path.join(tmpDir, 'cert.csr');
        const crtFile = path.join(tmpDir, 'cert.crt');
        const keyFile = path.join(tmpDir, 'cert.key');
        const certSigningRequest = 'openssl req ' +
            '-passout env:CERT_PASSPHRASE ' +
            '-newkey rsa:2048 ' +
            `-days ${certValidFor} ` +
            `-out ${csrFile} -keyout ${keyFile} ` +
            `-subj ${subject.toString()}`;
        const crtCreate = 'openssl x509 -sha256 -req ' +
            '-passin env:SIGNING_PASSPHRASE ' +
            `-days ${certValidFor} ` +
            `-in ${csrFile} ` +
            `-CA ${signingCertFile} -CAkey ${signingKeyFile} -CAcreateserial ` +
            `-out ${crtFile}`;
        console.debug(`Running: ${certSigningRequest}`);
        await exec(certSigningRequest, { env: { CERT_PASSPHRASE: passphrase, PATH: process.env.PATH } });
        console.debug(`Running: ${crtCreate}`);
        await exec(crtCreate, { env: { PATH: process.env.PATH, SIGNING_PASSPHRASE: signingCertificate.passphrase } });
        const cert = await filesystem_1.readAsciiFile(crtFile);
        const key = await filesystem_1.readAsciiFile(keyFile);
        // Return the certificate, private key, and certificate chain. The certificate chain is the signing certificate
        // prepended to its own certificate chain.
        return [cert, key, caChain];
    }
    async toPkcs12(passphrase) {
        const tmpDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'tmp.'));
        try {
            const crtFileName = path.join(tmpDir, 'input.crt');
            const keyFileName = path.join(tmpDir, 'input.key');
            const caCert = this.certChain ? this.cert + this.certChain : this.cert;
            await filesystem_1.writeAsciiFile(crtFileName, caCert);
            await filesystem_1.writeAsciiFile(keyFileName, this.key);
            const pkcs12FileName = path.join(tmpDir, 'cert.p12');
            const command = 'openssl pkcs12 -export -nodes -passin env:PASSIN -passout env:PASSOUT ' +
                `-out ${pkcs12FileName} -inkey ${keyFileName} -in ${crtFileName}`;
            await exec(command, { env: {
                    PASSIN: this.passphrase,
                    PASSOUT: passphrase,
                    PATH: process.env.PATH,
                } });
            const pkcs12Data = await filesystem_1.readBinaryFile(pkcs12FileName);
            return pkcs12Data;
        }
        finally {
            const unlinks = [];
            const filenames = await fs.promises.readdir(tmpDir);
            for (const file of filenames) {
                unlinks.push(fs.promises.unlink(path.join(tmpDir, file)));
            }
            await Promise.all(unlinks);
            await fs.promises.rmdir(tmpDir);
        }
    }
}
exports.Certificate = Certificate;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2VydGlmaWNhdGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjZXJ0aWZpY2F0ZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7OztHQUdHOzs7QUFFSCwrQkFBK0I7QUFFL0IsK0NBQStDO0FBQy9DLHlCQUF5QjtBQUN6Qix5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLCtCQUFpQztBQUVqQyw4Q0FJdUI7QUFHdkIsTUFBTSxJQUFJLEdBQUcsZ0JBQVMsQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7QUFTM0MsTUFBYSxXQUFXO0lBMkl0QixZQUNFLElBQVksRUFDWixHQUFXLEVBQ1gsVUFBa0IsRUFDbEIsU0FBaUI7UUFFakIsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUM7UUFDakIsSUFBSSxDQUFDLEdBQUcsR0FBRyxHQUFHLENBQUM7UUFDZixJQUFJLENBQUMsVUFBVSxHQUFHLFVBQVUsQ0FBQztRQUM3QixJQUFJLENBQUMsU0FBUyxHQUFHLFNBQVMsQ0FBQztJQUM3QixDQUFDO0lBcEpNLE1BQU0sQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUMvQixPQUEwQixFQUMxQixVQUFrQixFQUNsQixZQUFxQixFQUNyQixrQkFBZ0M7UUFFaEMsTUFBTSxNQUFNLEdBQUcsTUFBTSxFQUFFLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQ3pFLElBQUk7WUFDRixJQUFJLElBQVksQ0FBQztZQUNqQixJQUFJLEdBQVcsQ0FBQztZQUNoQixJQUFJLFNBQWlCLENBQUM7WUFDdEIsSUFBSSxDQUFDLGtCQUFrQixFQUFFO2dCQUN2QixDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsR0FBRyxNQUFNLFdBQVcsQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLEVBQUUsT0FBTyxFQUFFLFVBQVUsRUFBRSxZQUFZLGFBQVosWUFBWSxjQUFaLFlBQVksR0FBSSxJQUFJLENBQUMsQ0FBQztnQkFDdEcsNkZBQTZGO2dCQUM3RixTQUFTLEdBQUcsRUFBRSxDQUFDO2FBQ2hCO2lCQUFNO2dCQUNMLENBQUMsSUFBSSxFQUFFLEdBQUcsRUFBRSxTQUFTLENBQUMsR0FBRyxNQUFNLFdBQVcsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLE9BQU8sRUFBRSxVQUFVLEVBQUUsWUFBWSxhQUFaLFlBQVksY0FBWixZQUFZLEdBQUksSUFBSSxFQUFFLGtCQUFrQixDQUFDLENBQUM7YUFDbEk7WUFDRCxPQUFPLElBQUksV0FBVyxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsVUFBVSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1NBQzFEO2dCQUFTO1lBQ1IsTUFBTSxPQUFPLEdBQXlCLEVBQUUsQ0FBQztZQUN6QyxNQUFNLFNBQVMsR0FBYSxNQUFNLEVBQUUsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQzlELEtBQUssTUFBTSxJQUFJLElBQUksU0FBUyxFQUFFO2dCQUM1QixPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUMzRDtZQUNELE1BQU0sT0FBTyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUMzQixNQUFNLEVBQUUsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ2pDO0lBQ0gsQ0FBQztJQUVNLE1BQU0sQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUM1QixHQUFXLEVBQ1gsVUFBa0I7UUFFbEIsTUFBTSxNQUFNLEdBQUcsTUFBTSxFQUFFLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQ3pFLElBQUk7WUFDRixNQUFNLGVBQWUsR0FBVyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxlQUFlLENBQUMsQ0FBQztZQUNuRSxNQUFNLDJCQUFjLENBQUMsZUFBZSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1lBQzNDLE1BQU0sZ0JBQWdCLEdBQVcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDcEUsTUFBTSxHQUFHLEdBQ1AsY0FBYztnQkFDZCxPQUFPLGVBQWUsR0FBRztnQkFDekIsOEJBQThCO2dCQUM5QixRQUFRLGdCQUFnQixFQUFFLENBQUM7WUFFN0IsT0FBTyxDQUFDLEtBQUssQ0FBQyxZQUFZLEdBQUcsRUFBRSxDQUFDLENBQUM7WUFDakMsTUFBTSxJQUFJLENBQUMsR0FBRyxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsZUFBZSxFQUFFLFVBQVUsRUFBRSxJQUFJLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFFbEYsTUFBTSxZQUFZLEdBQUcsTUFBTSwwQkFBYSxDQUFDLGdCQUFnQixDQUFDLENBQUM7WUFFM0QsT0FBTyxZQUFZLENBQUM7U0FDckI7Z0JBQVM7WUFDUixNQUFNLE9BQU8sR0FBeUIsRUFBRSxDQUFDO1lBQ3pDLE1BQU0sU0FBUyxHQUFhLE1BQU0sRUFBRSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDOUQsS0FBSyxNQUFNLElBQUksSUFBSSxTQUFTLEVBQUU7Z0JBQzVCLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQzNEO1lBQ0QsTUFBTSxPQUFPLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQzNCLE1BQU0sRUFBRSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDakM7SUFDSCxDQUFDO0lBRU8sTUFBTSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FDckMsTUFBYyxFQUNkLE9BQTBCLEVBQzFCLFVBQWtCLEVBQ2xCLFlBQW9CO1FBRXBCLE1BQU0sT0FBTyxHQUFXLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ2pELE1BQU0sT0FBTyxHQUFXLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ2pELE1BQU0sR0FBRyxHQUNQLG9CQUFvQjtZQUNwQiwrQkFBK0I7WUFDL0IsbUJBQW1CO1lBQ25CLFNBQVMsWUFBWSxHQUFHO1lBQ3hCLG9CQUFvQjtZQUNwQixXQUFXLE9BQU8sU0FBUyxPQUFPLEdBQUc7WUFDckMsU0FBUyxPQUFPLENBQUMsUUFBUSxFQUFFLEVBQUUsQ0FBQztRQUVoQyxPQUFPLENBQUMsS0FBSyxDQUFDLFlBQVksR0FBRyxFQUFFLENBQUMsQ0FBQztRQUNqQyxNQUFNLElBQUksQ0FBQyxHQUFHLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxlQUFlLEVBQUUsVUFBVSxFQUFFLElBQUksRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVsRixNQUFNLElBQUksR0FBVyxNQUFNLDBCQUFhLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDbEQsTUFBTSxHQUFHLEdBQVcsTUFBTSwwQkFBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRWpELE9BQU8sQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUM7SUFDckIsQ0FBQztJQUVPLE1BQU0sQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUNqQyxNQUFjLEVBQ2QsT0FBMEIsRUFDMUIsVUFBa0IsRUFDbEIsWUFBb0IsRUFDcEIsa0JBQStCO1FBRS9CLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQ3pELE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQ3hELE1BQU0sT0FBTyxHQUFHLGtCQUFrQixDQUFDLElBQUksR0FBRyxrQkFBa0IsQ0FBQyxTQUFTLENBQUM7UUFDdkUsTUFBTSwyQkFBYyxDQUFDLGVBQWUsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUMvQyxNQUFNLDJCQUFjLENBQUMsY0FBYyxFQUFFLGtCQUFrQixDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRTdELE1BQU0sT0FBTyxHQUFXLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQ3RELE1BQU0sT0FBTyxHQUFXLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQ3RELE1BQU0sT0FBTyxHQUFXLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBRXRELE1BQU0sa0JBQWtCLEdBQ2hCLGNBQWM7WUFDZCwrQkFBK0I7WUFDL0IsbUJBQW1CO1lBQ25CLFNBQVMsWUFBWSxHQUFHO1lBQ3hCLFFBQVEsT0FBTyxZQUFZLE9BQU8sR0FBRztZQUNyQyxTQUFTLE9BQU8sQ0FBQyxRQUFRLEVBQUUsRUFBRSxDQUFDO1FBQ3RDLE1BQU0sU0FBUyxHQUNQLDRCQUE0QjtZQUM1QixpQ0FBaUM7WUFDakMsU0FBUyxZQUFZLEdBQUc7WUFDeEIsT0FBTyxPQUFPLEdBQUc7WUFDakIsT0FBTyxlQUFlLFdBQVcsY0FBYyxtQkFBbUI7WUFDbEUsUUFBUSxPQUFPLEVBQUUsQ0FBQztRQUUxQixPQUFPLENBQUMsS0FBSyxDQUFDLFlBQVksa0JBQWtCLEVBQUUsQ0FBQyxDQUFDO1FBQ2hELE1BQU0sSUFBSSxDQUFDLGtCQUFrQixFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsZUFBZSxFQUFFLFVBQVUsRUFBRSxJQUFJLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsRUFBQyxDQUFDLENBQUM7UUFDaEcsT0FBTyxDQUFDLEtBQUssQ0FBQyxZQUFZLFNBQVMsRUFBRSxDQUFDLENBQUM7UUFDdkMsTUFBTSxJQUFJLENBQUMsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsSUFBSSxFQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFLGtCQUFrQixDQUFDLFVBQVUsRUFBRSxFQUFDLENBQUMsQ0FBQztRQUU3RyxNQUFNLElBQUksR0FBVyxNQUFNLDBCQUFhLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDbEQsTUFBTSxHQUFHLEdBQVcsTUFBTSwwQkFBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRWpELCtHQUErRztRQUMvRywwQ0FBMEM7UUFDMUMsT0FBTyxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQW1CTSxLQUFLLENBQUMsUUFBUSxDQUFDLFVBQWtCO1FBQ3RDLE1BQU0sTUFBTSxHQUFHLE1BQU0sRUFBRSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUFFLEVBQUUsTUFBTSxDQUFDLENBQUMsQ0FBQztRQUN6RSxJQUFJO1lBQ0YsTUFBTSxXQUFXLEdBQVcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsV0FBVyxDQUFDLENBQUM7WUFDM0QsTUFBTSxXQUFXLEdBQVcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsV0FBVyxDQUFDLENBQUM7WUFDM0QsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO1lBQ3ZFLE1BQU0sMkJBQWMsQ0FBQyxXQUFXLEVBQUUsTUFBTSxDQUFDLENBQUM7WUFDMUMsTUFBTSwyQkFBYyxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7WUFFNUMsTUFBTSxjQUFjLEdBQVcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFDN0QsTUFBTSxPQUFPLEdBQVcsd0VBQXdFO2dCQUM5RixRQUFRLGNBQWMsV0FBVyxXQUFXLFFBQVEsV0FBVyxFQUFFLENBQUM7WUFDcEUsTUFBTSxJQUFJLENBQ1IsT0FBTyxFQUNQLEVBQUUsR0FBRyxFQUFFO29CQUNMLE1BQU0sRUFBRSxJQUFJLENBQUMsVUFBVTtvQkFDdkIsT0FBTyxFQUFFLFVBQVU7b0JBQ25CLElBQUksRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUk7aUJBQ3ZCLEVBQUMsQ0FDSCxDQUFDO1lBRUYsTUFBTSxVQUFVLEdBQVcsTUFBTSwyQkFBYyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBRWhFLE9BQU8sVUFBVSxDQUFDO1NBQ25CO2dCQUFTO1lBQ1IsTUFBTSxPQUFPLEdBQXlCLEVBQUUsQ0FBQztZQUN6QyxNQUFNLFNBQVMsR0FBYSxNQUFNLEVBQUUsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQzlELEtBQUssTUFBTSxJQUFJLElBQUksU0FBUyxFQUFFO2dCQUM1QixPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUMzRDtZQUNELE1BQU0sT0FBTyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUMzQixNQUFNLEVBQUUsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ2pDO0lBQ0gsQ0FBQztDQUNGO0FBekxELGtDQXlMQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbi8qIGVzbGludC1kaXNhYmxlIG5vLWNvbnNvbGUgKi9cblxuaW1wb3J0ICogYXMgY2hpbGRfcHJvY2VzcyBmcm9tICdjaGlsZF9wcm9jZXNzJztcbmltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIG9zIGZyb20gJ29zJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBwcm9taXNpZnkgfSBmcm9tICd1dGlsJztcblxuaW1wb3J0IHtcbiAgcmVhZEFzY2lpRmlsZSxcbiAgcmVhZEJpbmFyeUZpbGUsXG4gIHdyaXRlQXNjaWlGaWxlLFxufSBmcm9tICcuLi9maWxlc3lzdGVtJztcbmltcG9ydCB7IERpc3Rpbmd1aXNoZWROYW1lIH0gZnJvbSAnLi9kaXN0aW5ndWlzaGVkLW5hbWUnO1xuXG5jb25zdCBleGVjID0gcHJvbWlzaWZ5KGNoaWxkX3Byb2Nlc3MuZXhlYyk7XG5cbmV4cG9ydCBpbnRlcmZhY2UgSUNlcnRpZmljYXRlIHtcbiAgcmVhZG9ubHkgY2VydDogc3RyaW5nO1xuICByZWFkb25seSBrZXk6IHN0cmluZztcbiAgcmVhZG9ubHkgcGFzc3BocmFzZTogc3RyaW5nO1xuICByZWFkb25seSBjZXJ0Q2hhaW46IHN0cmluZztcbn1cblxuZXhwb3J0IGNsYXNzIENlcnRpZmljYXRlIGltcGxlbWVudHMgSUNlcnRpZmljYXRlIHtcbiAgcHVibGljIHN0YXRpYyBhc3luYyBmcm9tR2VuZXJhdGVkKFxuICAgIHN1YmplY3Q6IERpc3Rpbmd1aXNoZWROYW1lLFxuICAgIHBhc3NwaHJhc2U6IHN0cmluZyxcbiAgICBjZXJ0VmFsaWRGb3I/OiBudW1iZXIsXG4gICAgc2lnbmluZ0NlcnRpZmljYXRlPzogQ2VydGlmaWNhdGUsXG4gICk6IFByb21pc2U8Q2VydGlmaWNhdGU+IHtcbiAgICBjb25zdCB0bXBEaXIgPSBhd2FpdCBmcy5wcm9taXNlcy5ta2R0ZW1wKHBhdGguam9pbihvcy50bXBkaXIoKSwgJ3RtcC4nKSk7XG4gICAgdHJ5IHtcbiAgICAgIGxldCBjZXJ0OiBzdHJpbmc7XG4gICAgICBsZXQga2V5OiBzdHJpbmc7XG4gICAgICBsZXQgY2VydENoYWluOiBzdHJpbmc7XG4gICAgICBpZiAoIXNpZ25pbmdDZXJ0aWZpY2F0ZSkge1xuICAgICAgICBbY2VydCwga2V5XSA9IGF3YWl0IENlcnRpZmljYXRlLmdlbmVyYXRlU2VsZlNpZ25lZCh0bXBEaXIsIHN1YmplY3QsIHBhc3NwaHJhc2UsIGNlcnRWYWxpZEZvciA/PyAxMDk1KTtcbiAgICAgICAgLy8gY2VydENoYWluIGNhbm5vdCBiZSBsZWZ0IHVuZGVmaW5lZC4gQ0ZOIGV4cGVjdHMgdGhhdCBhdHRyaWJ1dGVzIHdpbGwgKmFsd2F5cyogaGF2ZSB2YWx1ZXMuXG4gICAgICAgIGNlcnRDaGFpbiA9ICcnO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgW2NlcnQsIGtleSwgY2VydENoYWluXSA9IGF3YWl0IENlcnRpZmljYXRlLmdlbmVyYXRlU2lnbmVkKHRtcERpciwgc3ViamVjdCwgcGFzc3BocmFzZSwgY2VydFZhbGlkRm9yID8/IDEwOTUsIHNpZ25pbmdDZXJ0aWZpY2F0ZSk7XG4gICAgICB9XG4gICAgICByZXR1cm4gbmV3IENlcnRpZmljYXRlKGNlcnQsIGtleSwgcGFzc3BocmFzZSwgY2VydENoYWluKTtcbiAgICB9IGZpbmFsbHkge1xuICAgICAgY29uc3QgdW5saW5rczogQXJyYXk8UHJvbWlzZTx2b2lkPj4gPSBbXTtcbiAgICAgIGNvbnN0IGZpbGVuYW1lczogc3RyaW5nW10gPSBhd2FpdCBmcy5wcm9taXNlcy5yZWFkZGlyKHRtcERpcik7XG4gICAgICBmb3IgKGNvbnN0IGZpbGUgb2YgZmlsZW5hbWVzKSB7XG4gICAgICAgIHVubGlua3MucHVzaChmcy5wcm9taXNlcy51bmxpbmsocGF0aC5qb2luKHRtcERpciwgZmlsZSkpKTtcbiAgICAgIH1cbiAgICAgIGF3YWl0IFByb21pc2UuYWxsKHVubGlua3MpO1xuICAgICAgYXdhaXQgZnMucHJvbWlzZXMucm1kaXIodG1wRGlyKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGFzeW5jIGRlY3J5cHRLZXkoXG4gICAga2V5OiBzdHJpbmcsXG4gICAgcGFzc3BocmFzZTogc3RyaW5nLFxuICApOiBQcm9taXNlPHN0cmluZz4ge1xuICAgIGNvbnN0IHRtcERpciA9IGF3YWl0IGZzLnByb21pc2VzLm1rZHRlbXAocGF0aC5qb2luKG9zLnRtcGRpcigpLCAndG1wLicpKTtcbiAgICB0cnkge1xuICAgICAgY29uc3QgZW5jcnlwZWRLZXlGaWxlOiBzdHJpbmcgPSBwYXRoLmpvaW4odG1wRGlyLCAnZW5jcnlwdGVkLmtleScpO1xuICAgICAgYXdhaXQgd3JpdGVBc2NpaUZpbGUoZW5jcnlwZWRLZXlGaWxlLCBrZXkpO1xuICAgICAgY29uc3QgZGVjcnlwdGVkS2V5RmlsZTogc3RyaW5nID0gcGF0aC5qb2luKHRtcERpciwgJ2RlY3J5cHRlZC5rZXknKTtcbiAgICAgIGNvbnN0IGNtZCA9XG4gICAgICAgICdvcGVuc3NsIHJzYSAnICtcbiAgICAgICAgYC1pbiAke2VuY3J5cGVkS2V5RmlsZX0gYCArXG4gICAgICAgICctcGFzc2luIGVudjpDRVJUX1BBU1NQSFJBU0UgJyArXG4gICAgICAgIGAtb3V0ICR7ZGVjcnlwdGVkS2V5RmlsZX1gO1xuXG4gICAgICBjb25zb2xlLmRlYnVnKGBSdW5uaW5nOiAke2NtZH1gKTtcbiAgICAgIGF3YWl0IGV4ZWMoY21kLCB7IGVudjogeyBDRVJUX1BBU1NQSFJBU0U6IHBhc3NwaHJhc2UsIFBBVEg6IHByb2Nlc3MuZW52LlBBVEggfSB9KTtcblxuICAgICAgY29uc3Qga2V5RGVjcnlwdGVkID0gYXdhaXQgcmVhZEFzY2lpRmlsZShkZWNyeXB0ZWRLZXlGaWxlKTtcblxuICAgICAgcmV0dXJuIGtleURlY3J5cHRlZDtcbiAgICB9IGZpbmFsbHkge1xuICAgICAgY29uc3QgdW5saW5rczogQXJyYXk8UHJvbWlzZTx2b2lkPj4gPSBbXTtcbiAgICAgIGNvbnN0IGZpbGVuYW1lczogc3RyaW5nW10gPSBhd2FpdCBmcy5wcm9taXNlcy5yZWFkZGlyKHRtcERpcik7XG4gICAgICBmb3IgKGNvbnN0IGZpbGUgb2YgZmlsZW5hbWVzKSB7XG4gICAgICAgIHVubGlua3MucHVzaChmcy5wcm9taXNlcy51bmxpbmsocGF0aC5qb2luKHRtcERpciwgZmlsZSkpKTtcbiAgICAgIH1cbiAgICAgIGF3YWl0IFByb21pc2UuYWxsKHVubGlua3MpO1xuICAgICAgYXdhaXQgZnMucHJvbWlzZXMucm1kaXIodG1wRGlyKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBhc3luYyBnZW5lcmF0ZVNlbGZTaWduZWQoXG4gICAgdG1wRGlyOiBzdHJpbmcsXG4gICAgc3ViamVjdDogRGlzdGluZ3Vpc2hlZE5hbWUsXG4gICAgcGFzc3BocmFzZTogc3RyaW5nLFxuICAgIGNlcnRWYWxpZEZvcjogbnVtYmVyLFxuICApOiBQcm9taXNlPFtzdHJpbmcsIHN0cmluZ10+IHtcbiAgICBjb25zdCBjcnRGaWxlOiBzdHJpbmcgPSBwYXRoLmpvaW4odG1wRGlyLCAnY3J0Jyk7XG4gICAgY29uc3Qga2V5RmlsZTogc3RyaW5nID0gcGF0aC5qb2luKHRtcERpciwgJ2tleScpO1xuICAgIGNvbnN0IGNtZDogc3RyaW5nID1cbiAgICAgICdvcGVuc3NsIHJlcSAteDUwOSAnICtcbiAgICAgICctcGFzc291dCBlbnY6Q0VSVF9QQVNTUEhSQVNFICcgK1xuICAgICAgJy1uZXdrZXkgcnNhOjIwNDggJyArXG4gICAgICBgLWRheXMgJHtjZXJ0VmFsaWRGb3J9IGAgK1xuICAgICAgJy1leHRlbnNpb25zIHYzX2NhICcgK1xuICAgICAgYC1rZXlvdXQgJHtrZXlGaWxlfSAtb3V0ICR7Y3J0RmlsZX0gYCArXG4gICAgICBgLXN1YmogJHtzdWJqZWN0LnRvU3RyaW5nKCl9YDtcblxuICAgIGNvbnNvbGUuZGVidWcoYFJ1bm5pbmc6ICR7Y21kfWApO1xuICAgIGF3YWl0IGV4ZWMoY21kLCB7IGVudjogeyBDRVJUX1BBU1NQSFJBU0U6IHBhc3NwaHJhc2UsIFBBVEg6IHByb2Nlc3MuZW52LlBBVEggfSB9KTtcblxuICAgIGNvbnN0IGNlcnQ6IHN0cmluZyA9IGF3YWl0IHJlYWRBc2NpaUZpbGUoY3J0RmlsZSk7XG4gICAgY29uc3Qga2V5OiBzdHJpbmcgPSBhd2FpdCByZWFkQXNjaWlGaWxlKGtleUZpbGUpO1xuXG4gICAgcmV0dXJuIFtjZXJ0LCBrZXldO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGF0aWMgYXN5bmMgZ2VuZXJhdGVTaWduZWQoXG4gICAgdG1wRGlyOiBzdHJpbmcsXG4gICAgc3ViamVjdDogRGlzdGluZ3Vpc2hlZE5hbWUsXG4gICAgcGFzc3BocmFzZTogc3RyaW5nLFxuICAgIGNlcnRWYWxpZEZvcjogbnVtYmVyLFxuICAgIHNpZ25pbmdDZXJ0aWZpY2F0ZTogQ2VydGlmaWNhdGUsXG4gICk6IFByb21pc2U8W3N0cmluZywgc3RyaW5nLCBzdHJpbmddPiB7XG4gICAgY29uc3Qgc2lnbmluZ0NlcnRGaWxlID0gcGF0aC5qb2luKHRtcERpciwgJ3NpZ25pbmcuY3J0Jyk7XG4gICAgY29uc3Qgc2lnbmluZ0tleUZpbGUgPSBwYXRoLmpvaW4odG1wRGlyLCAnc2lnbmluZy5rZXknKTtcbiAgICBjb25zdCBjYUNoYWluID0gc2lnbmluZ0NlcnRpZmljYXRlLmNlcnQgKyBzaWduaW5nQ2VydGlmaWNhdGUuY2VydENoYWluO1xuICAgIGF3YWl0IHdyaXRlQXNjaWlGaWxlKHNpZ25pbmdDZXJ0RmlsZSwgY2FDaGFpbik7XG4gICAgYXdhaXQgd3JpdGVBc2NpaUZpbGUoc2lnbmluZ0tleUZpbGUsIHNpZ25pbmdDZXJ0aWZpY2F0ZS5rZXkpO1xuXG4gICAgY29uc3QgY3NyRmlsZTogc3RyaW5nID0gcGF0aC5qb2luKHRtcERpciwgJ2NlcnQuY3NyJyk7XG4gICAgY29uc3QgY3J0RmlsZTogc3RyaW5nID0gcGF0aC5qb2luKHRtcERpciwgJ2NlcnQuY3J0Jyk7XG4gICAgY29uc3Qga2V5RmlsZTogc3RyaW5nID0gcGF0aC5qb2luKHRtcERpciwgJ2NlcnQua2V5Jyk7XG5cbiAgICBjb25zdCBjZXJ0U2lnbmluZ1JlcXVlc3QgPVxuICAgICAgICAgICAgJ29wZW5zc2wgcmVxICcgK1xuICAgICAgICAgICAgJy1wYXNzb3V0IGVudjpDRVJUX1BBU1NQSFJBU0UgJyArXG4gICAgICAgICAgICAnLW5ld2tleSByc2E6MjA0OCAnICtcbiAgICAgICAgICAgIGAtZGF5cyAke2NlcnRWYWxpZEZvcn0gYCArXG4gICAgICAgICAgICBgLW91dCAke2NzckZpbGV9IC1rZXlvdXQgJHtrZXlGaWxlfSBgICtcbiAgICAgICAgICAgIGAtc3ViaiAke3N1YmplY3QudG9TdHJpbmcoKX1gO1xuICAgIGNvbnN0IGNydENyZWF0ZSA9XG4gICAgICAgICAgICAnb3BlbnNzbCB4NTA5IC1zaGEyNTYgLXJlcSAnICtcbiAgICAgICAgICAgICctcGFzc2luIGVudjpTSUdOSU5HX1BBU1NQSFJBU0UgJyArXG4gICAgICAgICAgICBgLWRheXMgJHtjZXJ0VmFsaWRGb3J9IGAgK1xuICAgICAgICAgICAgYC1pbiAke2NzckZpbGV9IGAgK1xuICAgICAgICAgICAgYC1DQSAke3NpZ25pbmdDZXJ0RmlsZX0gLUNBa2V5ICR7c2lnbmluZ0tleUZpbGV9IC1DQWNyZWF0ZXNlcmlhbCBgICtcbiAgICAgICAgICAgIGAtb3V0ICR7Y3J0RmlsZX1gO1xuXG4gICAgY29uc29sZS5kZWJ1ZyhgUnVubmluZzogJHtjZXJ0U2lnbmluZ1JlcXVlc3R9YCk7XG4gICAgYXdhaXQgZXhlYyhjZXJ0U2lnbmluZ1JlcXVlc3QsIHsgZW52OiB7IENFUlRfUEFTU1BIUkFTRTogcGFzc3BocmFzZSwgUEFUSDogcHJvY2Vzcy5lbnYuUEFUSCB9fSk7XG4gICAgY29uc29sZS5kZWJ1ZyhgUnVubmluZzogJHtjcnRDcmVhdGV9YCk7XG4gICAgYXdhaXQgZXhlYyhjcnRDcmVhdGUsIHsgZW52OiB7IFBBVEg6IHByb2Nlc3MuZW52LlBBVEgsIFNJR05JTkdfUEFTU1BIUkFTRTogc2lnbmluZ0NlcnRpZmljYXRlLnBhc3NwaHJhc2UgfX0pO1xuXG4gICAgY29uc3QgY2VydDogc3RyaW5nID0gYXdhaXQgcmVhZEFzY2lpRmlsZShjcnRGaWxlKTtcbiAgICBjb25zdCBrZXk6IHN0cmluZyA9IGF3YWl0IHJlYWRBc2NpaUZpbGUoa2V5RmlsZSk7XG5cbiAgICAvLyBSZXR1cm4gdGhlIGNlcnRpZmljYXRlLCBwcml2YXRlIGtleSwgYW5kIGNlcnRpZmljYXRlIGNoYWluLiBUaGUgY2VydGlmaWNhdGUgY2hhaW4gaXMgdGhlIHNpZ25pbmcgY2VydGlmaWNhdGVcbiAgICAvLyBwcmVwZW5kZWQgdG8gaXRzIG93biBjZXJ0aWZpY2F0ZSBjaGFpbi5cbiAgICByZXR1cm4gW2NlcnQsIGtleSwgY2FDaGFpbl07XG4gIH1cblxuICBwdWJsaWMgcmVhZG9ubHkgY2VydDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkga2V5OiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBwYXNzcGhyYXNlOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBjZXJ0Q2hhaW46IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihcbiAgICBjZXJ0OiBzdHJpbmcsXG4gICAga2V5OiBzdHJpbmcsXG4gICAgcGFzc3BocmFzZTogc3RyaW5nLFxuICAgIGNlcnRDaGFpbjogc3RyaW5nLFxuICApIHtcbiAgICB0aGlzLmNlcnQgPSBjZXJ0O1xuICAgIHRoaXMua2V5ID0ga2V5O1xuICAgIHRoaXMucGFzc3BocmFzZSA9IHBhc3NwaHJhc2U7XG4gICAgdGhpcy5jZXJ0Q2hhaW4gPSBjZXJ0Q2hhaW47XG4gIH1cblxuICBwdWJsaWMgYXN5bmMgdG9Qa2NzMTIocGFzc3BocmFzZTogc3RyaW5nKTogUHJvbWlzZTxCdWZmZXI+IHtcbiAgICBjb25zdCB0bXBEaXIgPSBhd2FpdCBmcy5wcm9taXNlcy5ta2R0ZW1wKHBhdGguam9pbihvcy50bXBkaXIoKSwgJ3RtcC4nKSk7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IGNydEZpbGVOYW1lOiBzdHJpbmcgPSBwYXRoLmpvaW4odG1wRGlyLCAnaW5wdXQuY3J0Jyk7XG4gICAgICBjb25zdCBrZXlGaWxlTmFtZTogc3RyaW5nID0gcGF0aC5qb2luKHRtcERpciwgJ2lucHV0LmtleScpO1xuICAgICAgY29uc3QgY2FDZXJ0ID0gdGhpcy5jZXJ0Q2hhaW4gPyB0aGlzLmNlcnQgKyB0aGlzLmNlcnRDaGFpbiA6IHRoaXMuY2VydDtcbiAgICAgIGF3YWl0IHdyaXRlQXNjaWlGaWxlKGNydEZpbGVOYW1lLCBjYUNlcnQpO1xuICAgICAgYXdhaXQgd3JpdGVBc2NpaUZpbGUoa2V5RmlsZU5hbWUsIHRoaXMua2V5KTtcblxuICAgICAgY29uc3QgcGtjczEyRmlsZU5hbWU6IHN0cmluZyA9IHBhdGguam9pbih0bXBEaXIsICdjZXJ0LnAxMicpO1xuICAgICAgY29uc3QgY29tbWFuZDogc3RyaW5nID0gJ29wZW5zc2wgcGtjczEyIC1leHBvcnQgLW5vZGVzIC1wYXNzaW4gZW52OlBBU1NJTiAtcGFzc291dCBlbnY6UEFTU09VVCAnICtcbiAgICAgICAgYC1vdXQgJHtwa2NzMTJGaWxlTmFtZX0gLWlua2V5ICR7a2V5RmlsZU5hbWV9IC1pbiAke2NydEZpbGVOYW1lfWA7XG4gICAgICBhd2FpdCBleGVjKFxuICAgICAgICBjb21tYW5kLFxuICAgICAgICB7IGVudjoge1xuICAgICAgICAgIFBBU1NJTjogdGhpcy5wYXNzcGhyYXNlLFxuICAgICAgICAgIFBBU1NPVVQ6IHBhc3NwaHJhc2UsXG4gICAgICAgICAgUEFUSDogcHJvY2Vzcy5lbnYuUEFUSCxcbiAgICAgICAgfX0sXG4gICAgICApO1xuXG4gICAgICBjb25zdCBwa2NzMTJEYXRhOiBCdWZmZXIgPSBhd2FpdCByZWFkQmluYXJ5RmlsZShwa2NzMTJGaWxlTmFtZSk7XG5cbiAgICAgIHJldHVybiBwa2NzMTJEYXRhO1xuICAgIH0gZmluYWxseSB7XG4gICAgICBjb25zdCB1bmxpbmtzOiBBcnJheTxQcm9taXNlPHZvaWQ+PiA9IFtdO1xuICAgICAgY29uc3QgZmlsZW5hbWVzOiBzdHJpbmdbXSA9IGF3YWl0IGZzLnByb21pc2VzLnJlYWRkaXIodG1wRGlyKTtcbiAgICAgIGZvciAoY29uc3QgZmlsZSBvZiBmaWxlbmFtZXMpIHtcbiAgICAgICAgdW5saW5rcy5wdXNoKGZzLnByb21pc2VzLnVubGluayhwYXRoLmpvaW4odG1wRGlyLCBmaWxlKSkpO1xuICAgICAgfVxuICAgICAgYXdhaXQgUHJvbWlzZS5hbGwodW5saW5rcyk7XG4gICAgICBhd2FpdCBmcy5wcm9taXNlcy5ybWRpcih0bXBEaXIpO1xuICAgIH1cbiAgfVxufVxuIl19