"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk");
const aws_sdk_mock_1 = require("aws-sdk-mock");
const read_certificate_1 = require("../read-certificate");
const secretArn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert';
// @ts-ignore
async function successRequestMock(request, returnValue) {
    return returnValue;
}
describe('readCertificateData', () => {
    beforeEach(() => {
        aws_sdk_mock_1.setSDKInstance(AWS);
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('SecretsManager');
    });
    test('success', async () => {
        // GIVEN
        const certData = 'BEGIN CERTIFICATE';
        const secretContents = {
            SecretString: certData,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const client = new AWS.SecretsManager();
        // WHEN
        const data = await read_certificate_1.readCertificateData(secretArn, client);
        // THEN
        expect(data).toStrictEqual(certData);
    });
    test('not a certificate', async () => {
        // GIVEN
        const certData = 'NOT A CERTIFICATE';
        const secretContents = {
            SecretString: certData,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const client = new AWS.SecretsManager();
        // WHEN
        const promise = read_certificate_1.readCertificateData(secretArn, client);
        // THEN
        await expect(promise).rejects.toThrowError(/must contain a Certificate in PEM format/);
    });
    test('binary data', async () => {
        // GIVEN
        const certData = Buffer.from('BEGIN CERTIFICATE', 'utf-8');
        const secretContents = {
            SecretBinary: certData,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const client = new AWS.SecretsManager();
        // WHEN
        const promise = read_certificate_1.readCertificateData(secretArn, client);
        // THEN
        await expect(promise).rejects.toThrowError(/must contain a Certificate in PEM format/);
    });
});
//# sourceMappingURL=data:application/json;base64,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