"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('ThinkboxDockerRecipes', () => {
    let app;
    let stack;
    let stage;
    // GIVEN
    const STAGE_PATH = path.join(__dirname, 'assets');
    const RCS_RECIPE_NAME = 'rcs';
    const RCS_RECIPE = {
        description: 'rcs',
        title: 'rcs',
        buildArgs: {
            a: 'a',
            b: 'b',
        },
        target: undefined,
    };
    const LICENSE_FORWARDER_RECIPE_NAME = 'license-forwarder';
    const LICENSE_FORWARDER_RECIPE = {
        title: 'license-forwarder',
        description: 'license-forwarder',
        buildArgs: {
            c: 'c',
            d: 'd',
        },
        target: 'lf',
    };
    const MAJOR_VERSION = 10;
    const MINOR_VERSION = 1;
    const RELEASE_VERSION = 9;
    const PATCH_VERSION = 2;
    const RELEASE_VERSION_STRING = `${MAJOR_VERSION}.${MINOR_VERSION}.${RELEASE_VERSION}`;
    const FULL_VERSION_STRING = `${RELEASE_VERSION_STRING}.${PATCH_VERSION}`;
    beforeEach(() => {
        app = new core_1.App();
        class TestStage extends lib_1.Stage {
            constructor(props) {
                super(props);
            }
        }
        stage = new TestStage({
            path: STAGE_PATH,
            manifest: {
                schema: 1,
                version: FULL_VERSION_STRING,
                recipes: {
                    [RCS_RECIPE_NAME]: RCS_RECIPE,
                    [LICENSE_FORWARDER_RECIPE_NAME]: LICENSE_FORWARDER_RECIPE,
                },
            },
        });
        stack = new core_1.Stack(app, 'Stack');
    });
    /**
     * Tests that the remote connection server and license forwarder Docker recipes
     * create the correct {@link DockerImageAsset} instances using the build arguments,
     * and target from the supplied manifest.
     *
     * This is done by comparing the `sourceHash` property of the expected vs actual
     * DockerImageAsset instances.
     */
    test.each([
        [
            'remoteConnnectionServer',
            () => {
                return new aws_ecr_assets_1.DockerImageAsset(stack, 'SomeID', {
                    directory: STAGE_PATH,
                    buildArgs: RCS_RECIPE.buildArgs,
                    target: RCS_RECIPE.target,
                });
            },
            (recipes) => {
                return recipes.remoteConnectionServer;
            },
        ],
        [
            'licenseForwarder',
            () => {
                return new aws_ecr_assets_1.DockerImageAsset(stack, 'SomeID', {
                    directory: STAGE_PATH,
                    buildArgs: LICENSE_FORWARDER_RECIPE.buildArgs,
                    target: LICENSE_FORWARDER_RECIPE.target,
                });
            },
            (recipes) => {
                return recipes.licenseForwarder;
            },
        ],
    ])('has proper %p container image', (_imageName, getExpected, getActual) => {
        // GIVEN
        const expectedImage = getExpected();
        // WHEN
        const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
            stage,
        });
        const actualImage = getActual(recipes);
        // THEN
        expect(actualImage.sourceHash).toEqual(expectedImage.sourceHash);
    });
    describe('.version', () => {
        test('creates a VersionQuery when referenced', () => {
            // GIVEN
            const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                stage,
            });
            // WHEN
            recipes.version;
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
                forceRun: assert_1.stringLike('*'),
                versionString: RELEASE_VERSION_STRING,
            }));
        });
        test('does not create a VersionQuery when not referenced', () => {
            // GIVEN
            new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                stage,
            });
            assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_DEADLINE_INSTALLERS'));
        });
    });
    test.each([
        ['rcs', {
                [LICENSE_FORWARDER_RECIPE_NAME]: LICENSE_FORWARDER_RECIPE,
            }],
        ['license-forwarder', {
                [RCS_RECIPE_NAME]: RCS_RECIPE,
            }],
    ])('manifest missing required recipe %s throws error', (_recipeName, recipes) => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        class StageWithPublicConstructor extends lib_1.Stage {
            constructor(props) {
                super(props);
            }
        }
        const manifest = {
            recipes,
            schema: 1,
            version: '1.2.3.4',
        };
        // WHEN
        expect(() => {
            new lib_1.ThinkboxDockerRecipes(isolatedStack, 'Recipes', {
                stage: new StageWithPublicConstructor({
                    manifest,
                    path: '/some/path',
                }),
            });
        })
            // THEN
            .toThrow();
    });
});
//# sourceMappingURL=data:application/json;base64,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