"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const sinon = require("sinon");
const lib_1 = require("../../core/lib");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_2 = require("../lib");
describe('DocumentDB', () => {
    let stack;
    let vpc;
    let database;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            backup: {
                retention: core_1.Duration.days(15),
            },
            engineVersion: '3.6.0',
        });
        if (!database.secret) {
            throw new Error('secret cannot be null');
        }
    });
    test('Grants access to Document DB Secret', () => {
        // GIVEN
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        connection.grantRead(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: { Ref: 'DbClusterSecretAttachment4201A1ED' },
                    }],
            },
        }));
    });
    test('addInstallerDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addInstallerDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        expect(userData).toContain('configure_database_installation_args(){\n');
        expect(userData).toContain('\nexport -f configure_database_installation_args');
        expect(userData).toContain('\nINSTALLER_DB_ARGS=(');
    });
    test('allow connection', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        // WHEN
        connection.allowConnectionsFrom(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: {
                'Fn::GetAtt': [
                    'DbCluster224236EF',
                    'Port',
                ],
            },
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'SGADB53937',
                    'GroupId',
                ],
            },
            ToPort: {
                'Fn::GetAtt': [
                    'DbCluster224236EF',
                    'Port',
                ],
            },
        }));
    });
    test('add child dependency', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: [
                'DbClusterInstance155835CE5',
                'InstanceInstanceRoleE9785DE5',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('add child dependency to attributes', () => {
        // GIVEN
        const docdb = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: docdb, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: [
                'InstanceInstanceRoleE9785DE5',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('add child dependency throws when cluster implementation changed', () => {
        // GIVEN
        const docdb = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: docdb, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        docdb.node.defaultChild = database; // Trick addChildDependency() into thinking this is a real construct.
        // THEN
        expect(() => {
            connection.addChildDependency(instance);
        }).toThrowError(/The internal implementation of the AWS CDK's DocumentDB cluster construct may have changed./);
    });
    test('asserts linux-only', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // THEN
        expect(() => {
            connection.addInstallerDBArgs(instance);
        }).toThrowError('Can only install Deadline from a Linux instance.');
        expect(() => {
            connection.addConnectionDBArgs(instance);
        }).toThrowError('Connecting to the Deadline Database is currently only supported for Linux.');
    });
    test('adds warning annotation when a security group cannot be added due to unsupported IDatabaseCluster implementation', () => {
        // GIVEN
        class FakeDatabaseCluster extends core_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                this.clusterIdentifier = '';
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new aws_docdb_1.Endpoint('address', 123);
                this.clusterReadEndpoint = new aws_docdb_1.Endpoint('readAddress', 123);
                this.instanceEndpoints = [];
                this.securityGroupId = '';
                this.connections = new aws_ec2_1.Connections();
                this.stack = core_1.Stack.of(scope);
                this.env = { account: this.stack.account, region: this.stack.region };
            }
            asSecretAttachmentTarget() {
                throw new Error('Method not implemented.');
            }
        }
        const fakeDatabase = new FakeDatabaseCluster(stack, 'FakeDatabase');
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', { vpc });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: fakeDatabase, login: database.secret });
        // WHEN
        connection.addSecurityGroup(securityGroup);
        // THEN
        expect(fakeDatabase.node.metadata).toEqual(expect.arrayContaining([
            expect.objectContaining({
                type: 'aws:cdk:warning',
                data: expect.stringMatching(new RegExp(`Failed to add the following security groups to ${fakeDatabase.node.id}: .*\\. ` +
                    'The \\"database\\" property passed to this class is not an instance of AWS CDK\'s DocumentDB cluster construct.')),
            }),
        ]));
    });
    // This test can be removed once the following CDK PR is merged:
    // https://github.com/aws/aws-cdk/pull/13290
    test('adds warning annotation when a security group cannot be added due to implementation changes in DatabaseCluster', () => {
        // GIVEN
        if (!database.node.tryRemoveChild('Resource')) {
            throw new Error('The internal implementation of AWS CDK\'s DocumentDB cluster construct has changed. The addSecurityGroup method needs to be updated.');
        }
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', { vpc });
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        connection.addSecurityGroup(securityGroup);
        // THEN
        expect(database.node.metadata).toEqual(expect.arrayContaining([
            expect.objectContaining({
                type: 'aws:cdk:warning',
                data: expect.stringMatching(new RegExp(`Failed to add the following security groups to ${database.node.id}: .*\\. ` +
                    'The internal implementation of AWS CDK\'s DocumentDB cluster construct has changed.')),
            }),
        ]));
    });
});
describe('DocumentDB Version Checks', () => {
    let stack;
    let vpc;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    });
    test('Compatible version', () => {
        // GIVEN
        const database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            backup: {
                retention: core_1.Duration.days(15),
            },
            engineVersion: '3.6.0',
        });
        // WHEN
        lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        expect(database.node.metadata.length).toBe(0);
    });
    test('When from attributes', () => {
        // GIVEN
        const sg = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        const secret = new aws_secretsmanager_1.Secret(stack, 'Secret');
        const database = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'DbCluster', {
            clusterEndpointAddress: '1.2.3.4',
            clusterIdentifier: 'foo',
            instanceEndpointAddresses: ['1.2.3.5'],
            instanceIdentifiers: ['i0'],
            port: 27001,
            readerEndpointAddress: '1.2.3.6',
            securityGroup: sg,
        });
        // WHEN
        lib_2.DatabaseConnection.forDocDB({ database, login: secret });
        // THEN
        expect(database.node.metadata.length).toBe(0);
    });
    test('No engineVersion given', () => {
        // GIVEN
        const database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            backup: {
                retention: core_1.Duration.days(15),
            },
        });
        // WHEN
        lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        expect(database.node.metadata).toEqual(expect.arrayContaining([
            expect.objectContaining({
                type: 'aws:cdk:error',
                data: 'engineVersion must be 3.6.0 to be compatible with Deadline',
            }),
        ]));
    });
    test('engineVersion not 3.6.0', () => {
        // GIVEN
        const database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
            vpc,
            vpcSubnets: {
                onePerAz: true,
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            backup: {
                retention: core_1.Duration.days(15),
            },
            engineVersion: '4.0.0',
        });
        // WHEN
        lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // THEN
        expect(database.node.metadata).toEqual(expect.arrayContaining([
            expect.objectContaining({
                type: 'aws:cdk:error',
                data: 'engineVersion must be 3.6.0 to be compatible with Deadline',
            }),
        ]));
    });
});
describe('MongoDB', () => {
    let stack;
    let vpc;
    let database;
    let clientCert;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        const hostname = 'mongo';
        const zoneName = 'deadline.internal';
        const dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        const caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        const serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
        clientCert = new lib_1.X509CertificatePem(stack, 'ClientCert', {
            subject: {
                cn: 'dbuser',
                ou: 'TestClient',
            },
            signingCertificate: caCert,
        });
        database = new lib_1.MongoDbInstance(stack, 'MongoDb', {
            vpc,
            mongoDb: {
                userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
                version: lib_1.MongoDbVersion.COMMUNITY_3_6,
                hostname,
                dnsZone,
                serverCertificate: serverCert,
            },
        });
    });
    test('Grants access to certificate Secrets', () => {
        // GIVEN
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // WHEN
        connection.grantRead(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            'Fn::GetAtt': [
                                'ClientCert',
                                'Cert',
                            ],
                        },
                    },
                    {
                        Effect: 'Allow',
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            Ref: 'ClientCertPassphrase8A71E1E1',
                        },
                    },
                ],
            },
        }));
    });
    test('addInstallerDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addInstallerDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        const token = '${Token[TOKEN.\\d+]}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('\'/tmp/' + token + token + '\' ' + token + ' /opt/Thinkbox/certs/mongo_client.pfx')));
        expect(userData).toContain('configure_database_installation_args(){\n');
        expect(userData).toContain('\nexport -f configure_database_installation_args');
        expect(userData).toContain('{ set +x; } 2>/dev/null');
        expect(userData).toContain('\nINSTALLER_DB_ARGS=( ["--dbssl"]=true ["--dbauth"]=true ["--dbsslauth"]=true');
        expect(userData).toContain('["--dbhost"]="mongo.deadline.internal"');
        expect(userData).toContain('["--dbport"]=27017');
        expect(userData).toContain('["--dbclientcert"]=');
        expect(userData).toContain('["--dbcertpass"]=$CERT_PASSWORD');
    });
    test('addConnectionDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addConnectionDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        const token = '${Token[TOKEN.\\d+]}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('\'/tmp/' + token + token + '\' ' + token + ' /opt/Thinkbox/certs/mongo_client.pfx')));
        expect(userData).toContain('configure_deadline_database(){\n');
        expect(userData).toContain('\nexport -f configure_deadline_database');
        expect(userData).toContain('{ set +x; } 2>/dev/null');
        expect(userData).toContain('\nexport DB_CERT_FILE=');
        expect(userData).toContain('\nexport DB_CERT_PASSWORD=');
    });
    test('defines required container environment variables', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // THEN
        expect(connection.containerEnvironment).toHaveProperty('DB_TLS_CLIENT_CERT_URI');
        expect(connection.containerEnvironment).toHaveProperty('DB_TLS_CLIENT_CERT_PASSWORD_URI');
    });
    test('allow connection', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        // WHEN
        connection.allowConnectionsFrom(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: 27017,
            ToPort: 27017,
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'SGADB53937',
                    'GroupId',
                ],
            },
            GroupId: {
                'Fn::GetAtt': [
                    'MongoDbServerAsgInstanceSecurityGroupCE623335',
                    'GroupId',
                ],
            },
        }));
    });
    test('add child dependency', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: [
                'InstanceInstanceRoleE9785DE5',
                'MongoDbServerAsgASG47B3D94E',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('asserts linux-only', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // WHEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // THEN
        expect(() => {
            connection.addInstallerDBArgs(instance);
        }).toThrowError('Can only install Deadline from a Linux instance.');
        expect(() => {
            connection.addConnectionDBArgs(instance);
        }).toThrowError('Connecting to the Deadline Database is currently only supported for Linux.');
    });
    test('adds security group', () => {
        // GIVEN
        const dbSpy = sinon.spy(database, 'addSecurityGroup');
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', {
            vpc,
        });
        // WHEN
        connection.addSecurityGroup(securityGroup);
        // THEN
        expect(dbSpy.calledOnce).toBeTruthy();
    });
});
//# sourceMappingURL=data:application/json;base64,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