"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpotEventPluginFleet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const spot_event_plugin_fleet_ref_1 = require("./spot-event-plugin-fleet-ref");
const worker_configuration_1 = require("./worker-configuration");
/**
 * This construct reperesents a fleet from the Spot Fleet Request created by the Spot Event Plugin.
 *
 * This fleet is intended to be used as input for the {@link @aws-rfdk/deadline#ConfigureSpotEventPlugin} construct.
 *
 * The construct itself doesn't create the Spot Fleet Request, but deploys all the resources
 * required for the Spot Fleet Request and generates the Spot Fleet Configuration setting:
 * a one to one mapping between a Deadline Group and Spot Fleet Request Configurations.
 *
 * Resources Deployed
 * ------------------------
 * - An Instance Role, corresponding IAM Policy and an Instance Profile.
 * - A Fleet Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Worker, Deadline Launcher, and instance-startup logs for the instances
 *    in the fleet.
 * - A security Group if security groups are not provided.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The data that is stored on your Worker's local EBS volume can include temporary working files from the applications
 *    that are rendering your jobs and tasks. That data can be sensitive or privileged, so we recommend that you encrypt
 *    the data volumes of these instances using either the provided option or by using an encrypted AMI as your source.
 * - The software on the AMI that is being used by this construct may pose a security risk. We recommend that you adopt a
 *    patching strategy to keep this software current with the latest security patches. Please see
 *    https://docs.aws.amazon.com/rfdk/latest/guide/patching-software.html for more information.
 *
 * @stability stable
 */
class SpotEventPluginFleet extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g;
        super(scope, id);
        this.validateProps(props);
        this.securityGroups = (_b = props.securityGroups) !== null && _b !== void 0 ? _b : [new aws_ec2_1.SecurityGroup(this, 'SpotFleetSecurityGroup', { vpc: props.vpc })];
        this.connections = new aws_ec2_1.Connections({ securityGroups: this.securityGroups });
        this.connections.allowToDefaultPort(props.renderQueue.endpoint);
        this.fleetInstanceRole = (_c = props.fleetInstanceRole) !== null && _c !== void 0 ? _c : new aws_iam_1.Role(this, 'SpotFleetInstanceRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
            ],
            description: `Spot Fleet instance role for ${id} in region ${core_1.Stack.of(scope).region}`,
        });
        this.instanceProfile = new aws_iam_1.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.fleetInstanceRole.roleName],
        });
        this.grantPrincipal = this.fleetInstanceRole;
        this.fleetRole = (_d = props.fleetRole) !== null && _d !== void 0 ? _d : new aws_iam_1.Role(this, 'SpotFleetRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
            ],
            description: `Spot Fleet role for ${id} in region ${core_1.Stack.of(scope).region}`,
        });
        this.blockDevices = props.blockDevices;
        this.subnets = props.vpc.selectSubnets((_e = props.vpcSubnets) !== null && _e !== void 0 ? _e : { subnetType: aws_ec2_1.SubnetType.PRIVATE });
        this.instanceTypes = props.instanceTypes;
        this.allocationStrategy = (_f = props.allocationStrategy) !== null && _f !== void 0 ? _f : spot_event_plugin_fleet_ref_1.SpotFleetAllocationStrategy.LOWEST_PRICE;
        this.maxCapacity = props.maxCapacity;
        this.validUntil = props.validUntil;
        this.keyName = props.keyName;
        this.deadlineGroups = props.deadlineGroups;
        const imageConfig = props.workerMachineImage.getImage(this);
        this.osType = imageConfig.osType;
        this.userData = (_g = props.userData) !== null && _g !== void 0 ? _g : imageConfig.userData;
        this.imageId = imageConfig.imageId;
        const workerConfig = new worker_configuration_1.WorkerInstanceConfiguration(this, id, {
            worker: this,
            cloudWatchLogSettings: {
                logGroupPrefix: SpotEventPluginFleet.DEFAULT_LOG_GROUP_PREFIX,
                ...props.logGroupProps,
            },
            renderQueue: props.renderQueue,
            workerSettings: {
                groups: props.deadlineGroups,
                pools: props.deadlinePools,
                region: props.deadlineRegion,
            },
            userDataProvider: props.userDataProvider,
        });
        this.remoteControlPorts = aws_ec2_1.Port.tcpRange(workerConfig.listenerPort, workerConfig.listenerPort + SpotEventPluginFleet.MAX_WORKERS_PER_HOST);
        this.tags = new core_1.TagManager(core_1.TagType.KEY_VALUE, 'RFDK::SpotEventPluginFleet');
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Allow access to the Worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that depends on this stack.
     *
     * If this stack depends on the other stack, use allowRemoteControlTo().
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/remote-control.html
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowRemoteControlFrom(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowRemoteControlFrom(Peer.ipv4('10.0.0.0/24'))`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowRemoteControlFrom(other) {
        this.connections.allowFrom(other.connections, this.remoteControlPorts, 'Worker remote command listening port');
    }
    /**
     * Allow access to the Worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that this stack depends on.
     *
     * If the other stack depends on this stack, use allowRemoteControlFrom().
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/remote-control.html
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowRemoteControlTo(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowRemoteControlTo(Peer.ipv4('10.0.0.0/24'))`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowRemoteControlTo(other) {
        other.connections.allowTo(this.connections, this.remoteControlPorts, 'Worker remote command listening port');
    }
    validateProps(props) {
        this.validateFleetInstanceRole(props.fleetInstanceRole);
        this.validateInstanceTypes(props.instanceTypes);
        this.validateSubnets(props.vpc, props.vpcSubnets);
        this.validateGroups('deadlineGroups', props.deadlineGroups);
        this.validateRegion('deadlineRegion', props.deadlineRegion);
        this.validateBlockDevices(props.blockDevices);
    }
    validateFleetInstanceRole(fleetInstanceRole) {
        if (fleetInstanceRole) {
            if (core_1.Stack.of(fleetInstanceRole) !== core_1.Stack.of(this)) {
                throw new Error(`Fleet instance role should be created on the same stack as ${this.constructor.name} to avoid circular dependencies.`);
            }
        }
    }
    validateInstanceTypes(array) {
        if (array.length === 0) {
            throw new Error('At least one instance type is required for a Spot Fleet Request Configuration');
        }
    }
    validateSubnets(vpc, vpcSubnets) {
        const { subnets } = vpc.selectSubnets(vpcSubnets);
        if (subnets.length === 0) {
            core_1.Annotations.of(this).addError(`Did not find any subnets matching '${JSON.stringify(vpcSubnets)}', please use a different selection.`);
        }
    }
    validateGroups(property, array) {
        const regex = /^(?!none$)[a-zA-Z0-9-_]+$/i;
        if (array.length === 0) {
            throw new Error('At least one Deadline Group is required for a Spot Fleet Request Configuration');
        }
        array.forEach(value => {
            if (!regex.test(value)) {
                throw new Error(`Invalid value: ${value} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. Also, group 'none' is reserved as the default group.`);
            }
        });
    }
    validateRegion(property, region) {
        const regex = /^(?!none$|all$|unrecognized$)[a-zA-Z0-9-_]+$/i;
        if (region && !regex.test(region)) {
            throw new Error(`Invalid value: ${region} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. ‘All’, ‘none’ and ‘unrecognized’ are reserved names that cannot be used.`);
        }
    }
    validateBlockDevices(blockDevices) {
        if (blockDevices === undefined) {
            core_1.Annotations.of(this).addWarning(`The spot-fleet ${this.node.id} is being created without being provided any block devices so the Source AMI's devices will be used. ` +
                'Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
        }
        else {
            blockDevices.forEach(device => {
                if (device.volume.ebsDevice === undefined) {
                    // Suppressed or Ephemeral Block Device
                    return;
                }
                const { iops, volumeType } = device.volume.ebsDevice;
                if (!iops) {
                    if (volumeType === aws_ec2_1.EbsDeviceVolumeType.IO1) {
                        throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                    }
                }
                else if (volumeType !== aws_ec2_1.EbsDeviceVolumeType.IO1) {
                    core_1.Annotations.of(this).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
                }
                // encrypted is not exposed as part of ebsDeviceProps so we need to confirm it exists then access it via [].
                // eslint-disable-next-line dot-notation
                if (('encrypted' in device.volume.ebsDevice === false) || ('encrypted' in device.volume.ebsDevice && !device.volume.ebsDevice['encrypted'])) {
                    core_1.Annotations.of(this).addWarning(`The BlockDevice "${device.deviceName}" on the spot-fleet ${this.node.id} is not encrypted. ` +
                        'Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.');
                }
            });
        }
    }
}
exports.SpotEventPluginFleet = SpotEventPluginFleet;
_a = JSII_RTTI_SYMBOL_1;
SpotEventPluginFleet[_a] = { fqn: "aws-rfdk.deadline.SpotEventPluginFleet", version: "0.33.0" };
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
SpotEventPluginFleet.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * This is the current maximum for number of workers that can be started on a single host. Currently the
 * only thing using this limit is the configuration of the listener ports. More than 8 workers can be started,
 * but only the first 8 will have their ports opened in the workers' security group.
 */
SpotEventPluginFleet.MAX_WORKERS_PER_HOST = 8;
//# sourceMappingURL=data:application/json;base64,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