"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let stack;
let vpc;
const instanceType = aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO);
const machineImage = new aws_ec2_1.AmazonLinuxImage();
beforeEach(() => {
    stack = new core_1.Stack();
    vpc = new aws_ec2_1.Vpc(stack, 'VPC');
});
test('Grant SSM permissions to Instance', () => {
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType,
        machineImage,
    });
    lib_1.SessionManagerHelper.grantPermissionsTo(instance);
    const instanceRole = stack.getLogicalId(instance.role.node.defaultChild);
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'ssmmessages:CreateControlChannel',
                        'ssmmessages:CreateDataChannel',
                        'ssmmessages:OpenControlChannel',
                        'ssmmessages:OpenDataChannel',
                        'ssm:UpdateInstanceInformation',
                    ],
                    Effect: 'Allow',
                    Resource: '*',
                },
            ],
        },
        Roles: [{ Ref: instanceRole }],
    }));
});
test('Grant SSM permissions to ASG', () => {
    const asg = new aws_autoscaling_1.AutoScalingGroup(stack, 'ASG', {
        vpc,
        instanceType,
        machineImage,
    });
    lib_1.SessionManagerHelper.grantPermissionsTo(asg);
    const asgRole = stack.getLogicalId(asg.role.node.defaultChild);
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'ssmmessages:CreateControlChannel',
                        'ssmmessages:CreateDataChannel',
                        'ssmmessages:OpenControlChannel',
                        'ssmmessages:OpenDataChannel',
                        'ssm:UpdateInstanceInformation',
                    ],
                    Effect: 'Allow',
                    Resource: '*',
                },
            ],
        },
        Roles: [{ Ref: asgRole }],
    }));
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2Vzc2lvbnMtbWFuYWdlci1oZWxwZXIudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNlc3Npb25zLW1hbmFnZXItaGVscGVyLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7R0FHRzs7QUFFSCw0Q0FHeUI7QUFDekIsOERBRWtDO0FBQ2xDLDhDQU8wQjtBQUMxQix3Q0FBa0Q7QUFFbEQsZ0NBQThDO0FBRTlDLElBQUksS0FBWSxDQUFDO0FBQ2pCLElBQUksR0FBUSxDQUFDO0FBQ2IsTUFBTSxZQUFZLEdBQUcsc0JBQVksQ0FBQyxFQUFFLENBQUMsdUJBQWEsQ0FBQyxFQUFFLEVBQUUsc0JBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQztBQUMzRSxNQUFNLFlBQVksR0FBRyxJQUFJLDBCQUFnQixFQUFFLENBQUM7QUFFNUMsVUFBVSxDQUFDLEdBQUcsRUFBRTtJQUNkLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBQ3BCLEdBQUcsR0FBRyxJQUFJLGFBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFDOUIsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsbUNBQW1DLEVBQUUsR0FBRyxFQUFFO0lBQzdDLE1BQU0sUUFBUSxHQUFHLElBQUksa0JBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1FBQy9DLEdBQUc7UUFDSCxZQUFZO1FBQ1osWUFBWTtLQUNiLENBQUMsQ0FBQztJQUNILDBCQUFvQixDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBRWxELE1BQU0sWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBMEIsQ0FBQyxDQUFDO0lBRXZGLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsa0JBQWtCLEVBQUU7UUFDdkQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFO2dCQUNUO29CQUNFLE1BQU0sRUFBRTt3QkFDTixrQ0FBa0M7d0JBQ2xDLCtCQUErQjt3QkFDL0IsZ0NBQWdDO3dCQUNoQyw2QkFBNkI7d0JBQzdCLCtCQUErQjtxQkFDaEM7b0JBQ0QsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFLEdBQUc7aUJBQ2Q7YUFDRjtTQUNGO1FBQ0QsS0FBSyxFQUFFLENBQUMsRUFBRSxHQUFHLEVBQUUsWUFBWSxFQUFFLENBQUM7S0FDL0IsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw4QkFBOEIsRUFBRSxHQUFHLEVBQUU7SUFDeEMsTUFBTSxHQUFHLEdBQUcsSUFBSSxrQ0FBZ0IsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1FBQzdDLEdBQUc7UUFDSCxZQUFZO1FBQ1osWUFBWTtLQUNiLENBQUMsQ0FBQztJQUNILDBCQUFvQixDQUFDLGtCQUFrQixDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBRTdDLE1BQU0sT0FBTyxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBMEIsQ0FBQyxDQUFDO0lBRTdFLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsa0JBQWtCLEVBQUU7UUFDdkQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFO2dCQUNUO29CQUNFLE1BQU0sRUFBRTt3QkFDTixrQ0FBa0M7d0JBQ2xDLCtCQUErQjt3QkFDL0IsZ0NBQWdDO3dCQUNoQyw2QkFBNkI7d0JBQzdCLCtCQUErQjtxQkFDaEM7b0JBQ0QsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFLEdBQUc7aUJBQ2Q7YUFDRjtTQUNGO1FBQ0QsS0FBSyxFQUFFLENBQUMsRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLENBQUM7S0FDMUIsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbmltcG9ydCB7XG4gIGV4cGVjdCBhcyBleHBlY3RDREssXG4gIGhhdmVSZXNvdXJjZUxpa2UsXG59IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQge1xuICBBdXRvU2NhbGluZ0dyb3VwLFxufSBmcm9tICdAYXdzLWNkay9hd3MtYXV0b3NjYWxpbmcnO1xuaW1wb3J0IHtcbiAgQW1hem9uTGludXhJbWFnZSxcbiAgSW5zdGFuY2UsXG4gIEluc3RhbmNlQ2xhc3MsXG4gIEluc3RhbmNlU2l6ZSxcbiAgSW5zdGFuY2VUeXBlLFxuICBWcGMsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0IHsgQ2ZuRWxlbWVudCwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuaW1wb3J0IHsgU2Vzc2lvbk1hbmFnZXJIZWxwZXIgfSBmcm9tICcuLi9saWInO1xuXG5sZXQgc3RhY2s6IFN0YWNrO1xubGV0IHZwYzogVnBjO1xuY29uc3QgaW5zdGFuY2VUeXBlID0gSW5zdGFuY2VUeXBlLm9mKEluc3RhbmNlQ2xhc3MuVDMsIEluc3RhbmNlU2l6ZS5NSUNSTyk7XG5jb25zdCBtYWNoaW5lSW1hZ2UgPSBuZXcgQW1hem9uTGludXhJbWFnZSgpO1xuXG5iZWZvcmVFYWNoKCgpID0+IHtcbiAgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1ZQQycpO1xufSk7XG5cbnRlc3QoJ0dyYW50IFNTTSBwZXJtaXNzaW9ucyB0byBJbnN0YW5jZScsICgpID0+IHtcbiAgY29uc3QgaW5zdGFuY2UgPSBuZXcgSW5zdGFuY2Uoc3RhY2ssICdJbnN0YW5jZScsIHtcbiAgICB2cGMsXG4gICAgaW5zdGFuY2VUeXBlLFxuICAgIG1hY2hpbmVJbWFnZSxcbiAgfSk7XG4gIFNlc3Npb25NYW5hZ2VySGVscGVyLmdyYW50UGVybWlzc2lvbnNUbyhpbnN0YW5jZSk7XG5cbiAgY29uc3QgaW5zdGFuY2VSb2xlID0gc3RhY2suZ2V0TG9naWNhbElkKGluc3RhbmNlLnJvbGUubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuRWxlbWVudCk7XG5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgJ3NzbW1lc3NhZ2VzOkNyZWF0ZUNvbnRyb2xDaGFubmVsJyxcbiAgICAgICAgICAgICdzc21tZXNzYWdlczpDcmVhdGVEYXRhQ2hhbm5lbCcsXG4gICAgICAgICAgICAnc3NtbWVzc2FnZXM6T3BlbkNvbnRyb2xDaGFubmVsJyxcbiAgICAgICAgICAgICdzc21tZXNzYWdlczpPcGVuRGF0YUNoYW5uZWwnLFxuICAgICAgICAgICAgJ3NzbTpVcGRhdGVJbnN0YW5jZUluZm9ybWF0aW9uJyxcbiAgICAgICAgICBdLFxuICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICBSZXNvdXJjZTogJyonLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9LFxuICAgIFJvbGVzOiBbeyBSZWY6IGluc3RhbmNlUm9sZSB9XSxcbiAgfSkpO1xufSk7XG5cbnRlc3QoJ0dyYW50IFNTTSBwZXJtaXNzaW9ucyB0byBBU0cnLCAoKSA9PiB7XG4gIGNvbnN0IGFzZyA9IG5ldyBBdXRvU2NhbGluZ0dyb3VwKHN0YWNrLCAnQVNHJywge1xuICAgIHZwYyxcbiAgICBpbnN0YW5jZVR5cGUsXG4gICAgbWFjaGluZUltYWdlLFxuICB9KTtcbiAgU2Vzc2lvbk1hbmFnZXJIZWxwZXIuZ3JhbnRQZXJtaXNzaW9uc1RvKGFzZyk7XG5cbiAgY29uc3QgYXNnUm9sZSA9IHN0YWNrLmdldExvZ2ljYWxJZChhc2cucm9sZS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5FbGVtZW50KTtcblxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAnc3NtbWVzc2FnZXM6Q3JlYXRlQ29udHJvbENoYW5uZWwnLFxuICAgICAgICAgICAgJ3NzbW1lc3NhZ2VzOkNyZWF0ZURhdGFDaGFubmVsJyxcbiAgICAgICAgICAgICdzc21tZXNzYWdlczpPcGVuQ29udHJvbENoYW5uZWwnLFxuICAgICAgICAgICAgJ3NzbW1lc3NhZ2VzOk9wZW5EYXRhQ2hhbm5lbCcsXG4gICAgICAgICAgICAnc3NtOlVwZGF0ZUluc3RhbmNlSW5mb3JtYXRpb24nLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgIFJlc291cmNlOiAnKicsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgUm9sZXM6IFt7IFJlZjogYXNnUm9sZSB9XSxcbiAgfSkpO1xufSk7XG4iXX0=