"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PadEfsStorage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const runtime_info_1 = require("./runtime-info");
/**
 * This construct provides a mechanism that adds 1GB-sized files containing only zero-bytes to an Amazon EFS filesystem through a given Access Point to that filesystem.
 *
 * This is being
 * provided to give you a way to increase the baseline throughput of an Amazon EFS filesystem
 * that has been deployed in bursting throughput mode (see: https://docs.aws.amazon.com/efs/latest/ug/performance.html#throughput-modes).
 * This is most useful for your Amazon EFS filesystems that contain a very small amount of data and
 * have a baseline throughput that exceeds the throughput provided by the size of the filesystem.
 *
 * When deployed in bursting throughput mode, an Amazon EFS filesystem provides you with a baseline
 * throughput that is proportional to the amount of data stored in that filesystem. However, usage
 * of that filesystem is allowed to burst above that throughput; doing so consumes burst credits that
 * are associated with the filesystem. When all burst credits have been expended, then your filesystem
 * is no longer allowed to burst throughput and you will be limited in throughput to the greater of 1MiB/s
 * or the throughput dictated by the amount of data stored in your filesystem; the filesystem will be able
 * to burst again if it is able to accrue burst credits by staying below its baseline throughput for a time.
 *
 * Customers that deploy the Deadline Repository Filesystem on an Amazon EFS filesystem may find that
 * the filesystem does not contain sufficient data to meet the throughput needs of Deadline; evidenced by
 * a downward trend in EFS bursting credits over time. When bursting credits are expended, then the render
 * farm may begin to exhibit failure mode behaviors such as the RenderQueue dropping or refusing connections,
 * or becoming unresponsive.
 *
 * Warning: The implementation of this construct creates and starts an AWS Step Function to add the files
 * to the filesystem. The execution of this Step Function occurs asynchronously from your deployment. We recommend
 * verifying that the step function completed successfully via your Step Functions console.
 *
 * Resources Deployed
 * --------------------------
 * - Two AWS Lambda Functions, with roles, with full access to the given EFS Access Point.
 * - An Elastic Network Interface (ENI) for each Lambda Function in each of the selected VPC Subnets, so
 *    that the Lambda Functions can connect to the given EFS Access Point.
 * - An AWS Step Function to coordinate execution of the two Lambda Functions.
 * - Security Groups for each AWS Lambda Function.
 * - A CloudFormation custom resource that executes StepFunctions.startExecution on the Step Function
 *    whenever the stack containing this construct is created or updated.
 *
 * Security Considerations
 * ---------------------------
 * - The AWS Lambdas that are deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by these Lambdas.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - By default, the network interfaces created by this construct's AWS Lambda Functions have Security Groups
 *    that restrict egress access from the Lambda Function into your VPC such that the Lambda Functions can
 *    access only the given EFS Access Point.
 *
 * @stability stable
 */
class PadEfsStorage extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        /*
        Implementation:
         This is implemented as an AWS Step Function that implements the following
         algorithm:
         try {
          du = diskUsage(<efs access point directory>)
          while (du != desiredPadding) {
            if (du < desiredPadding) {
              <grow padding by adding up to 20 1GB numbered files to the filesystem.>
            } else if (du > desiredPadding) {
              <delete 1GB numbered files from the filesystem to reduce the padding to the desired amount>
              // Note: We break here to prevent two separate invocations of the step function (e.g. accidental manual
              // invocations) from looping indefinitely. Without a break, one invocation trying to grow while another
              // tries to shrink will infinitely loop both -- the diskUsage will never settle on the value that either
              // invocation wants.
              break;
            }
            du = diskUsage(<efs access point directory>)
          }
          return success
        } catch (error) {
          return failure
        }
         */
        const diskUsageTimeout = core_1.Duration.minutes(5);
        const paddingTimeout = core_1.Duration.minutes(15);
        // Location in the lambda environment where the EFS will be mounted.
        const efsMountPoint = '/mnt/efs';
        let desiredSize;
        try {
            desiredSize = props.desiredPadding.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL });
        }
        catch (err) {
            core_1.Annotations.of(this).addError('Failed to round desiredSize to an integer number of GiB. The size must be in GiB.');
        }
        const securityGroup = (_b = props.securityGroup) !== null && _b !== void 0 ? _b : new aws_ec2_1.SecurityGroup(this, 'LambdaSecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: false,
        });
        const lambdaProps = {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            runtime: aws_lambda_1.Runtime.NODEJS_14_X,
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            // Required for access point...
            vpc: props.vpc,
            vpcSubnets: (_c = props.vpcSubnets) !== null && _c !== void 0 ? _c : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            securityGroups: [securityGroup],
            filesystem: aws_lambda_1.FileSystem.fromEfsAccessPoint(props.accessPoint, efsMountPoint),
        };
        const diskUsage = new aws_lambda_1.Function(this, 'DiskUsage', {
            description: 'Used by RFDK PadEfsStorage to calculate disk usage of an EFS access point',
            handler: 'pad-efs-storage.getDiskUsage',
            timeout: diskUsageTimeout,
            memorySize: 128,
            ...lambdaProps,
        });
        // Implicit reference should have been fine, but the lambda is unable to mount the filesystem if
        // executed before the filesystem has been fully formed. We shouldn't have the lambda created until
        // after the EFS is created.
        diskUsage.node.addDependency(props.accessPoint);
        const doPadding = new aws_lambda_1.Function(this, 'PadFilesystem', {
            description: 'Used by RFDK PadEfsStorage to add or remove numbered 1GB files in an EFS access point',
            handler: 'pad-efs-storage.padFilesystem',
            timeout: paddingTimeout,
            // Execution requires about 70MB for just the lambda, but the filesystem driver will use every available byte.
            // Larger sizes do not seem to make a difference on filesystem write performance.
            // Set to 256MB just to give a buffer.
            memorySize: 256,
            ...lambdaProps,
        });
        // Implicit reference should have been fine, but the lambda is unable to mount the filesystem if
        // executed before the filesystem has been fully formed. We shouldn't have the lambda created until
        // after the EFS is created.
        doPadding.node.addDependency(props.accessPoint);
        // Build the step function's state machine.
        const fail = new aws_stepfunctions_1.Fail(this, 'Fail');
        const succeed = new aws_stepfunctions_1.Succeed(this, 'Succeed');
        const diskUsageTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'QueryDiskUsage', {
            lambdaFunction: diskUsage,
            comment: 'Determine the number of GB currently stored in the EFS access point',
            timeout: diskUsageTimeout,
            payload: {
                type: aws_stepfunctions_1.InputType.OBJECT,
                value: {
                    'desiredPadding.$': '$.desiredPadding',
                    'mountPoint': efsMountPoint,
                },
            },
            resultPath: '$.diskUsage',
        });
        const growTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'GrowTask', {
            lambdaFunction: doPadding,
            comment: 'Add up to 20 numbered 1GB files to the EFS access point',
            timeout: paddingTimeout,
            payload: {
                type: aws_stepfunctions_1.InputType.OBJECT,
                value: {
                    'desiredPadding.$': '$.desiredPadding',
                    'mountPoint': efsMountPoint,
                },
            },
            resultPath: '$.null',
        });
        const shrinkTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'ShrinkTask', {
            lambdaFunction: doPadding,
            comment: 'Remove 1GB numbered files from the EFS access point to shrink the padding',
            timeout: paddingTimeout,
            payload: {
                type: aws_stepfunctions_1.InputType.OBJECT,
                value: {
                    'desiredPadding.$': '$.desiredPadding',
                    'mountPoint': efsMountPoint,
                },
            },
            resultPath: '$.null',
        });
        const choice = new aws_stepfunctions_1.Choice(this, 'BranchOnDiskUsage')
            .when(aws_stepfunctions_1.Condition.numberLessThanJsonPath('$.diskUsage.Payload', '$.desiredPadding'), growTask)
            .when(aws_stepfunctions_1.Condition.numberGreaterThanJsonPath('$.diskUsage.Payload', '$.desiredPadding'), shrinkTask)
            .otherwise(succeed);
        diskUsageTask.next(choice);
        diskUsageTask.addCatch(fail, {
            // See: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-error-handling.html
            errors: ['States.ALL'],
        });
        growTask.next(diskUsageTask);
        growTask.addCatch(fail, {
            errors: ['States.ALL'],
        });
        shrinkTask.next(succeed);
        shrinkTask.addCatch(fail, {
            errors: ['States.ALL'],
        });
        const statemachine = new aws_stepfunctions_1.StateMachine(this, 'StateMachine', {
            definition: diskUsageTask,
        });
        // ==========
        // Invoke the step function on stack create & update.
        const invokeCall = {
            action: 'startExecution',
            service: 'StepFunctions',
            apiVersion: '2016-11-23',
            region: core_1.Stack.of(this).region,
            physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse('executionArn'),
            parameters: {
                stateMachineArn: statemachine.stateMachineArn,
                input: JSON.stringify({
                    desiredPadding: desiredSize,
                }),
            },
        };
        const resource = new custom_resources_1.AwsCustomResource(this, 'Default', {
            installLatestAwsSdk: true,
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            onCreate: invokeCall,
            onUpdate: invokeCall,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [statemachine.stateMachineArn],
            }),
        });
        resource.node.addDependency(statemachine);
        // Add RFDK tags to the construct tree.
        runtime_info_1.tagConstruct(this);
    }
}
exports.PadEfsStorage = PadEfsStorage;
_a = JSII_RTTI_SYMBOL_1;
PadEfsStorage[_a] = { fqn: "aws-rfdk.PadEfsStorage", version: "0.33.0" };
//# sourceMappingURL=data:application/json;base64,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