"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthMonitor = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_cloudwatch_actions_1 = require("@aws-cdk/aws-cloudwatch-actions");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sns_1 = require("@aws-cdk/aws-sns");
const aws_sns_subscriptions_1 = require("@aws-cdk/aws-sns-subscriptions");
const core_1 = require("@aws-cdk/core");
const load_balancer_manager_1 = require("./load-balancer-manager");
const runtime_info_1 = require("./runtime-info");
/**
 *  A new or imported Health Monitor.
 */
class HealthMonitorBase extends core_1.Construct {
}
/**
 * This construct is responsible for the deep health checks of compute instances.
 *
 * It also replaces unhealthy instances and suspends unhealthy fleets.
 * Although, using this constructs adds up additional costs for monitoring,
 * it is highly recommended using this construct to help avoid / minimize runaway costs for compute instances.
 *
 * An instance is considered to be unhealthy when:
 *    1) Deadline client is not installed on it;
 *    2) Deadline client is installed but not running on it;
 *    3) RCS is not configured correctly for Deadline client;
 *    4) it is unable to connect to RCS due to any infrastructure issues;
 *    5) the health monitor is unable to reach it because of some infrastructure issues.
 *
 * A fleet is considered to be unhealthy when:
 *    1) at least 1 instance is unhealthy for the configured grace period;
 *    2) a percentage of unhealthy instances in the fleet is above a threshold at any given point of time.
 *
 * This internally creates an array of application load balancers and attaches
 * the worker-fleet (which internally is implemented as an Auto Scaling Group) to its listeners.
 * There is no load-balancing traffic on the load balancers,
 * it is only used for health checks.
 * Intention is to use the default properties of laod balancer health
 * checks which does HTTP pings at frequent intervals to all the
 * instances in the fleet and determines its health. If any of the
 * instance is found unhealthy, it is replaced. The target group
 * also publishes the unhealthy target count metric which is used
 * to identify the unhealthy fleet.
 *
 * Other than the default instance level protection, it also creates a lambda
 * which is responsible to set the fleet size to 0 in the event of a fleet
 * being sufficiently unhealthy to warrant termination.
 * This lambda is triggered by CloudWatch alarms via SNS (Simple Notification Service).
 *
 * Resources Deployed
 * ------------------------
 * - Application Load Balancer(s) doing frequent pings to the workers.
 * - An Amazon Simple Notification Service (SNS) topic for all unhealthy fleet notifications.
 * - An AWS Key Management Service (KMS) Key to encrypt SNS messages - If no encryption key is provided.
 * - An Amazon CloudWatch Alarm that triggers if a worker fleet is unhealthy for a long period.
 * - Another CloudWatch Alarm that triggers if the healthy host percentage of a worker fleet is lower than allowed.
 * - A single AWS Lambda function that sets fleet size to 0 when triggered in response to messages on the SNS Topic.
 * - Execution logs of the AWS Lambda function are published to a log group in Amazon CloudWatch.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda that is created by this construct to terminate unhealthy worker fleets has permission to
 *    UpdateAutoScalingGroup ( https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_UpdateAutoScalingGroup.html )
 *    on all of the fleets that this construct is monitoring. You should not grant any additional actors/principals the
 *    ability to modify or execute this Lambda.
 * - Execution of the AWS Lambda for terminating unhealthy workers is triggered by messages to the Amazon Simple
 *    Notification Service (SNS) Topic that is created by this construct. Any principal that is able to publish notification
 *    to this SNS Topic can cause the Lambda to execute and reduce one of your worker fleets to zero instances. You should
 *    not grant any additional principals permissions to publish to this SNS Topic.
 *
 * @stability stable
 */
class HealthMonitor extends HealthMonitorBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.env = {
            account: this.stack.account,
            region: this.stack.region,
        };
        this.props = props;
        this.lbFactory = new load_balancer_manager_1.LoadBalancerFactory(this, props.vpc);
        const topicEncryptKey = props.encryptionKey || new aws_kms_1.Key(this, 'SNSEncryptionKey', {
            description: `This key is used to encrypt SNS messages for ${core_1.Names.uniqueId(this)}.`,
            enableKeyRotation: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            trustAccountIdentities: true,
        });
        // allow cloudwatch service to send encrypted messages
        topicEncryptKey.grant(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'), 'kms:Decrypt', 'kms:GenerateDataKey');
        this.unhealthyFleetActionTopic = new aws_sns_1.Topic(this, 'UnhealthyFleetTopic', {
            masterKey: topicEncryptKey,
        });
        this.unhealthyFleetActionTopic.grantPublish(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'));
        this.alarmTopicAction = new aws_cloudwatch_actions_1.SnsAction(this.unhealthyFleetActionTopic);
        this.unhealthyFleetActionLambda = new aws_lambda_1.SingletonFunction(this, 'UnhealthyFleetAction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs', 'unhealthyFleetAction')),
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            lambdaPurpose: 'unhealthyFleetTermination',
            timeout: core_1.Duration.seconds(300),
            uuid: '28bccf6a-aa76-478c-9239-e2f5bcc0254c',
        });
        this.unhealthyFleetActionTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(this.unhealthyFleetActionLambda));
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Attaches the load-balancing target to the ELB for instance-level monitoring.
     *
     * The ELB does frequent pings to the workers and determines
     * if a worker node is unhealthy. If so, it replaces the instance.
     *
     * It also creates an Alarm for healthy host percent and suspends the
     * fleet if the given alarm is breaching. It sets the maxCapacity
     * property of the auto-scaling group to 0. This should be
     * reset manually after fixing the issue.
     *
     * @stability stable
     */
    registerFleet(monitorableFleet, healthCheckConfig) {
        const { loadBalancer, targetGroup } = this.lbFactory.registerWorkerFleet(monitorableFleet, healthCheckConfig, this.props);
        this.createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup);
    }
    createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup) {
        monitorableFleet.connections.allowFrom(loadBalancer, aws_ec2_1.Port.tcp(healthCheckConfig.port || HealthMonitor.LOAD_BALANCER_LISTENING_PORT));
        const percentMetric = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD,
        });
        // When unhealthy fleet is more than healthyFleetThresholdPercent or 35% at any given period of 5 minutes
        const immediateTerminationAlarm = percentMetric.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetTermination', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: 100 - (healthCheckConfig.healthyFleetThresholdPercent || HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            actionsEnabled: true,
        });
        immediateTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        // When at least one node is unhealthy over a period of 2 hours
        const percentMetricGracePeriod = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE,
        });
        const gracePeriodTerminationAlarm = percentMetricGracePeriod.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetGracePeriod', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            actionsEnabled: true,
        });
        gracePeriodTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        monitorableFleet.targetUpdatePolicy.attachToRole(this.unhealthyFleetActionLambda.role);
    }
}
exports.HealthMonitor = HealthMonitor;
_a = JSII_RTTI_SYMBOL_1;
HealthMonitor[_a] = { fqn: "aws-rfdk.HealthMonitor", version: "0.33.0" };
/**
 * Default health check listening port.
 *
 * @stability stable
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_PORT = 63415;
/**
 * Resource Tracker in Deadline currently publish health status every 5 min, hence keeping this same.
 *
 * @stability stable
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_INTERVAL = core_1.Duration.minutes(5);
/**
 * Resource Tracker in Deadline currently determines host unhealthy in 15 min, hence keeping this count.
 *
 * @stability stable
 */
HealthMonitor.DEFAULT_UNHEALTHY_HOST_THRESHOLD = 3;
/**
 * This is the minimum possible value of ALB health-check config, we want to mark worker healthy ASAP.
 *
 * @stability stable
 */
HealthMonitor.DEFAULT_HEALTHY_HOST_THRESHOLD = 2;
/**
 * Since we are not doing any load balancing, this port is just an arbitrary port.
 *
 * @stability stable
 */
HealthMonitor.LOAD_BALANCER_LISTENING_PORT = 8081;
/**
 * This number is taken from Resource Tracker implementation. If a fleet's healthy percent
 * is less than this threshold at any given point of time, it is suspended.
 */
HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD = 65;
/**
 * This number is taken from Resource Tracker implementation. If a fleet has at least 1
 * unhealthy host for a period of 2 hours, it is suspended.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE = 0;
/**
 * This number is taken from Resource Tracker implementation. We monitor unhealthy fleet for immediate
 * termination for a period fo 5 minutes.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD = core_1.Duration.minutes(5);
/**
 * In Resource Tracker, we evaluate the fleet's health for determining the grace period over a period
 * of 5 minutes. For the first unhealthy signal, a instance can take upto 10min (max), hence we are
 * setting this period to be 15.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE = core_1.Duration.minutes(15);
/**
 * This number is taken from Resource Tracker implementation. Fleet is terminated immediately if it
 * has unhealthy host percent above the hard limit.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD = 1;
/**
 * This number is taken from Resource Tracker implementation. The grace period duration is 2 hours,
 * since the grace period is 15 minutes, we need continuous 8 data points crossing the threshold.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE = 8;
//# sourceMappingURL=data:application/json;base64,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