/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { Connections, IConnectable, Port, Protocol } from '@aws-cdk/aws-ec2';
import { ApplicationProtocol } from '@aws-cdk/aws-elasticloadbalancingv2';
/**
 * Properties for constructing an {@link Endpoint}.
 *
 * @stability stable
 */
export interface EndpointProps {
    /**
     * The address (either an IP or hostname) of the endpoint.
     *
     * @stability stable
     */
    readonly address: string;
    /**
     * The port number of the endpoint.
     *
     * @stability stable
     */
    readonly port: number;
    /**
     * The transport protocol of the endpoint.
     *
     * @default TCP
     * @stability stable
     */
    readonly protocol?: Protocol;
}
/**
 * Connection endpoint.
 *
 * Consists of a combination of hostname, port, and transport protocol.
 *
 * @stability stable
 */
export declare class Endpoint {
    /**
     * The minimum port value
     */
    private static readonly MIN_PORT;
    /**
     * The maximum port value
     */
    private static readonly MAX_PORT;
    /**
     * Determines if a port is valid
     *
     * @param port: The port number
     * @returns boolean whether the port is valid
     */
    private static isValidPort;
    /**
     * The hostname of the endpoint.
     *
     * @stability stable
     */
    readonly hostname: string;
    /**
     * The port of the endpoint.
     *
     * @stability stable
     */
    readonly port: Port;
    /**
     * The port number of the endpoint.
     *
     * This can potentially be a CDK token. If you need to embed the port in a string (e.g. instance user data script),
     * use {@link Endpoint.portAsString}.
     *
     * @stability stable
     */
    readonly portNumber: number;
    /**
     * The protocol of the endpoint.
     *
     * @stability stable
     */
    readonly protocol: Protocol;
    /**
     * The combination of "HOSTNAME:PORT" for this endpoint.
     *
     * @stability stable
     */
    readonly socketAddress: string;
    /**
     * Constructs an Endpoint instance.
     *
     * @param props The properties for the endpoint.
     * @stability stable
     */
    constructor(props: EndpointProps);
    /**
     * Returns the port number as a string representation that can be used for embedding within other strings.
     *
     * This is intended to deal with CDK's token system. Numeric CDK tokens are not expanded when their string
     * representation is embedded in a string. This function returns the port either as an unresolved string token or
     * as a resolved string representation of the port value.
     *
     * @returns An (un)resolved string representation of the endpoint's port number
     * @stability stable
     */
    portAsString(): string;
    private renderPort;
}
/**
 * Properties for constructing an {@link ApplicationEndpoint}.
 *
 * @stability stable
 */
export interface ApplicationEndpointProps {
    /**
     * The address (either an IP or hostname) of the endpoint.
     *
     * @stability stable
     */
    readonly address: string;
    /**
     * The port number of the endpoint.
     *
     * @stability stable
     */
    readonly port: number;
    /**
     * The application layer protocol of the endpoint.
     *
     * @default HTTPS
     * @stability stable
     */
    readonly protocol?: ApplicationProtocol;
}
/**
 * Properties for constructing an {@link ConnectableApplicationEndpoint}.
 *
 * @stability stable
 */
export interface ConnectableApplicationEndpointProps extends ApplicationEndpointProps {
    /**
     * The connection object of the application this endpoint is for.
     *
     * @stability stable
     */
    readonly connections: Connections;
}
/**
 * An endpoint serving http or https for an application.
 *
 * @stability stable
 */
export declare class ApplicationEndpoint extends Endpoint {
    /**
     * The http protocol that this web application listens on.
     *
     * @stability stable
     */
    readonly applicationProtocol: ApplicationProtocol;
    /**
     * Constructs a {@link ApplicationEndpoint} instance.
     *
     * @param props The properties for the application endpoint.
     * @stability stable
     */
    constructor(props: ApplicationEndpointProps);
}
/**
 * An endpoint serving http or https for an application.
 *
 * @stability stable
 */
export declare class ConnectableApplicationEndpoint extends ApplicationEndpoint implements IConnectable {
    /**
     * Allows specifying security group connections for the application.
     *
     * @stability stable
     */
    readonly connections: Connections;
    /**
     * Constructs a {@link ApplicationEndpoint} instance.
     *
     * @param props The properties for the application endpoint.
     * @stability stable
     */
    constructor(props: ConnectableApplicationEndpointProps);
}
