# **************************************************************************
# *
# * Authors:     Estrella Fernandez Gimenez (me.fernandez@cnb.csic.es)
# *
# *  BCU, Centro Nacional de Biotecnologia, CSIC
# *
# * This program is free software; you can redistribute it and/or modify
# * it under the terms of the GNU General Public License as published by
# * the Free Software Foundation; either version 2 of the License, or
# * (at your option) any later version.
# *
# * This program is distributed in the hope that it will be useful,
# * but WITHOUT ANY WARRANTY; without even the implied warranty of
# * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# * GNU General Public License for more details.
# *
# * You should have received a copy of the GNU General Public License
# * along with this program; if not, write to the Free Software
# * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
# * 02111-1307  USA
# *
# *  All comments concerning this program package may be sent to the
# *  e-mail address 'scipion@cnb.csic.es'
# *
# **************************************************************************

from os.path import basename, isfile

from pyworkflow import BETA
from pyworkflow.protocol.params import PathParam, PointerParam
from pyworkflow.utils.path import createAbsLink, copyFile
from pwem.emlib.image import ImageHandler

from tomo.protocols.protocol_base import ProtTomoImportFiles
from tomo.objects import SubTomogram
from tomo.utils import _getUniqueFileName

from ..convert import readDynTable


class DynamoImportSubtomos(ProtTomoImportFiles):
    """ This protocol imports subtomograms with metadata generated by a Dynamo table.
    A Dynamo catalogue can be also imported in order to relate subtomograms with their original tomograms. """

    _label = 'import subtomos from Dynamo'
    _devStatus = BETA

    # -------------------------- DEFINE param functions ----------------------
    def _defineParams(self, form):
        ProtTomoImportFiles._defineParams(self, form)
        form.addParam('tablePath', PathParam,
                      label="Dynamo table file:",
                      help='Select dynamo table (.tbl) to link dynamo metadata to the subtomograms that will be '
                           'imported to Scipion. ')
        form.addParam('tomoSet', PointerParam, pointerClass='SetOfTomograms',
                      label="Tomograms:",
                      help="Select the Tomograms used to extract the SubTomograms")

    # --------------------------- STEPS functions ------------------------------
    def _insertAllSteps(self):
        self._insertFunctionStep('importSubTomogramsStep', self.getPattern(), self.samplingRate.get())
        self._insertFunctionStep('createOutputStep')

    # --------------------------- STEPS functions -----------------------------
    def importSubTomogramsStep(self, pattern, samplingRate):
        self.info("Using pattern: '%s'" % pattern)
        subtomo = SubTomogram()
        subtomo.setSamplingRate(samplingRate)
        imgh = ImageHandler()
        self.subtomoSet = self._createSetOfSubTomograms()
        self.subtomoSet.setSamplingRate(samplingRate)
        dynTable = self._getExtraPath('dynamo_table.tbl')
        copyFile(self.tablePath.get(), dynTable)
        self.fhTable = open(dynTable, 'r')
        for fileName, fileId in self.iterFiles():
            _, _, _, n = imgh.getDimensions(fileName)
            newFileName = _getUniqueFileName(self.getPattern(), fileName)
            if fileName.endswith(':mrc'):
                fileName = fileName[:-4]
            createAbsLink(fileName, newFileName)
            if n == 1:
                self._addSubtomogram(subtomo, fileName, newFileName)
            else:
                for index in range(1, n + 1):
                    self._addSubtomogram(subtomo, newFileName, index=index)
        self.fhTable.close()

    def _addSubtomogram(self, subtomo, newFileName, index=None):
        """ adds a subtomogram to a set """
        tomoSet = self.tomoSet.get()
        subtomo.cleanObjId()
        if index is None:
            subtomo.setFileName(newFileName)
        else:
            subtomo.setLocation(index, newFileName)
        readDynTable(self, subtomo, tomoSet)
        self.subtomoSet.append(subtomo)

    def createOutputStep(self):
        self._defineOutputs(outputSubTomograms=self.subtomoSet)

    # --------------------------- INFO functions ------------------------------
    def _hasOutput(self):
        return self.hasAttribute('outputSubTomograms')

    def _getSubTomMessage(self):
        return "SubTomograms %s" % self.getObjectTag('outputSubTomograms')

    def _summary(self):
        summary = []
        if self._hasOutput():
            summary.append("%s imported from:\n%s"
                           % (self._getSubTomMessage(), self.getPattern()))
            if self.samplingRate.get():
                summary.append(u"Sampling rate: *%0.2f* (Å/px)" %
                               self.samplingRate.get())
        return summary

    def _methods(self):
        methods = []
        if self._hasOutput():
            methods.append(" %s imported with a sampling rate *%0.2f*" %
                           (self._getSubTomMessage(), self.samplingRate.get()))
        return methods

    def _getVolumeFileName(self, fileName, extension=None):
        if extension is not None:
            baseFileName = "import_" + str(basename(fileName)).split(".")[0] + ".%s" % extension
        else:
            baseFileName = "import_" + str(basename(fileName)).split(":")[0]
        return self._getExtraPath(baseFileName)

    def _validate(self):
        errors = []
        try:
            next(self.iterFiles())
        except StopIteration:
            errors.append('No files matching the pattern %s were found.' % self.getPattern())
        if not self.tablePath.get() or not isfile(self.tablePath.get()):
            errors.append("Could not find specified dynamo catalogue file")
        return errors
