"use strict";
const autoscaling = require("@aws-cdk/aws-autoscaling");
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const user_data_1 = require("../lib/user-data");
function newFixtures(spot = false) {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'my-stack', { env: { region: 'us-west-33' } });
    const vpc = new ec2.Vpc(stack, 'vpc');
    const asg = new autoscaling.AutoScalingGroup(stack, 'ASG', {
        instanceType: new ec2.InstanceType('m4.xlarge'),
        machineImage: new ec2.AmazonLinuxImage(),
        spotPrice: spot ? '0.01' : undefined,
        vpc,
    });
    return { stack, vpc, asg };
}
module.exports = {
    'default user data'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderAmazonLinuxUserData('my-cluster-name', asg));
        // THEN
        test.deepEqual(userData, [
            'set -o xtrace',
            '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true',
            '/opt/aws/bin/cfn-signal --exit-code $? --stack my-stack --resource ASG46ED3070 --region us-west-33',
        ]);
        test.done();
    },
    '--use-max-pods=true'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderAmazonLinuxUserData('my-cluster-name', asg, {
            useMaxPods: true,
        }));
        // THEN
        test.deepEqual(userData[1], '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true');
        test.done();
    },
    '--use-max-pods=false'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderAmazonLinuxUserData('my-cluster-name', asg, {
            useMaxPods: false,
        }));
        // THEN
        test.deepEqual(userData[1], '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods false');
        test.done();
    },
    '--aws-api-retry-attempts'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderAmazonLinuxUserData('my-cluster-name', asg, {
            awsApiRetryAttempts: 123,
        }));
        // THEN
        test.deepEqual(userData[1], '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true --aws-api-retry-attempts 123');
        test.done();
    },
    '--docker-config-json'(test) {
        // GIVEN
        const { asg } = newFixtures();
        // WHEN
        const userData = user_data_1.renderAmazonLinuxUserData('my-cluster-name', asg, {
            dockerConfigJson: '{"docker":123}',
        });
        // THEN
        test.deepEqual(userData[1], '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true --docker-config-json \'{"docker":123}\'');
        test.done();
    },
    '--enable-docker-bridge=true'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderAmazonLinuxUserData('my-cluster-name', asg, {
            enableDockerBridge: true,
        }));
        // THEN
        test.deepEqual(userData[1], '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true --enable-docker-bridge');
        test.done();
    },
    '--enable-docker-bridge=false'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderAmazonLinuxUserData('my-cluster-name', asg, {
            enableDockerBridge: false,
        }));
        // THEN
        test.deepEqual(userData[1], '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true');
        test.done();
    },
    '--kubelet-extra-args'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderAmazonLinuxUserData('my-cluster-name', asg, {
            kubeletExtraArgs: '--extra-args-for --kubelet',
        }));
        // THEN
        test.deepEqual(userData[1], '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand  --extra-args-for --kubelet" --use-max-pods true');
        test.done();
    },
    'arbitrary additional bootstrap arguments can be passed through "additionalArgs"'(test) {
        // GIVEN
        const { asg, stack } = newFixtures();
        // WHEN
        const userData = stack.resolve(user_data_1.renderAmazonLinuxUserData('my-cluster-name', asg, {
            additionalArgs: '--apiserver-endpoint 1111 --foo-bar',
        }));
        // THEN
        test.deepEqual(userData[1], '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true --apiserver-endpoint 1111 --foo-bar');
        test.done();
    },
    'if asg has spot instances, the correct label and taint is used'(test) {
        // GIVEN
        const { asg, stack } = newFixtures(true);
        // WHEN
        const userData = stack.resolve(user_data_1.renderAmazonLinuxUserData('my-cluster-name', asg, {
            kubeletExtraArgs: '--node-labels X=y',
        }));
        // THEN
        test.deepEqual(userData[1], '/etc/eks/bootstrap.sh my-cluster-name --kubelet-extra-args "--node-labels lifecycle=Ec2Spot --register-with-taints=spotInstance=true:PreferNoSchedule --node-labels X=y" --use-max-pods true');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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