# ==================================================================
#       文 件 名: abstract_models.py
#       概    要: 模型基类
#       作    者: IT小强 
#       创建时间: 6/3/20 5:36 PM
#       修改时间: 
#       copyright (c) 2016 - 2020 mail@xqitw.cn
# ==================================================================

from django.db import models
from django.utils.translation import gettext_lazy as _


class Model(models.Model):
    """
    数据模型基类
    """

    # 外外键关联时的删除方式
    FOREIGN_DELETE_TYPE = models.PROTECT

    class Meta:
        abstract = True


class ModelCreatedBy(models.Model):
    """
    添加创建人外键
    """

    # 创建人
    created_by = models.ForeignKey(
        'User',
        verbose_name=_('创建用户ID'),
        related_name="%(app_label)s_%(class)s_created_by",
        on_delete=Model.FOREIGN_DELETE_TYPE,
        null=True,
        blank=True,
        editable=False,
        default=None
    )

    class Meta:
        abstract = True


class ModelUpdatedBy(models.Model):
    """
    添加更新人外键
    """

    # 更新人
    updated_by = models.ForeignKey(
        'User',
        verbose_name=_('更新用户ID'),
        related_name="%(app_label)s_%(class)s_updated_by",
        on_delete=Model.FOREIGN_DELETE_TYPE,
        null=True,
        blank=True,
        editable=False,
        default=None
    )

    class Meta:
        abstract = True


class ModelCreatedTime(models.Model):
    """
    添加创建时间
    """

    # 创建时间
    created_at = models.DateTimeField(verbose_name=_('创建时间'), auto_now_add=True, editable=True)

    class Meta:
        abstract = True


class ModelUpdatedTime(models.Model):
    """
    添加更新时间
    """

    # 更新时间
    updated_at = models.DateTimeField(verbose_name=_('更新时间'), auto_now=True)

    class Meta:
        abstract = True


class ModelStatus(models.Model):
    """
    添加状态字段
    """

    STATUS_CHOICES = [
        (-1, '草稿'),
        (0, '待审'),
        (1, '通过'),
        (2, '驳回'),
    ]

    # 状态
    status = models.IntegerField(
        verbose_name=_('状态'),
        null=False,
        default=0,
        choices=STATUS_CHOICES
    )

    class Meta:
        abstract = True


class ModelEnabled(models.Model):
    """
    添加是否启用字段
    """

    # 是否启用
    enabled = models.BooleanField(
        verbose_name=_('是否启用'),
        default=False,
        db_index=True,
    )

    class Meta:
        abstract = True


class ModelSort(models.Model):
    """
    添加排序字段
    """

    # 排序
    sort = models.IntegerField(
        verbose_name=_('排序'),
        null=False,
        default=0,
    )

    class Meta:
        abstract = True


class ModelBaseAll(
    ModelCreatedBy, ModelUpdatedBy,
    ModelCreatedTime, ModelUpdatedTime,
    ModelSort, ModelStatus, ModelEnabled
):
    """
    数据模型基类(添加全部公用字段)
    """

    class Meta:
        abstract = True
