# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['matterapi',
 'matterapi.client',
 'matterapi.endpoints',
 'matterapi.endpoints.async_api',
 'matterapi.endpoints.sync_api']

package_data = \
{'': ['*']}

install_requires = \
['httpx>=0.21.0',
 'pydantic>=1.9.0,<2.0.0',
 'python-dateutil>=2.8.0,<3.0.0',
 'websockets>=10.0,<11.0']

setup_kwargs = {
    'name': 'matterapi',
    'version': '0.4.0',
    'description': 'A client library for Mattermost API supporting sync/async',
    'long_description': '[![Readthedocs](https://readthedocs.org/projects/matterapi/badge/?version=latest)](https://matterapi.readthedocs.io/en/latest/)\n[![PyPI version](https://img.shields.io/pypi/v/matterapi)](https://pypi.org/project/matterapi/)\n\n# MatterApi\n\nA python client library for the Mattermost API with sync/async support.\n\n\n## Features\n+ Endpoints generated from the [API documentation](https://api.mattermost.com/)\n  - However: if the description is wrong, this library will be wrong\n+ Based on [httpx](https://www.python-httpx.org/) and supports sync and async operation\n+ Response parsing into models with [pydantic](https://pydantic-docs.helpmanual.io/)\n+ Typing support\n+ Websocket handling to connect to the Mattermost event stream.\n\n## Getting Started\n\nThe client has synch and async support. Depending on your workflow you will want to choose between\nthe `SyncClient` and `AsyncClient`.\n\n\n### SyncClient\n\nFirst, let\'s look at an example on how to use the SyncClient:\n\n```python\nfrom matterapi import SyncClient\nfrom matterapi.client.exceptions import ApiError, InvalidOrMissingParameters\nfrom matterapi.models import (\n    CreateChannelJsonBody,\n    CreatePostJsonBody,\n    CreateTeamJsonBody,\n    CreateUserJsonBody,\n)\n\n# set the options for the client\noptions = { \'url\' : \'http://localhost:8095\',\n    \'auth\' : { \n      \'token\' : \'<yourtokenhere>\' \n      }\n}\n\n# Create a sync client\nsd = SyncClient(options=options)\n\n# Use the client\n""" You can use the client directly. This will create a new httpx.Client for each\nrequest (an potentially run the login flow again, if you use username/password authentication).\n"""\n\n## Get your own user object\nsd.users.get_user("me")\n\n## Get channels\nchannels = sd.channels.get_all_channels(per_page=20)\nprint(channels[0])\n\n## Get posts for channel\npost_list = sd.posts.get_posts_for_channel("<channel_id>")\nfor post in post_list.posts:\n  print(post, post_list.posts[post].id)\n\n\n""" Or you can open a session, which will reuse a single httpx client instance\nfor the requests, thereby reusing connections ( the login flow is then only run once).\nThe returned object can be used exactly the same as the SyncClient.\nMost often you will want to use this as it requires fewer connections to the server\n"""\n\nwith sd.session() as api_session:\n    # Get the current user\n    user = api_session.users.get_user("me")\n    print("User id:", user.id)\n\n    # Get a list of plugins\n    plugins_avail = api_session.plugins.get_plugins()\n    print("Number of active plugins:", len(plugins_avail.active))\n    print("Name of first active plugin:", plugins_avail.active[0].name)\n\n    # Create a new Team or get existing\n    new_team = CreateTeamJsonBody(name="rebels", display_name="Rebels", type="I")\n    try:\n        team = api_session.teams.create_team(json_body=new_team)\n        print("New Team created with id:", team.id)\n    except InvalidOrMissingParameters as e:\n        print("> Exception:", e)\n        team = api_session.teams.get_team_by_name(name=new_team.name)\n        print("Got existing Team with id:", team.id)\n\n    # Create a channel or get existing\n    new_channel = CreateChannelJsonBody(\n        team_id=team.id, name="newchannelname", display_name="New Channel", type="O"\n    )\n    try:\n        channel = api_session.channels.create_channel(json_body=new_channel)\n        print(f"New Channel with id:", channel.id)\n    except InvalidOrMissingParameters as e:\n        print("> Exception:", e)\n        channel = api_session.channels.get_channel_by_name(\n            team_id=team.id, channel_name=new_channel.name\n        )\n        print(f"Got existing Channel with id:", channel.id)\n\n    # Upload files to be used in a channel\n    file_info = api_session.files.upload_file(\n        channel_id=channel.id,\n        multipart_data={\n            "files": {\n                "upload1.png": open("testfile1.png", "rb"),\n                "upload2.png": open("testfile2.png", "rb"),\n            },\n        },\n    )\n    print("File id for second file:", file_info.file_infos[1].id)\n    \n    # Create a post including the uploaded files\n    new_post = CreatePostJsonBody(\n        channel_id=channel.id,\n        message="A wild message appears",\n        file_ids=[info.id for info in file_info.file_infos],\n    )\n    post = api_session.posts.create_post(json_body=new_post)\n    print("Creation time for new post:", post.create_at)\n```\n\n### AsyncClient\n\nAnd this is how you can use the AsyncClient\n\n```python\n\nimport asyncio\nfrom matterapi import AsyncClient\n\n# set the options for the client\noptions = { \'url\' : \'https://localhost:8095\',\n    # User username and password authentication\n    \'auth\' : { \n      \'login_id\' : \'hansolo\', \n      \'password\' : \'lea1234\' \n      },\n    # Disable TLS verification for the client\n    # httpx_client_options are passed to the underlying httpx.Client\n    \'httpx_client_options\' : {\n      \'verify\' : False\n    }\n}\n\n# Create a async client\nad = AsyncClient(options=options)\n\nasync def do_something():\n  # Use the client\n  users = await ad.users.get_users()\n  print(users)\n\n  # To upload files, you could for example use the following request\n  data = {\n          "files": {\n              "test1.png": open("testfile1.png", "rb"),\n              "test2.png": open("testfile2.png", "rb"),\n          }\n      }\n\n  file_infos = await ad.files.upload_file(\n      channel_id="7bzsijaqopfczygxm1qc3r63do",\n      multipart_data=data)\n\n  print(file_info)\n\nasyncio.run(do_something())\n```\n\nThe AsyncClient supports sessions as well, same as the SyncClient.\n\n## Websocket connection\n\nYou can also listen for event data from Mattermost\n\n```python\n\ndef handler(message):\n  print(message)\n\n# either use the sync version\nsd.start_ws_sync(handler)\n\n# or the async version with \nasync def wrapper():\n  await ad.start_ws(handler)\n\n```\n\n\nContributing\n------------\n\nThe actual endpoints in Matterapi are autogenerated with a fork of [openapi-python-client](https://github.com/gmerz/openapi-python-client). If anything there needs changing, please refer to the generator project.\n\nSome endpoints might return wrong results or miss parameters. Currently, I can image the following two reasons:\n\n1. The generator is broken and/or does not support the required feature set.\n    - Create an issue in the [matterapi-generator](https://github.com/gmerz/matterapi-generator) repository\n    - If you know how to fix it, consider to do a pull request\n\n2. The [mattermost api documentation](https://github.com/mattermost/mattermost-api-reference), which this client is generated from, is not correct\n    - Sometimes entries in the documentation might not be completely correct\n    - Consider fixing the API documentation/do a pull request/post an issue there\n    - This will help everybody who is using the documentation\n\n3. The API documentation changed or included new endpoints, but the library was not updated.\n    - Currently creation of the api client is not automated, so you have two options:\n        1. Help with the automation so that this proejct is automatically updated with changes\n        2. Create a new issue here that the client is out of sync\n        3. Clone the generator repo, update the mattermost api with the generator and do a pull request here. (This is currently the only save way to get changes into the matterapi folder)\n\n\nCredits\n-------\n\nCredits where credits are due:\n\n+ This library is autogenerated from the Mattermost API documentation using a fork of [openapi-python-client](https://github.com/triaxtec/openapi-python-client). \n+ It\'s heavily inspired by (but not a 1:1 drop-in replacement for) [mattermostdriver](https://github.com/Vaelor/python-mattermost-driver), which I used for several years already. This is still your go-to if you look for something stable that has been in use for years by lot\'s of people.\n\n',
    'author': 'Georg Merzdovnik',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://matterapi.readthedocs.io/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0.0',
}


setup(**setup_kwargs)
