# BSD 2-CLAUSE LICENSE

# Redistribution and use in source and binary forms, with or without modification,
# are permitted provided that the following conditions are met:

# Redistributions of source code must retain the above copyright notice, this
# list of conditions and the following disclaimer.
# Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
# #ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# original author: Albert Chen
"""Constants used by code in `~greykite.common` or in multiple places:
`~greykite.algo`, `~greykite.sklearn`,
and/or `~greykite.framework`.
"""

# The time series data is represented in pandas dataframes
# The default column names for the series are given below
TIME_COL = "ts"
"""The default name for the column with the timestamps of the time series"""
VALUE_COL = "y"
"""The default name for the column with the values of the time series"""
ACTUAL_COL = "actual"
"""The column name representing actual (observed) values"""
PREDICTED_COL = "forecast"
"""The column name representing the predicted values"""
PREDICTED_LOWER_COL = "forecast_lower"
"""The column name representing upper bounds of prediction interval"""
PREDICTED_UPPER_COL = "forecast_upper"
"""The column name representing lower bounds of prediction interval"""
NULL_PREDICTED_COL = "forecast_null"
"""The column name representing predicted values from null model"""
ERR_STD_COL = "err_std"
"""The column name representing the error standard deviation from models"""

# Evaluation metrics corresponding to `~greykite.common.evaluation`
R2_null_model_score = "R2_null_model_score"
"""Evaluation metric. Improvement in the specified loss function compared to the predictions of a null model."""
FRACTION_OUTSIDE_TOLERANCE = "Outside Tolerance (fraction)"
"""Evaluation metric. The fraction of predictions outside the specified tolerance level"""
PREDICTION_BAND_WIDTH = "Prediction Band Width (%)"
"""Evaluation metric. Relative size of prediction bands vs actual, as a percent"""
PREDICTION_BAND_COVERAGE = "Prediction Band Coverage (fraction)"
"""Evaluation metric. Fraction of observations within the bands"""
LOWER_BAND_COVERAGE = "Coverage: Lower Band"
"""Evaluation metric. Fraction of observations within the lower band"""
UPPER_BAND_COVERAGE = "Coverage: Upper Band"
"""Evaluation metric. Fraction of observations within the upper band"""
COVERAGE_VS_INTENDED_DIFF = "Coverage Diff: Actual_Coverage - Intended_Coverage"
"""Evaluation metric. Difference between actual and intended coverage"""

# Column names used by `~greykite.common.features.timeseries_features`
EVENT_DF_DATE_COL = "date"
"""Name of date column for the DataFrames passed to silverkite `custom_daily_event_df_dict`"""
EVENT_DF_LABEL_COL = "event_name"
"""Name of event column for the DataFrames passed to silverkite `custom_daily_event_df_dict`"""
EVENT_PREFIX = "events"
"""Prefix for naming event features."""
EVENT_DEFAULT = ""
"""Label used for days without an event."""
EVENT_INDICATOR = "event"
"""Binary indicatory for an event"""
CHANGEPOINT_COL_PREFIX = "changepoint"
"""Prefix for naming changepoint features."""
CHANGEPOINT_COL_PREFIX_SHORT = "cp"
"""Short prefix for naming changepoint features."""

# Column names used by
# `~greykite.common.features.adjust_anomalous_data.adjust_anomalous_data`
START_DATE_COL = "start_date"
"""Start timestamp column name"""
END_DATE_COL = "end_date"
"""Standard end timestamp column"""
ADJUSTMENT_DELTA_COL = "adjustment_delta"
"""Adjustment column"""
METRIC_COL = "metric"
"""Column to denote metric of interest"""
DIMENSION_COL = "dimension"
"""Dimension column"""

# Constants related to
# `~greykite.common.features.timeseries_features.build_time_features_df`.
GROWTH_COL_ALIAS = {
    "linear": "ct1",
    "quadratic": "ct2",
    "cubic": "ct3",
    "sqrt": "ct_sqrt",
    "cuberoot": "ct_root3"
}
"""Human-readable names for the growth columns generated by
`~greykite.common.features.timeseries_features.build_time_features_df`
"""
TIME_FEATURES = [
    "datetime", "date",
    "year", "year_length",
    "quarter", "quarter_start", "quarter_length",
    "month", "month_length",
    "woy", "doy", "doq", "dom", "dow",
    "str_dow", "str_doy",
    "hour", "minute", "second",
    "year_month",
    "year_woy", "month_dom", "year_woy_dow", "woy_dow", "dow_hr", "dow_hr_min",
    "tod", "tow", "tom", "toq", "toy",
    "conti_year", "is_weekend", "dow_grouped",
    "ct1", "ct2", "ct3", "ct_sqrt", "ct_root3"]
"""Time features generated by
`~greykite.common.features.timeseries_features.build_time_features_df`"""

# Column names used by
# `~greykite.common.features.timeseries_lags`
LAG_INFIX = "_lag"
"""Infix for lagged feature names"""
AGG_LAG_INFIX = "avglag"
"""Infix for aggregated lag feature names"""

# Patterns for categorizing timeseries features
TREND_REGEX = f"{CHANGEPOINT_COL_PREFIX}\\d|ct\\d|ct_|{CHANGEPOINT_COL_PREFIX_SHORT}\\d"
"""Growth terms, including changepoints."""
SEASONALITY_REGEX = "sin\\d|cos\\d"
"""Seasonality terms modeled by fourier series."""
EVENT_REGEX = f"{EVENT_PREFIX}_"
"""Event terms."""
LAG_REGEX = f"{LAG_INFIX}\\d|_{AGG_LAG_INFIX}_\\d"
"""Lag terms."""

LOGGER_NAME = "Greykite"
"""Name used by the logger."""
