"use strict";
const assert_1 = require("@aws-cdk/assert");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
function minimalPipeline(stack, options = {}) {
    const sourceOutput = new codepipeline.Artifact();
    const sourceAction = new cpactions.GitHubSourceAction({
        actionName: 'Source',
        owner: 'aws',
        repo: 'aws-cdk',
        output: sourceOutput,
        oauthToken: core_1.SecretValue.plainText('secret'),
    });
    const pipeline = new codepipeline.Pipeline(stack, 'MyPipeline', {
        stages: [
            {
                stageName: 'Source',
                actions: [sourceAction],
            }
        ]
    });
    const deploymentStage = pipeline.addStage({
        stageName: 'Deploy',
        actions: [
            new cpactions.S3DeployAction({
                accessControl: options.accessControl,
                actionName: 'CopyFiles',
                bucket: options.bucket || new s3.Bucket(stack, 'MyBucket'),
                cacheControl: options.cacheControl,
                extract: options.extract,
                input: sourceOutput,
                objectKey: options.objectKey,
            }),
        ],
    });
    return deploymentStage;
}
module.exports = {
    'S3 Deploy Action': {
        'by default extract artifacts'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack);
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    {
                        "Name": "Source",
                        "Actions": [
                            {
                                "Name": "Source",
                                "ActionTypeId": {
                                    "Category": "Source",
                                    "Owner": "ThirdParty"
                                }
                            },
                        ]
                    },
                    {
                        "Name": "Deploy",
                        "Actions": [
                            {
                                "ActionTypeId": {
                                    "Category": "Deploy",
                                    "Provider": "S3",
                                },
                                "Configuration": {
                                    "Extract": "true"
                                },
                                "Name": "CopyFiles"
                            }
                        ]
                    }
                ],
            }));
            test.done();
        },
        'grant the pipeline correct access to the target bucket'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack);
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Effect": "Allow",
                            "Action": [
                                "s3:GetObject*",
                                "s3:GetBucket*",
                                "s3:List*",
                                "s3:DeleteObject*",
                                "s3:PutObject*",
                                "s3:Abort*"
                            ]
                        },
                        {},
                        {
                            "Effect": "Allow",
                            "Action": "sts:AssumeRole"
                        }
                    ]
                }
            }));
            test.done();
        },
        'kebab-case CannedACL value'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, {
                accessControl: s3.BucketAccessControl.PUBLIC_READ_WRITE
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    {},
                    {
                        "Actions": [
                            {
                                "Configuration": {
                                    "CannedACL": "public-read-write"
                                }
                            }
                        ]
                    }
                ]
            }));
            test.done();
        },
        'allow customizing cache-control'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, {
                cacheControl: [
                    cpactions.CacheControl.setPublic(),
                    cpactions.CacheControl.maxAge(core_1.Duration.hours(12)),
                ]
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    {},
                    {
                        "Actions": [
                            {
                                "Configuration": {
                                    "CacheControl": "public, max-age: 43200"
                                }
                            }
                        ]
                    }
                ]
            }));
            test.done();
        },
        'allow customizing objectKey (deployment path on S3)'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, {
                objectKey: "/a/b/c"
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                "Stages": [
                    {},
                    {
                        "Actions": [
                            {
                                "Configuration": {
                                    "ObjectKey": "/a/b/c"
                                }
                            }
                        ]
                    }
                ]
            }));
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5zMy1kZXBsb3ktYWN0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5zMy1kZXBsb3ktYWN0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw0Q0FBMkQ7QUFDM0QsMERBQTBEO0FBQzFELHNDQUFzQztBQUN0Qyx3Q0FBNkQ7QUFFN0QsdUNBQXVDO0FBZ0t2QyxTQUFTLGVBQWUsQ0FBQyxLQUFZLEVBQUUsVUFBa0MsRUFBRTtJQUN6RSxNQUFNLFlBQVksR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztJQUNqRCxNQUFNLFlBQVksR0FBRyxJQUFJLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQztRQUNwRCxVQUFVLEVBQUUsUUFBUTtRQUNwQixLQUFLLEVBQUUsS0FBSztRQUNaLElBQUksRUFBRSxTQUFTO1FBQ2YsTUFBTSxFQUFFLFlBQVk7UUFDcEIsVUFBVSxFQUFFLGtCQUFXLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQztLQUM1QyxDQUFDLENBQUM7SUFFSCxNQUFNLFFBQVEsR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtRQUM5RCxNQUFNLEVBQUU7WUFDTjtnQkFDRSxTQUFTLEVBQUUsUUFBUTtnQkFDbkIsT0FBTyxFQUFFLENBQUUsWUFBWSxDQUFFO2FBQzFCO1NBQ0Y7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLGVBQWUsR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDO1FBQ3hDLFNBQVMsRUFBRSxRQUFRO1FBQ25CLE9BQU8sRUFBRTtZQUNQLElBQUksU0FBUyxDQUFDLGNBQWMsQ0FBQztnQkFDM0IsYUFBYSxFQUFFLE9BQU8sQ0FBQyxhQUFhO2dCQUNwQyxVQUFVLEVBQUUsV0FBVztnQkFDdkIsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLElBQUksSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxVQUFVLENBQUM7Z0JBQzFELFlBQVksRUFBRSxPQUFPLENBQUMsWUFBWTtnQkFDbEMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxPQUFPO2dCQUN4QixLQUFLLEVBQUUsWUFBWTtnQkFDbkIsU0FBUyxFQUFFLE9BQU8sQ0FBQyxTQUFTO2FBQzdCLENBQUM7U0FDSDtLQUNGLENBQUMsQ0FBQztJQUVILE9BQU8sZUFBZSxDQUFDO0FBQ3pCLENBQUM7QUEvTEQsaUJBQVM7SUFDUCxrQkFBa0IsRUFBRTtRQUNsQiw4QkFBOEIsQ0FBQyxJQUFVO1lBQ3ZDLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsZUFBZSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBRXZCLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsNkJBQTZCLEVBQUU7Z0JBQy9ELFFBQVEsRUFBRTtvQkFDUjt3QkFDRSxNQUFNLEVBQUUsUUFBUTt3QkFDaEIsU0FBUyxFQUFFOzRCQUNUO2dDQUNFLE1BQU0sRUFBRSxRQUFRO2dDQUNoQixjQUFjLEVBQUU7b0NBQ2QsVUFBVSxFQUFFLFFBQVE7b0NBQ3BCLE9BQU8sRUFBRSxZQUFZO2lDQUN0Qjs2QkFDRjt5QkFDRjtxQkFDRjtvQkFDRDt3QkFDRSxNQUFNLEVBQUUsUUFBUTt3QkFDaEIsU0FBUyxFQUFFOzRCQUNUO2dDQUNFLGNBQWMsRUFBRTtvQ0FDZCxVQUFVLEVBQUUsUUFBUTtvQ0FDcEIsVUFBVSxFQUFFLElBQUk7aUNBQ2pCO2dDQUNELGVBQWUsRUFBRTtvQ0FDZixTQUFTLEVBQUUsTUFBTTtpQ0FDbEI7Z0NBQ0QsTUFBTSxFQUFFLFdBQVc7NkJBQ3BCO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsd0RBQXdELENBQUMsSUFBVTtZQUNqRSxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLGVBQWUsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUV2QixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGtCQUFrQixFQUFFO2dCQUNwRCxnQkFBZ0IsRUFBRTtvQkFDaEIsV0FBVyxFQUFFO3dCQUNYOzRCQUNFLFFBQVEsRUFBRSxPQUFPOzRCQUNqQixRQUFRLEVBQUU7Z0NBQ1IsZUFBZTtnQ0FDZixlQUFlO2dDQUNmLFVBQVU7Z0NBQ1Ysa0JBQWtCO2dDQUNsQixlQUFlO2dDQUNmLFdBQVc7NkJBQ1o7eUJBQ0Y7d0JBQ0QsRUFBRTt3QkFDRjs0QkFDRSxRQUFRLEVBQUUsT0FBTzs0QkFDakIsUUFBUSxFQUFFLGdCQUFnQjt5QkFDM0I7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw0QkFBNEIsQ0FBQyxJQUFVO1lBQ3JDLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsZUFBZSxDQUFDLEtBQUssRUFBRTtnQkFDckIsYUFBYSxFQUFFLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxpQkFBaUI7YUFDeEQsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyw2QkFBNkIsRUFBRTtnQkFDL0QsUUFBUSxFQUFFO29CQUNSLEVBQUU7b0JBQ0Y7d0JBQ0UsU0FBUyxFQUFFOzRCQUNUO2dDQUNFLGVBQWUsRUFBRTtvQ0FDZixXQUFXLEVBQUUsbUJBQW1CO2lDQUNqQzs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELGlDQUFpQyxDQUFDLElBQVU7WUFDMUMsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixlQUFlLENBQUMsS0FBSyxFQUFFO2dCQUNyQixZQUFZLEVBQUU7b0JBQ1osU0FBUyxDQUFDLFlBQVksQ0FBQyxTQUFTLEVBQUU7b0JBQ2xDLFNBQVMsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLGVBQVEsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7aUJBQ2xEO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyw2QkFBNkIsRUFBRTtnQkFDL0QsUUFBUSxFQUFFO29CQUNSLEVBQUU7b0JBQ0Y7d0JBQ0UsU0FBUyxFQUFFOzRCQUNUO2dDQUNFLGVBQWUsRUFBRTtvQ0FDZixjQUFjLEVBQUUsd0JBQXdCO2lDQUN6Qzs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELHFEQUFxRCxDQUFDLElBQVU7WUFDOUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixlQUFlLENBQUMsS0FBSyxFQUFFO2dCQUNyQixTQUFTLEVBQUUsUUFBUTthQUNwQixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDZCQUE2QixFQUFFO2dCQUMvRCxRQUFRLEVBQUU7b0JBQ1IsRUFBRTtvQkFDRjt3QkFDRSxTQUFTLEVBQUU7NEJBQ1Q7Z0NBQ0UsZUFBZSxFQUFFO29DQUNmLFdBQVcsRUFBRSxRQUFRO2lDQUN0Qjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlTGlrZSB9IGZyb20gXCJAYXdzLWNkay9hc3NlcnRcIjtcbmltcG9ydCAqIGFzIGNvZGVwaXBlbGluZSBmcm9tICdAYXdzLWNkay9hd3MtY29kZXBpcGVsaW5lJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgeyBEdXJhdGlvbiwgU2VjcmV0VmFsdWUsIFN0YWNrIH0gZnJvbSBcIkBhd3MtY2RrL2NvcmVcIjtcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgKiBhcyBjcGFjdGlvbnMgZnJvbSAnLi4vLi4vbGliJztcblxuLy8gdHNsaW50OmRpc2FibGU6b2JqZWN0LWxpdGVyYWwta2V5LXF1b3Rlc1xuXG5leHBvcnQgPSB7XG4gICdTMyBEZXBsb3kgQWN0aW9uJzoge1xuICAgICdieSBkZWZhdWx0IGV4dHJhY3QgYXJ0aWZhY3RzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgbWluaW1hbFBpcGVsaW5lKHN0YWNrKTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVQaXBlbGluZTo6UGlwZWxpbmUnLCB7XG4gICAgICAgIFwiU3RhZ2VzXCI6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBcIk5hbWVcIjogXCJTb3VyY2VcIixcbiAgICAgICAgICAgIFwiQWN0aW9uc1wiOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBcIk5hbWVcIjogXCJTb3VyY2VcIixcbiAgICAgICAgICAgICAgICBcIkFjdGlvblR5cGVJZFwiOiB7XG4gICAgICAgICAgICAgICAgICBcIkNhdGVnb3J5XCI6IFwiU291cmNlXCIsXG4gICAgICAgICAgICAgICAgICBcIk93bmVyXCI6IFwiVGhpcmRQYXJ0eVwiXG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXVxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgXCJOYW1lXCI6IFwiRGVwbG95XCIsXG4gICAgICAgICAgICBcIkFjdGlvbnNcIjogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgXCJBY3Rpb25UeXBlSWRcIjoge1xuICAgICAgICAgICAgICAgICAgXCJDYXRlZ29yeVwiOiBcIkRlcGxveVwiLFxuICAgICAgICAgICAgICAgICAgXCJQcm92aWRlclwiOiBcIlMzXCIsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBcIkNvbmZpZ3VyYXRpb25cIjoge1xuICAgICAgICAgICAgICAgICAgXCJFeHRyYWN0XCI6IFwidHJ1ZVwiXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBcIk5hbWVcIjogXCJDb3B5RmlsZXNcIlxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2dyYW50IHRoZSBwaXBlbGluZSBjb3JyZWN0IGFjY2VzcyB0byB0aGUgdGFyZ2V0IGJ1Y2tldCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIG1pbmltYWxQaXBlbGluZShzdGFjayk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFwiQWN0aW9uXCI6IFtcbiAgICAgICAgICAgICAgICBcInMzOkdldE9iamVjdCpcIixcbiAgICAgICAgICAgICAgICBcInMzOkdldEJ1Y2tldCpcIixcbiAgICAgICAgICAgICAgICBcInMzOkxpc3QqXCIsXG4gICAgICAgICAgICAgICAgXCJzMzpEZWxldGVPYmplY3QqXCIsXG4gICAgICAgICAgICAgICAgXCJzMzpQdXRPYmplY3QqXCIsXG4gICAgICAgICAgICAgICAgXCJzMzpBYm9ydCpcIlxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge30sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJBY3Rpb25cIjogXCJzdHM6QXNzdW1lUm9sZVwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXVxuICAgICAgICB9XG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAna2ViYWItY2FzZSBDYW5uZWRBQ0wgdmFsdWUnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBtaW5pbWFsUGlwZWxpbmUoc3RhY2ssIHtcbiAgICAgICAgYWNjZXNzQ29udHJvbDogczMuQnVja2V0QWNjZXNzQ29udHJvbC5QVUJMSUNfUkVBRF9XUklURVxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2RlUGlwZWxpbmU6OlBpcGVsaW5lJywge1xuICAgICAgICBcIlN0YWdlc1wiOiBbXG4gICAgICAgICAge30sXG4gICAgICAgICAge1xuICAgICAgICAgICAgXCJBY3Rpb25zXCI6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFwiQ29uZmlndXJhdGlvblwiOiB7XG4gICAgICAgICAgICAgICAgICBcIkNhbm5lZEFDTFwiOiBcInB1YmxpYy1yZWFkLXdyaXRlXCJcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIF1cbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdhbGxvdyBjdXN0b21pemluZyBjYWNoZS1jb250cm9sJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgbWluaW1hbFBpcGVsaW5lKHN0YWNrLCB7XG4gICAgICAgIGNhY2hlQ29udHJvbDogW1xuICAgICAgICAgIGNwYWN0aW9ucy5DYWNoZUNvbnRyb2wuc2V0UHVibGljKCksXG4gICAgICAgICAgY3BhY3Rpb25zLkNhY2hlQ29udHJvbC5tYXhBZ2UoRHVyYXRpb24uaG91cnMoMTIpKSxcbiAgICAgICAgXVxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2RlUGlwZWxpbmU6OlBpcGVsaW5lJywge1xuICAgICAgICBcIlN0YWdlc1wiOiBbXG4gICAgICAgICAge30sXG4gICAgICAgICAge1xuICAgICAgICAgICAgXCJBY3Rpb25zXCI6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFwiQ29uZmlndXJhdGlvblwiOiB7XG4gICAgICAgICAgICAgICAgICBcIkNhY2hlQ29udHJvbFwiOiBcInB1YmxpYywgbWF4LWFnZTogNDMyMDBcIlxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgXVxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2FsbG93IGN1c3RvbWl6aW5nIG9iamVjdEtleSAoZGVwbG95bWVudCBwYXRoIG9uIFMzKScodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIG1pbmltYWxQaXBlbGluZShzdGFjaywge1xuICAgICAgICBvYmplY3RLZXk6IFwiL2EvYi9jXCJcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q29kZVBpcGVsaW5lOjpQaXBlbGluZScsIHtcbiAgICAgICAgXCJTdGFnZXNcIjogW1xuICAgICAgICAgIHt9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiQWN0aW9uc1wiOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBcIkNvbmZpZ3VyYXRpb25cIjoge1xuICAgICAgICAgICAgICAgICAgXCJPYmplY3RLZXlcIjogXCIvYS9iL2NcIlxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgXVxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9XG4gIH1cbn07XG5cbmludGVyZmFjZSBNaW5pbWFsUGlwZWxpbmVPcHRpb25zIHtcbiAgcmVhZG9ubHkgYWNjZXNzQ29udHJvbD86IHMzLkJ1Y2tldEFjY2Vzc0NvbnRyb2w7XG4gIHJlYWRvbmx5IGJ1Y2tldD86IHMzLklCdWNrZXQ7XG4gIHJlYWRvbmx5IGNhY2hlQ29udHJvbD86IGNwYWN0aW9ucy5DYWNoZUNvbnRyb2xbXTtcbiAgcmVhZG9ubHkgZXh0cmFjdD86IGJvb2xlYW47XG4gIHJlYWRvbmx5IG9iamVjdEtleT86IHN0cmluZztcbn1cblxuZnVuY3Rpb24gbWluaW1hbFBpcGVsaW5lKHN0YWNrOiBTdGFjaywgb3B0aW9uczogTWluaW1hbFBpcGVsaW5lT3B0aW9ucyA9IHt9KTogY29kZXBpcGVsaW5lLklTdGFnZSB7XG4gIGNvbnN0IHNvdXJjZU91dHB1dCA9IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoKTtcbiAgY29uc3Qgc291cmNlQWN0aW9uID0gbmV3IGNwYWN0aW9ucy5HaXRIdWJTb3VyY2VBY3Rpb24oe1xuICAgIGFjdGlvbk5hbWU6ICdTb3VyY2UnLFxuICAgIG93bmVyOiAnYXdzJyxcbiAgICByZXBvOiAnYXdzLWNkaycsXG4gICAgb3V0cHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgb2F1dGhUb2tlbjogU2VjcmV0VmFsdWUucGxhaW5UZXh0KCdzZWNyZXQnKSxcbiAgfSk7XG5cbiAgY29uc3QgcGlwZWxpbmUgPSBuZXcgY29kZXBpcGVsaW5lLlBpcGVsaW5lKHN0YWNrLCAnTXlQaXBlbGluZScsIHtcbiAgICBzdGFnZXM6IFtcbiAgICAgIHtcbiAgICAgICAgc3RhZ2VOYW1lOiAnU291cmNlJyxcbiAgICAgICAgYWN0aW9uczogWyBzb3VyY2VBY3Rpb24gXSxcbiAgICAgIH1cbiAgICBdXG4gIH0pO1xuXG4gIGNvbnN0IGRlcGxveW1lbnRTdGFnZSA9IHBpcGVsaW5lLmFkZFN0YWdlKHtcbiAgICBzdGFnZU5hbWU6ICdEZXBsb3knLFxuICAgIGFjdGlvbnM6IFtcbiAgICAgIG5ldyBjcGFjdGlvbnMuUzNEZXBsb3lBY3Rpb24oe1xuICAgICAgICBhY2Nlc3NDb250cm9sOiBvcHRpb25zLmFjY2Vzc0NvbnRyb2wsXG4gICAgICAgIGFjdGlvbk5hbWU6ICdDb3B5RmlsZXMnLFxuICAgICAgICBidWNrZXQ6IG9wdGlvbnMuYnVja2V0IHx8IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdNeUJ1Y2tldCcpLFxuICAgICAgICBjYWNoZUNvbnRyb2w6IG9wdGlvbnMuY2FjaGVDb250cm9sLFxuICAgICAgICBleHRyYWN0OiBvcHRpb25zLmV4dHJhY3QsXG4gICAgICAgIGlucHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgICAgIG9iamVjdEtleTogb3B0aW9ucy5vYmplY3RLZXksXG4gICAgICB9KSxcbiAgICBdLFxuICB9KTtcblxuICByZXR1cm4gZGVwbG95bWVudFN0YWdlO1xufVxuIl19