"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * CodePipeline Action to deploy an ECS Service.
 */
class EcsDeployAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'ECS',
            artifactBounds: common_1.deployArtifactBounds(),
            inputs: [determineInputArtifact(props)],
            resource: props.service
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        // permissions based on CodePipeline documentation:
        // https://docs.aws.amazon.com/codepipeline/latest/userguide/how-to-custom-role.html#how-to-update-role-new-services
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DescribeServices',
                'ecs:DescribeTaskDefinition',
                'ecs:DescribeTasks',
                'ecs:ListTasks',
                'ecs:RegisterTaskDefinition',
                'ecs:UpdateService',
            ],
            resources: ['*']
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: ['*'],
            conditions: {
                StringEqualsIfExists: {
                    'iam:PassedToService': [
                        'ec2.amazonaws.com',
                        'ecs-tasks.amazonaws.com',
                    ],
                }
            }
        }));
        options.bucket.grantRead(options.role);
        return {
            configuration: {
                ClusterName: this.props.service.cluster.clusterName,
                ServiceName: this.props.service.serviceName,
                FileName: this.props.imageFile && this.props.imageFile.fileName,
            },
        };
    }
}
exports.EcsDeployAction = EcsDeployAction;
function determineInputArtifact(props) {
    if (props.imageFile && props.input) {
        throw new Error("Exactly one of 'input' or 'imageFile' can be provided in the ECS deploy Action");
    }
    if (props.imageFile) {
        return props.imageFile.artifact;
    }
    if (props.input) {
        return props.input;
    }
    throw new Error("Specifying one of 'input' or 'imageFile' is required for the ECS deploy Action");
}
//# sourceMappingURL=data:application/json;base64,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