# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fractal_analysis',
 'fractal_analysis.estimator',
 'fractal_analysis.simulator',
 'fractal_analysis.tester']

package_data = \
{'': ['*']}

install_requires = \
['pandas>=1.4.0,<2.0.0', 'scipy>=1.8.0,<2.0.0']

setup_kwargs = {
    'name': 'fractal-analysis',
    'version': '0.1.13',
    'description': '',
    'long_description': '# Fractal Analysis\nFractal and multifractal methods, including \n\n- fractional Brownian motion (FBM) tester\n- multifractional Brownian motion (MBM) tester\n- IR hurst exponents estimator of multifractional Brownian motion (MBM)\n- QV hurst exponents estimator of multifractional Brownian motion (MBM)\n\n## FBM / MBM tester\nTest if a series is FBM (MBM) given the hurst parameter (hurst exponents series).\nThe implementation is based on the following papers:\n\n>Michał Balcerek, Krzysztof Burnecki. (2020)  \nTesting of fractional Brownian motion in a noisy environment.  \nChaos, Solitons & Fractals, Volume 140, 110097.  \nhttps://doi.org/10.1016/j.chaos.2020.110097\n\n>Balcerek, Michał, and Krzysztof Burnecki. (2020)  \nTesting of Multifractional Brownian Motion. Entropy 22, no. 12: 1403.  \nhttps://doi.org/10.3390/e22121403 \n\nWe added the following improvements to the FBM and/or MBM tester:\n- option for automatically estimating sigma \n  - based on Theorem 2.3 of the following paper:\n    >Ayache A., Peng Q. (2012)  \n     Stochastic Volatility and Multifractional Brownian Motion.  \n     In: Zili M., Filatova D. (eds) Stochastic Differential Equations and Processes. \n     Springer Proceedings in Mathematics, vol 7. Springer, Berlin, Heidelberg.  \n     https://doi.org/10.1007/978-3-642-22368-6_6\n  - a detailed introduction can be found in the section 5 of the following paper:\n    > todo: add paper name\n- option for testing if the series itself is a FBM (MBM)\n- option for testing if the increment of the series is the increment of a FBM (MBM)\n- option for testing if the series is a FBM (MBM) with an add-on noise\n- option for testing if the increment of the series is the increment of a FBM (MBM) with an add-on noise\n\n## IR / QV hurst estimator of MBM\nEstimate the hurst parameter (hurst exponent series) of a MBM.\nThe implementation is based on the following paper:\n\n>Bardet, Jean-Marc & Surgailis, Donatas, 2013.  \nNonparametric estimation of the local Hurst function of multifractional Gaussian processes.  \nStochastic Processes and their Applications, Elsevier, vol. 123(3), pages 1004-1045.\n\n\nBardet, the author in the above paper, provides a Matlab code that can be found at: \n>http://samm.univ-paris1.fr/Sofwares-Logiciels  \nSoftware for estimating the Hurst function H of a Multifractional Brownian Motion:\n Quadratic Variation estimator and IR estimator\n\n\n## To install\nTo get started, simply do:\n```\npip install fractal-analysis\n```\nor check out the code from out GitHub repository.\n\nYou can now use the series tester module in Python with:\n```\nfrom fractal_analysis import tester\n```\nor use the hurst estimator with\n```\nfrom fractal_analysis import estimator\n```\n\n## Examples\n### FBM / MBM tester\nImport:\n```\nfrom fractal_analysis.tester.series_tester import MBMSeriesTester, FBMSeriesTester\nfrom fractal_analysis.tester.critical_surface import CriticalSurfaceFBM, CriticalSurfaceMFBM\n```\nTo test if a series ```series``` is FBM, one needs to use ```CriticalSurfaceFBM``` with length of the series ```N```,\nthe significance level ```alpha``` (look at quantiles of order ```alpha/2``` and ```1 − alpha/2```), and  choose to test\non the series itself or its increment series using ```is_increment_series``` (default is ```False```, meaning to test on\nthe series itself),\n```\nfbm_tester = FBMSeriesTester(critical_surface=CriticalSurfaceFBM(N=N, alpha=0.05, is_increment_series=False))\n```\n\nTo test if the series is FBM with hurst parameter 0.3 and use auto estimated sigma square (set ```sig2=None```):\n\n```\nis_fbm, sig2 = fbm_tester.test(h=0.3, x=series, sig2=None, add_on_sig2=0)\n```\nIf the output contains, for example:\n> Bad auto sigma square calculated with error 6.239236333681868. Suggest to give sigma square and rerun.\n\nThe auto sigma square estimated is not accurate. One may want to manually choose a sigma square and rerun. For example:\n```\nis_fbm, sig2 = fbm_tester.test(h=0.3, x=series, sig2=1, add_on_sig2=0)\n```\nIf one wants to test with an add-no noise, change the value of ```add_on_sig2```.\n\n\n\n\nTo test if the series is MBM, one needs to use ```CriticalSurfaceMFBM``` with length of the series ```N```\nand the significance level ```alpha``` (look at quantiles of order ```alpha/2``` and ```1 − alpha/2```) \n```\nmbm_tester = MBMSeriesTester(critical_surface=CriticalSurfaceMFBM(N=N, alpha=0.05, is_increment_series=False))\n```\nTo test if the series is MBM with a given holder exponent series ```h_mbm_series``` and use auto estimated sigma square:\n```\nis_mbm, sig2 = mbm_tester.test(h=h_mbm_series, x=series, sig2=None, add_on_sig2=0)\n```\nBe aware that ```MBMSeriesTester``` requires ```len(h_mbm_series)==len(series)```.\n\n#### Use of cache\nUse caching to speed up the testing process. If the series ```x``` for testing is unchanged and multiple ```h``` \nand/or ```sig2``` are used, one may want to set \n```is_cache_stat=True``` to allow cache variable ```stat```. If ```h``` and ```sig2``` are unchanged and multiple ```x```\nare used, one may want to set ```is_cache_quantile=True``` to allow cache variable ```quantile```. For example:\n```\nmbm_tester = MBMSeriesTester(critical_surface=CriticalSurfaceMFBM(N=N, alpha=0.05), is_cache_stat=True, is_cache_quantile=False)\n```\n\n### IR / QV hurst estimator of MBM\nImport:\n```\nfrom fractal_analysis.estimator.hurst_estimator import IrHurstEstimator, QvHurstEstimator\nimport numpy as np\nimport math\n```\nGenerate a standard brownian motion\n```\nN = 100\nseries = np.random.randn(N) * 0.5 * math.sqrt(1 / N)\nseries = np.cumsum(series)\n```\nTo estimate the hurst exponents series of the above series with ```alpha=0.2``` using IR estimator,\n```\nestimator = IrHurstEstimator(mbm_series=series, alpha=0.2)\nprint(estimator.holder_exponents)\n```\nTo estimate the hurst exponents series of the above series with ```alpha=0.2``` using QV estimator,\n```\nestimator = QvHurstEstimator(mbm_series=series, alpha=0.2)\nprint(estimator.holder_exponents)\n```\nHere the value of ```alpha``` decides how many observations on the ```mbm_series``` is used to estimate a point of the\nholder exponent; small alpha means more observations are used for a single point and therefore the variance is small.\n   \n',
    'author': 'yujiading',
    'author_email': 'yujia.ding@cgu.edu',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/yujiading/fractals',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<3.11',
}


setup(**setup_kwargs)
