# This file is part of fedmsg.
# Copyright (C) 2012-2014 Red Hat, Inc.
#
# fedmsg is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# fedmsg is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with fedmsg; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
#
# Authors:  Ralph Bean <rbean@redhat.com>
#
""" Tests for fedmsg.meta """

import os
import unittest

from fedmsg_meta_fedora_infrastructure.tests.base import Base

from fedmsg_meta_fedora_infrastructure.tests.common import add_doc


full_irc_logs = """======================================
#fedora-meeting-1: RELENG (2015-02-09)
======================================


Meeting started by dgilmore at 16:31:17 UTC. The full logs are available
at
http://meetbot.fedoraproject.org/fedora-meeting-1/2015-02-09/releng.2015-02-09-16.31.log.html
.



Meeting summary
---------------
* init process  (dgilmore, 16:31:30)

* Secondary Architectures updates  (dgilmore, 16:33:42)

* Secondary Architectures update - ppc  (dgilmore, 16:33:43)

* Secondary Architectures update - s390  (dgilmore, 16:36:35)

* Secondary Architectures update - arm  (dgilmore, 16:37:12)

* #6016 Use fedpkg-minimal in Fedora buildroots  (dgilmore, 16:39:22)
  * LINK: https://fedorahosted.org/rel-eng/ticket/6016   (dgilmore,
    16:39:38)
  * LINK: https://koji.fedoraproject.org/koji/buildinfo?buildID=609961
    (bochecha, 16:40:40)

* #5959 Enable keep-alive connections for koji (primary and secondaries)
  (dgilmore, 16:41:44)
  * LINK: https://fedorahosted.org/rel-eng/ticket/5959   (dgilmore,
    16:41:56)
  * ACTION: pbrobinson to send patch to infra list enabling keepalive
    (dgilmore, 16:43:08)

* #6023 allow Peter Robinson to restart sigul bridges  (dgilmore,
  16:43:39)
  * LINK: https://fedorahosted.org/rel-eng/ticket/6023   (dgilmore,
    16:43:48)

* #6027 secondary arch old mash trees cleanup  (dgilmore, 16:45:31)
  * LINK: https://fedorahosted.org/rel-eng/ticket/6027   (dgilmore,
    16:45:47)

* #6096 add individual email addressses to Fedoras GPG keys  (dgilmore,
  16:52:06)
  * LINK: https://fedorahosted.org/rel-eng/ticket/6096   (dgilmore,
    16:52:20)
  * ACTION: dgilmore will make a choice when making the keys  (dgilmore,
    16:58:51)

* #6100 create bsd-style checksum files for CHECKSUM files  (dgilmore,
  16:59:11)
  * LINK: https://fedorahosted.org/rel-eng/ticket/6100   (dgilmore,
    16:59:22)
  * ACTION: dgilmore to make this happen  (dgilmore, 17:03:13)

* open floor  (dgilmore, 17:03:21)

Meeting ended at 17:58:27 UTC.




Action Items
------------
* pbrobinson to send patch to infra list enabling keepalive
* dgilmore will make a choice when making the keys
* dgilmore to make this happen




Action Items, by person
-----------------------
* dgilmore
  * dgilmore will make a choice when making the keys
  * dgilmore to make this happen
* pbrobinson
  * pbrobinson to send patch to infra list enabling keepalive
* **UNASSIGNED**
  * (none)




People Present (lines said)
---------------------------
* dgilmore (116)
* bochecha (33)
* tyll (16)
* pbrobinson (16)
* zodbot (5)
* janeznemanic (1)
* masta (1)
* nirik (0)
* sharkcz (0)




Generated by `MeetBot`_ 0.1.4

.. _`MeetBot`: http://wiki.debian.org/MeetBot
"""

class TestSupybotStartMeetingNoName(Base):
    """ Trusty old `zodbot <https://meetbot.fedoraproject.org/>`_ publishes
    messages too!  Messages on this topic get published (somewhat obviously)
    when a new IRC meeting is started.  The user starting the meeting may
    specify a meeting title, but doesn't have to.  Here's an example
    message with no meeting title specified:
    """
    expected_title = "meetbot.meeting.start"
    expected_subti = 'ralph started a meeting in #channel'
    expected_icon = 'https://apps.fedoraproject.org/img/icons/meetbot.png'
    expected_secondary_icon = "https://seccdn.libravatar.org/avatar/" + \
        "9c9f7784935381befc302fe3c814f9136e7a33953d0318761669b8643f4df55c?s=64&d=retro"
    expected_usernames = set(['ralph'])
    expected_objects = set([
        'attendees/ralph',
        'channels/#channel',
    ])

    msg = {
        "i": 16,
        "msg": {
            "meeting_topic": None,
            "attendees": {
                "zodbot": 2,
                "threebean": 2
            },
            "chairs": {},
            "url": "http://logs.com/awesome",
            "owner": "threebean",
            "channel": "#channel"
        },
        "topic": "org.fedoraproject.dev.meetbot.meeting.start",
        "timestamp": 1345572862.556145
    }


class TestSupybotStartMeeting(Base):
    """ Trusty old `zodbot <https://meetbot.fedoraproject.org/>`_ publishes
    messages too!  Messages on this topic get published (somewhat obviously)
    when a new IRC meeting is started.  The user starting the meeting may
    specify a meeting title, but doesn't have to.  Here's an example
    message with a specified meeting title:
    """
    expected_title = "meetbot.meeting.start"
    expected_subti = 'ralph started meeting "title" in #channel'
    expected_usernames = set(['ralph'])
    expected_objects = set([
        'attendees/ralph',
        'channels/#channel',
        'titles/title',
    ])

    msg = {
        "i": 16,
        "msg": {
            "meeting_topic": "title",
            "attendees": {
                "zodbot": 2,
                "threebean": 2
            },
            "chairs": {},
            "url": "http://logs.com/awesome",
            "owner": "threebean",
            "channel": "#channel"
        },
        "topic": "org.fedoraproject.dev.meetbot.meeting.start",
        "timestamp": 1345572862.556145
    }


class TestSupybotEndMeeting(Base):
    """ Trusty old `zodbot <https://meetbot.fedoraproject.org/>`_ publishes
    messages too!  Messages on this topic get published when an IRC meeting
    ends.  Meetings may or may not have a title (which can be tricky).
    Here's an example message where the title is specified:
    """
    expected_title = "meetbot.meeting.complete"
    expected_subti = 'dgilmore\'s meeting titled "RELENG (2015-02-09)" ' + \
        'ended in #fedora-meeting-1'
    expected_link = 'https://meetbot.fedoraproject.org/fedora-meeting-1/' + \
        '2015-02-09/releng.2015-02-09-16.31.html'
    expected_usernames = set([
        'masta',
        'nirik',
        'tyll',
        'bochecha',
        'pbrobinson',
        'sharkcz',
        'janeznemanic',
        'dgilmore',
    ])
    expected_objects = set([
        'attendees/masta',
        'attendees/nirik',
        'attendees/tyll',
        'attendees/bochecha',
        'attendees/pbrobinson',
        'attendees/sharkcz',
        'attendees/janeznemanic',
        'attendees/dgilmore',
        'channels/#fedora-meeting-1',
        'titles/RELENG (2015-02-09)',
        'topics/open floor',
    ])

    msg = {
        "source_name": "datanommer",
        "i": 94,
        "timestamp": 1423504707.0,
        "msg_id": "2015-ff043996-5aaa-471b-bc87-b8551f7fd3b6",
        "topic": "org.fedoraproject.prod.meetbot.meeting.complete",
        "source_version": "0.6.4",
        "msg": {
            "meeting_topic": "RELENG (2015-02-09)",
            "attendees": {
                "masta": 1,
                "nirik": 0,
                "tyll": 16,
                "zodbot": 5,
                "bochecha": 33,
                "pbrobinson": 16,
                "sharkcz": 0,
                "janeznemanic": 1,
                "dgilmore": 116
            },
            "chairs": {
                "masta": True,
                "nirik": True,
                "tyll": True,
                "bochecha": True,
                "pbrobinson": True,
                "dgilmore": True,
                "sharkcz": True
            },
            "topic": "open floor",
            "url": "http://meetbot.fedoraproject.org/fedora-meeting-1/2015-02-09/releng.2015-02-09-16.31",
            "owner": "dgilmore",
            "channel": "#fedora-meeting-1"
        }
    }


class TestSupybotEndMeetingNoTitle(Base):
    """ Trusty old `zodbot <https://meetbot.fedoraproject.org/>`_ publishes
    messages too!  Messages on this topic get published when an IRC meeting
    ends.  Meetings may or may not have a title (which can be tricky).
    Here's an example message where the title is **not** specified:
    """
    expected_title = "meetbot.meeting.complete"
    expected_subti = 'ralph\'s meeting ended in #channel'
    expected_link = 'https://logs.com/awesome.html'
    expected_usernames = set(['ralph'])
    expected_objects = set([
        'attendees/ralph',
        'channels/#channel',
    ])

    msg = {
        "i": 16,
        "msg": {
            "meeting_topic": None,
            "attendees": {
                "zodbot": 2,
                "threebean": 2
            },
            "chairs": {},
            "url": "http://logs.com/awesome",
            "owner": "threebean",
            "channel": "#channel"
        },
        "topic": "org.fedoraproject.dev.meetbot.meeting.complete",
        "timestamp": 1345572862.556145
    }


class TestSupybotChangeTopic(Base):
    """ As IRC meetings chug along, the chairperson may change the meeting;
    zodbot publishes message for that!  An example **with** a title specified:
    """
    expected_title = "meetbot.meeting.topic.update"
    expected_subti = 'The topic of ralph\'s "title" meeting changed ' +\
        'to "Food" in #channel'
    expected_usernames = set(['ralph'])
    expected_objects = set([
        'attendees/ralph',
        'channels/#channel',
        'titles/title',
        'topics/Food',
    ])

    msg = {
        "i": 16,
        "msg": {
            "meeting_topic": "title",
            "attendees": {
                "zodbot": 2,
                "threebean": 2
            },
            "chairs": {},
            "url": "http://logs.com/awesome",
            "owner": "threebean",
            "channel": "#channel",
            "topic": "Food",
        },
        "topic": "org.fedoraproject.dev.meetbot.meeting.topic.update",
        "timestamp": 1345572862.556145
    }


class TestSupybotChangeTopicNoTitle(Base):
    """ As IRC meetings chug along, the chairperson may change the meeting;
    zodbot publishes message for that!  An example **without** a title
    specified:
    """
    expected_title = "meetbot.meeting.topic.update"
    expected_subti = 'The topic of ralph\'s meeting changed ' +\
        'to "Food" in #channel'
    expected_usernames = set(['ralph'])
    expected_objects = set([
        'attendees/ralph',
        'channels/#channel',
        'topics/Food'
    ])

    msg = {
        "i": 16,
        "msg": {
            "meeting_topic": None,
            "attendees": {
                "zodbot": 2,
                "threebean": 2
            },
            "chairs": {},
            "url": "http://logs.com/awesome",
            "owner": "threebean",
            "channel": "#channel",
            "topic": "Food",
        },
        "topic": "org.fedoraproject.dev.meetbot.meeting.topic.update",
        "timestamp": 1345572862.556145
    }


class TestSupybotMeetingItemHelp(Base):
    """ As IRC meetings chug along, people can declare items for the record.
    Here's an example of someone **calling for help** on some item:
    """
    expected_title = "meetbot.meeting.item.help"
    expected_subti = 'ralph called for help in the "fancytown" meeting in ' + \
        '#fedora-meeting: "The #fedora-apps team needs people to help ' + \
        'review code http://ambre.pingoured.fr/fedora-infra/"'
    expected_secondary_icon = 'https://seccdn.libravatar.org/avatar/' + \
        '9c9f7784935381befc302fe3c814f9136e7a33953d0318761669b8643f4df55c' + \
        '?s=64&d=retro'
    expected_usernames = set(['ralph', 'lmacken'])
    expected_objects = set([
        'attendees/ralph',
        'attendees/lmacken',
        'channels/#fedora-meeting',
        'titles/fancytown',
    ])

    msg = {
        "username": "threebean",
        "timestamp": 1426553001,
        "msg_id": "2015-ab5c6d46-4010-4966-9d0e-dd5f7ae819a5",
        "topic": "org.fedoraproject.prod.meetbot.meeting.item.help",
        "msg": {
            "meeting_topic": "fancytown",
            "attendees": {
                "lmacken": 2,
                "threebean": 2,
                "zodbot": 2
            },
            "chairs": {
                "threebean": True,
                "lmacken": True,
            },
            "topic": "",
            "url": "threebean/2015/threebean.2015-03-17-00.42",
            "details": {
                "linenum": 4,
                "line": "The #fedora-apps team needs people to help "
                "review code http://ambre.pingoured.fr/fedora-infra/",
                "time_": 1426553001.0,
                "nick": "threebean",
            },
            "owner": "lmacken",
            "channel": "#fedora-meeting"
        }
    }


class TestSupybotMeetingItemLink(Base):
    """ As IRC meetings chug along, people can declare items for the record.
    Here's an example of someone **linking to information** on some item:
    """
    expected_title = "meetbot.meeting.item.link"
    expected_subti = (
        'mizmo linked to more information in the "Server SIG Weekly Meeting '
        '(2015-06-23)" meeting in #fedora-meeting-1: "https://bugzilla.redhat.'
        'com/show_bug.cgi?id=1234653 basically stops all media booting in most'
        ' cases"')

    expected_secondary_icon = 'https://seccdn.libravatar.org/avatar/' + \
        '2b820632db285cd534651e5f0da285d7c0a0a1bf600da16e19063163ce21db31' + \
        '?s=64&d=retro'
    expected_link = "https://bugzilla.redhat.com/show_bug.cgi?id=1234653"
    expected_usernames = set(['mizmo', 'nirik'])
    expected_objects = set([
        'attendees/mizmo',
        'attendees/nirik',
        'channels/#fedora-meeting-1',
        'titles/Server SIG Weekly Meeting (2015-06-23)',
        'topics/Open Floor',
    ])

    msg = {
        "i": 999,
        "msg": {
            "attendees": {
                "mizmo": 0,
                "nirik": 9,
                "zodbot": 5
            },
            "chairs": {
                "mizmo": True,
                "nirik": True,
            },
            "channel": "#fedora-meeting-1",
            "details": {
                "line": "https://bugzilla.redhat.com/show_bug.cgi?id=1234653 "
                "basically stops all media booting in most cases",
                "linenum": 122,
                "nick": "mizmo",
                "time_": 1435074475.0
            },
            "meeting_topic": "Server SIG Weekly Meeting (2015-06-23)",
            "owner": "nirik",
            "topic": "Open Floor",
            "url": "http://meetbot.fedoraproject.org/fedora-meeting-1/"
            "2015-06-23/fedora-meeting-1.2015-06-23-15.03"
        },
        "msg_id": "2015-2fc83ae5-2cbc-445c-85c3-266debd8df2d",
        "source_name": "datanommer",
        "source_version": "0.6.5",
        "timestamp": 1435074475.0,
        "topic": "org.fedoraproject.prod.meetbot.meeting.item.link"
    }


class TestSupybotKarma(Base):
    """ In IRC, users can give each other "karma" by saying ``user++``
    (and zodbot publishes messages for that too).

    Here's an example of that:
    """
    expected_title = 'irc.karma'
    expected_subti = 'ralph gave lmacken(36) a karma cookie in ' + \
        '#fedora-apps.  "thanks for all the fish, lmacken++"'
    expected_icon = "https://seccdn.libravatar.org/avatar/" + \
        "9c9f7784935381befc302fe3c814f9136e7a33953d0318761669b8643f4df55c" + \
        "?s=64&d=retro"
    expected_secondary_icon = "https://seccdn.libravatar.org/avatar/" + \
        "203f6cb95b44b5d38aa21425b066dd522d3e19d8919cf4b339f29e0ea7f03e9b" + \
        "?s=64&d=retro"
    expected_link = 'https://badges.fedoraproject.org/user/lmacken'
    expected_usernames = set(['ralph', 'lmacken'])
    expected_objects = set([
        'karma/lmacken',
    ])

    msg = {
        "i": 16,
        "msg": {
            'agent': 'ralph',
            'recipient': 'lmacken',
            'total': 36,
            'vote': 1,
            'channel': '#fedora-apps',
            'line': 'thanks for all the fish, lmacken++',
        },
        "topic": "org.fedoraproject.prod.irc.karma",
        "timestamp": 1345572862.556145
    }


if not 'FEDMSG_META_NO_NETWORK' in os.environ:
    TestSupybotEndMeeting.expected_long_form = full_irc_logs


add_doc(locals())

if __name__ == '__main__':
    unittest.main()
