#!/usr/bin/env python
# -*- coding: utf-8 -*--

# Copyright (c) 2023 Oracle and/or its affiliates.
# Licensed under the Universal Permissive License v 1.0 as shown at https://oss.oracle.com/licenses/upl/

import logging
from typing import Any, Dict, List, Optional

import requests
from langchain.callbacks.manager import CallbackManagerForLLMRun
from langchain.pydantic_v1 import root_validator
from langchain.utils import get_from_dict_or_env
from oci.auth import signers

from ads.llm.langchain.plugins.base import BaseLLM
from ads.llm.langchain.plugins.contant import (
    DEFAULT_CONTENT_TYPE_JSON,
    DEFAULT_TIME_OUT,
)

logger = logging.getLogger(__name__)


class ModelDeploymentLLM(BaseLLM):
    """Base class for LLM deployed on OCI Model Deployment."""

    endpoint: str = ""
    """The uri of the endpoint from the deployed Model Deployment model."""

    best_of: int = 1
    """Generates best_of completions server-side and returns the "best"
    (the one with the highest log probability per token).
    """

    @root_validator()
    def validate_environment(  # pylint: disable=no-self-argument
        cls, values: Dict
    ) -> Dict:
        """Fetch endpoint from environment variable or arguments."""
        values["endpoint"] = get_from_dict_or_env(
            values,
            "endpoint",
            "OCI_LLM_ENDPOINT",
        )
        return values

    @property
    def _default_params(self) -> Dict[str, Any]:
        """Default parameters for the model."""
        raise NotImplementedError()

    @property
    def _identifying_params(self) -> Dict[str, Any]:
        """Get the identifying parameters."""
        return {
            **{"endpoint": self.endpoint},
            **self._default_params,
        }

    def _construct_json_body(self, prompt, params):
        """Constructs the request body as a dictionary (JSON)."""
        raise NotImplementedError

    def _invocation_params(self, stop: Optional[List[str]], **kwargs: Any) -> dict:
        """Combines the invocation parameters with default parameters."""
        params = self._default_params
        if self.stop is not None and stop is not None:
            raise ValueError("`stop` found in both the input and default params.")
        elif self.stop is not None:
            params["stop"] = self.stop
        elif stop is not None:
            params["stop"] = stop
        else:
            # Don't set "stop" in param as None. It should be a list.
            params["stop"] = []

        return {**params, **kwargs}

    def _process_response(self, response_json: dict):
        return response_json

    def _call(
        self,
        prompt: str,
        stop: Optional[List[str]] = None,
        run_manager: Optional[CallbackManagerForLLMRun] = None,
        **kwargs: Any,
    ) -> str:
        """Call out to OCI Data Science Model Deployment endpoint.

        Parameters
        ----------
        prompt (str):
            The prompt to pass into the model.
        stop (List[str], Optional):
            List of stop words to use when generating.

        Returns
        -------
            The string generated by the model.

        Example
        -------

            .. code-block:: python

                response = oci_md("Tell me a joke.")

        """
        params = self._invocation_params(stop, **kwargs)
        body = self._construct_json_body(prompt, params)
        self._print_request(prompt, params)
        response = self.send_request(data=body, endpoint=self.endpoint)
        completion = self._process_response(response)
        self._print_response(completion, response)
        return completion

    def send_request(
        self,
        data,
        endpoint: str,
        header: dict = None,
        **kwargs,
    ) -> Dict:
        """Sends request to the model deployment endpoint.

        Parameters
        ----------
        data (Json serializable):
            data need to be sent to the endpoint.
        endpoint (str):
            The model HTTP endpoint.
        header (dict, optional):
            A dictionary of HTTP headers to send to the specified url. Defaults to {}.

        Raises
        ------
        Exception:
            Raise when invoking fails.

        Returns
        -------
            A JSON representation of a requests.Response object.
        """
        if not header:
            header = {}
        header["Content-Type"] = (
            header.pop("content_type", DEFAULT_CONTENT_TYPE_JSON)
            or DEFAULT_CONTENT_TYPE_JSON
        )
        timeout = kwargs.pop("timeout", DEFAULT_TIME_OUT)
        request_kwargs = {"json": data}
        request_kwargs["headers"] = header
        signer = self.auth.get("signer")

        attempts = 0
        while attempts < 2:
            request_kwargs["auth"] = signer
            response = requests.post(
                endpoint, timeout=timeout, **request_kwargs, **kwargs
            )
            if response.status_code == 401 and self.is_principal_signer(signer):
                signer.refresh_security_token()
                attempts += 1
                continue
            break

        try:
            response.raise_for_status()
            response_json = response.json()

        except Exception:
            logger.error(
                "DEBUG INFO: request_kwargs=%s, status_code=%s, content=%s",
                request_kwargs,
                response.status_code,
                response.content,
            )
            raise

        return response_json

    @staticmethod
    def is_principal_signer(signer):
        """Checks if the signer is instance principal or resource principal signer."""
        if (
            isinstance(signer, signers.InstancePrincipalsSecurityTokenSigner)
            or isinstance(signer, signers.ResourcePrincipalsFederationSigner)
            or isinstance(signer, signers.EphemeralResourcePrincipalSigner)
            or isinstance(signer, signers.EphemeralResourcePrincipalV21Signer)
            or isinstance(signer, signers.NestedResourcePrincipals)
            or isinstance(signer, signers.OkeWorkloadIdentityResourcePrincipalSigner)
        ):
            return True
        else:
            return False


class ModelDeploymentTGI(ModelDeploymentLLM):
    """OCI Data Science Model Deployment TGI Endpoint.

    Example
    -------

        .. code-block:: python

            from ads.llm import ModelDeploymentTGI

            oci_md = ModelDeploymentTGI(endpoint="<url_of_model_deployment_endpoint>")

    """

    do_sample: bool = True
    """if set to True, this parameter enables decoding strategies such as
    multi-nominal sampling, beam-search multi-nominal sampling, Top-K sampling and Top-p sampling.
    """

    watermark = True
    """Watermarking with `A Watermark for Large Language Models <https://arxiv.org/abs/2301.10226>`_.
    Defaults to True."""

    return_full_text = False
    """Whether to prepend the prompt to the generated text. Defaults to False."""

    @property
    def _llm_type(self) -> str:
        """Return type of llm."""
        return "oci_model_deployment_tgi_endpoint"

    @property
    def _default_params(self) -> Dict[str, Any]:
        """Get the default parameters for invoking OCI model deployment TGI endpoint."""
        return {
            "best_of": self.best_of,
            "max_new_tokens": self.max_tokens,
            "temperature": self.temperature,
            "top_k": self.k
            if self.k > 0
            else None,  # `top_k` must be strictly positive'
            "top_p": self.p,
            "do_sample": self.do_sample,
            "return_full_text": self.return_full_text,
            "watermark": self.watermark,
        }

    def _construct_json_body(self, prompt, params):
        return {
            "inputs": prompt,
            "parameters": params,
        }

    def _process_response(self, response_json: dict):
        return str(response_json.get("generated_text", response_json))


class ModelDeploymentVLLM(ModelDeploymentLLM):
    """VLLM deployed on OCI Model Deployment"""

    model: str
    """Name of the model."""

    n: int = 1
    """Number of output sequences to return for the given prompt."""

    k: int = -1
    """Number of most likely tokens to consider at each step."""

    frequency_penalty: float = 0.0
    """Penalizes repeated tokens according to frequency. Between 0 and 1."""

    presence_penalty: float = 0.0
    """Penalizes repeated tokens. Between 0 and 1."""

    use_beam_search: bool = False
    """Whether to use beam search instead of sampling."""

    ignore_eos: bool = False
    """Whether to ignore the EOS token and continue generating tokens after
    the EOS token is generated."""

    logprobs: Optional[int] = None
    """Number of log probabilities to return per output token."""

    @property
    def _llm_type(self) -> str:
        """Return type of llm."""
        return "oci_model_deployment_vllm_endpoint"

    @property
    def _default_params(self) -> Dict[str, Any]:
        """Get the default parameters for calling vllm."""
        return {
            "n": self.n,
            "best_of": self.best_of,
            "max_tokens": self.max_tokens,
            "top_k": self.k,
            "top_p": self.p,
            "temperature": self.temperature,
            "presence_penalty": self.presence_penalty,
            "frequency_penalty": self.frequency_penalty,
            "stop": self.stop,
            "ignore_eos": self.ignore_eos,
            "use_beam_search": self.use_beam_search,
            "logprobs": self.logprobs,
            "model": self.model,
        }

    def _construct_json_body(self, prompt, params):
        return {
            "prompt": prompt,
            **params,
        }

    def _process_response(self, response_json: dict):
        return response_json["choices"][0]["text"]
