#!/usr/bin/env python
# -*- coding: utf-8 -*--

# Copyright (c) 2023, 2024 Oracle and/or its affiliates.
# Licensed under the Universal Permissive License v 1.0 as shown at https://oss.oracle.com/licenses/upl/
from typing import Any, Dict, List, Optional

from langchain.llms.base import LLM
from langchain.pydantic_v1 import BaseModel, Field, root_validator

from ads.common.auth import default_signer
from ads.config import COMPARTMENT_OCID


class BaseLLM(LLM):
    """Base OCI LLM class. Contains common attributes."""

    auth: dict = Field(default_factory=default_signer, exclude=True)
    """ADS auth dictionary for OCI authentication.
    This can be generated by calling `ads.common.auth.api_keys()` or `ads.common.auth.resource_principal()`.
    If this is not provided then the `ads.common.default_signer()` will be used."""

    max_tokens: int = 256
    """Denotes the number of tokens to predict per generation."""

    temperature: float = 0.2
    """A non-negative float that tunes the degree of randomness in generation."""

    k: int = 0
    """Number of most likely tokens to consider at each step."""

    p: int = 0.75
    """Total probability mass of tokens to consider at each step."""

    stop: Optional[List[str]] = None
    """Stop words to use when generating. Model output is cut off at the first occurrence of any of these substrings."""

    def _print_request(self, prompt, params):
        if self.verbose:
            print(f"LLM API Request:\n{prompt}")

    def _print_response(self, completion, response):
        if self.verbose:
            print(f"LLM API Completion:\n{completion}")

    @classmethod
    def get_lc_namespace(cls) -> List[str]:
        """Get the namespace of the LangChain object."""
        return ["ads", "llm"]

    @classmethod
    def is_lc_serializable(cls) -> bool:
        """This class can be serialized with default LangChain serialization."""
        return True


class GenerativeAiClientModel(BaseModel):
    """Base model for generative AI embedding model and LLM."""

    # This auth is the same as the auth in BaseLLM class.
    # However, this is needed for the Gen AI embedding model.
    # Do not remove this attribute
    auth: dict = Field(default_factory=default_signer, exclude=True)
    """ADS auth dictionary for OCI authentication.
    This can be generated by calling `ads.common.auth.api_keys()` or `ads.common.auth.resource_principal()`.
    If this is not provided then the `ads.common.default_signer()` will be used."""

    client: Any  #: :meta private:
    """OCI GenerativeAiClient."""

    compartment_id: str = None
    """Compartment ID of the caller."""

    endpoint_kwargs: Dict[str, Any] = {}
    """Optional attributes passed to the OCI API call."""

    client_kwargs: Dict[str, Any] = {}
    """Holds any client parameters for creating GenerativeAiClient"""

    @staticmethod
    def _import_client():
        try:
            from oci.generative_ai_inference import GenerativeAiInferenceClient
        except ImportError as ex:
            raise ImportError(
                "Could not import GenerativeAiInferenceClient from oci. "
                "The OCI SDK installed does not support generative AI service."
            ) from ex
        return GenerativeAiInferenceClient

    @root_validator()
    def validate_environment(  # pylint: disable=no-self-argument
        cls, values: Dict
    ) -> Dict:
        """Validate that python package exists in environment."""
        # Initialize client only if user does not pass in client.
        # Users may choose to initialize the OCI client by themselves and pass it into this model.
        if not values.get("client"):
            auth = values.get("auth", {})
            client_kwargs = auth.get("client_kwargs") or {}
            client_kwargs.update(values["client_kwargs"])
            # Import the GenerativeAIClient here so that there will be no error when user import ads.llm
            # and the install OCI SDK does not support generative AI service yet.
            client_class = cls._import_client()
            values["client"] = client_class(**auth, **client_kwargs)
        # Set default compartment ID
        if not values.get("compartment_id"):
            if COMPARTMENT_OCID:
                values["compartment_id"] = COMPARTMENT_OCID
            else:
                raise ValueError("Please specify compartment_id.")
        return values
