# Airtime API

The implementation is based on the [Airtime API Docs](https://docs.reloadly.com/airtime/).

## Usage

Create an `AirtimeAPI` instance by providing the Application details (client id & secret) from
the [dashboard](https://www.reloadly.com/developers/api-settings).

Some key things to keep in mind regarding the Airtime API :

* The API has 2 environments, SANDBOX (for development & testing) and LIVE. `Environment.AIRTIME` is used for the Live environmnent and `Environment.AIRTIME_SANDBOX` is for the Sandbox.
* If neither environment is specified, the SDK defaults to SANDBOX
* Each environment has a set of credentials (client id & secret) that are different from the other.<br />
    * SANBOX credentials can only be used for SANDBOX environment
    * LIVE credentials can only be used for LIVE environment
* If not environment is specified the SDK defaults to SANDBOX
* You MUST supply either the credentials, or an access token in order to call the API
  <br /><br />

As stated above, requests to the Airtime API require authentication/authorization, there are several options :

### Option 1

Set the client id & client secret; this is probably the most straight-forward or simplest way. An access token will be
acquired automatically before the API call is made.

```python
airtimeAPI = AirtimeAPI(clientId="*****", clientSecret="*****", environment=Environment.AIRTIME_SANDBOX)  
response = airtimeAPI.accounts().getBalance()
print (response)
```

### Option 2

You may alternatively acquire an access token from the
[AuthenticationAPI](https://github.com/reloadly/reloadly-sdk-python/blob/master/docs/authentication/USAGE.md)
and then set it.

```python
sample = AuthenticationAPI()
a = sample.clientCredentials(clientId="*****", clientSecret="*****" service=Service.AIRTIME_SANDBOX).getAccessToken(Service.AIRTIME_SANDBOX)

airtimeAPI = AirtimeAPI(accessToken = a, Environment.AIRTIME_SANDBOX)
response = airtimeAPI.accounts().getBalance()

print (response)
```

**Note : Access tokens obtain for Reloadly APIs have a finite
lifetime. [See the API docs](https://developers.reloadly.com/#authentication_auth_anc)**

Using the example above has some benefits and drawbacks:

#### Pro

* API requests become efficient & performant.
* Setting the access token skips the automatic token acquisition API calls that would have otherwise been made before
  each Airtime API service calls.

#### Cons

* However, because access tokens have a finite lifetime, you now have to manage or handle the expiration of the access
  token in your application code.
* In the sample above, the AirtimeAPI will continue using the same access token until it expires. Therefore, the
  responsibility false on you to handle token renewal when the token expires.

### Sample token expiration handling

```python
# Retrieve token using AuthenticationAPI
sample = AuthenticationAPI()
a = sample.clientCredentials(clientId="*****", clientSecret="*****" service=Service.AIRTIME_SANDBOX).getAccessToken(Service.AIRTIME_SANDBOX)

airtimeAPI = AirtimeAPI(accessToken = a, Environment.AIRTIME_SANDBOX)
try:
    request = airtimeAPI.accounts().getBalance()
    if response['errorCode'] == "TOKEN_EXPIRED":
        response = airtimeAPI.refreshAccessToken(request)
        return response
    else:
        # add logic 
except:
    raise Exception("ReloadlyException")
```

### Logging request & response

To enable API request/response logging, set `enablelogging` to true when intiaiting the AirtimeAPI class

```python
    airtimeAPI = AirtimeAPI(clientId="*****", clientSecret="*****", environment=Environment.AIRTIME_SANDBOX, enablelogging=True)
    .... 
```

## Customizing The API Client Instance

### Configuring Timeouts

Used to configure additional options, connect and read timeouts can be configured globally:

```python
airtimeAPI = AirtimeAPI(
    clientId="*****", 
    clientSecret="*****", 
    environment=Environment.AIRTIME_SANDBOX, 
    options = HttpOptions(readTimeout=60, writeTimeout=60, connectTimeout=60)
    )
    ....
```

### Proxy Configuration

```python
proxyPort = 8085; # Your proxy port
proxyUsername = "your-proxy-authentication-username"; # Optional proxy username if your proxy requires authentication
proxyPassword = "your-proxy-authentication-password"; # Optional proxy password if your proxy requires authentication
proxyHost = "you-proxy-host-name.com";
Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxyHost, proxyPort));

airtimeAPI = AirtimeAPI(
    clientId="*****", 
    clientSecret="*****", 
    environment=Environment.AIRTIME_SANDBOX, 
    options = HttpOptions()
    ....
    )
        ....
        
# If proxy does NOT require authentication


        ....                
```

### Request latency telemetry

By default, the library sends request latency telemetry to Reloadly. These numbers help Reloadly improve the overall
latency of its API for all users.

You can disable this behavior if you prefer:

```python
airtimeAPI = AirtimeAPI(clientId="*****", clientSecret="*****", environment=Environment.AIRTIME_SANDBOX, enableTelemetry=False)
....   
```