"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerDeploymentConfig = exports.MinimumHealthyHosts = void 0;
const cdk = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
/**
 * Minimum number of healthy hosts for a server deployment.
 */
class MinimumHealthyHosts {
    constructor(json) {
        this.json = json;
    }
    /**
     * The minimum healhty hosts threshold expressed as an absolute number.
     */
    static count(value) {
        return new MinimumHealthyHosts({
            type: 'HOST_COUNT',
            value,
        });
    }
    /**
     * The minmum healhty hosts threshold expressed as a percentage of the fleet.
     */
    static percentage(value) {
        return new MinimumHealthyHosts({
            type: 'FLEET_PERCENT',
            value,
        });
    }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.MinimumHealthyHosts = MinimumHealthyHosts;
/**
 * A custom Deployment Configuration for an EC2/on-premise Deployment Group.
 *
 * @resource AWS::CodeDeploy::DeploymentConfig
 */
class ServerDeploymentConfig extends cdk.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentConfigName,
        });
        const resource = new codedeploy_generated_1.CfnDeploymentConfig(this, 'Resource', {
            deploymentConfigName: this.physicalName,
            minimumHealthyHosts: props.minimumHealthyHosts._json,
        });
        this.deploymentConfigName = resource.ref;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
    }
    /**
     * Import a custom Deployment Configuration for an EC2/on-premise Deployment Group defined either outside the CDK app, or in a different region.
     *
     * @param scope the parent Construct for this new Construct.
     * @param id the logical ID of this new Construct.
     * @param serverDeploymentConfigName the properties of the referenced custom Deployment Configuration.
     * @returns a Construct representing a reference to an existing custom Deployment Configuration
     */
    static fromServerDeploymentConfigName(scope, id, serverDeploymentConfigName) {
        ignore(scope);
        ignore(id);
        return deploymentConfig(serverDeploymentConfigName);
    }
}
exports.ServerDeploymentConfig = ServerDeploymentConfig;
/**
 *
 */
ServerDeploymentConfig.ONE_AT_A_TIME = deploymentConfig('CodeDeployDefault.OneAtATime');
/**
 *
 */
ServerDeploymentConfig.HALF_AT_A_TIME = deploymentConfig('CodeDeployDefault.HalfAtATime');
/**
 *
 */
ServerDeploymentConfig.ALL_AT_ONCE = deploymentConfig('CodeDeployDefault.AllAtOnce');
function deploymentConfig(name) {
    return {
        deploymentConfigName: name,
        deploymentConfigArn: utils_1.arnForDeploymentConfig(name),
    };
}
function ignore(_x) { return; }
//# sourceMappingURL=data:application/json;base64,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