"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getOrCreateKubectlLayer = exports.KubectlProvider = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const cluster_1 = require("./cluster");
const kubectl_layer_1 = require("./kubectl-layer");
class KubectlProvider extends core_1.NestedStack {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const cluster = props.cluster;
        if (!cluster.kubectlRole) {
            throw new Error('"kubectlRole" is not defined, cannot issue kubectl commands against this cluster');
        }
        if (cluster.kubectlPrivateSubnets && !cluster.kubectlSecurityGroup) {
            throw new Error('"kubectlSecurityGroup" is required if "kubectlSubnets" is specified');
        }
        const layer = (_a = cluster.kubectlLayer) !== null && _a !== void 0 ? _a : getOrCreateKubectlLayer(this);
        const memorySize = cluster.kubectlMemory ? cluster.kubectlMemory.toMebibytes() : 1024;
        const handler = new lambda.Function(this, 'Handler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'kubectl-handler')),
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(15),
            description: 'onEvent handler for EKS kubectl resource provider',
            layers: [layer],
            memorySize,
            environment: cluster.kubectlEnvironment,
            // defined only when using private access
            vpc: cluster.kubectlPrivateSubnets ? cluster.vpc : undefined,
            securityGroups: cluster.kubectlSecurityGroup ? [cluster.kubectlSecurityGroup] : undefined,
            vpcSubnets: cluster.kubectlPrivateSubnets ? { subnets: cluster.kubectlPrivateSubnets } : undefined,
        });
        this.handlerRole = handler.role;
        this.handlerRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeCluster'],
            resources: [cluster.clusterArn],
        }));
        // allow this handler to assume the kubectl role
        cluster.kubectlRole.grant(this.handlerRole, 'sts:AssumeRole');
        const provider = new cr.Provider(this, 'Provider', {
            onEventHandler: handler,
            vpc: cluster.kubectlPrivateSubnets ? cluster.vpc : undefined,
            vpcSubnets: cluster.kubectlPrivateSubnets ? { subnets: cluster.kubectlPrivateSubnets } : undefined,
        });
        this.serviceToken = provider.serviceToken;
        this.roleArn = cluster.kubectlRole.roleArn;
    }
    static getOrCreate(scope, cluster) {
        // if this is an "owned" cluster, it has a provider associated with it
        if (cluster instanceof cluster_1.Cluster) {
            return cluster._attachKubectlResourceScope(scope);
        }
        // if this is an imported cluster, we need to provision a custom resource provider in this stack
        // we will define one per stack for each cluster based on the cluster uniqueid
        const uid = `${core_1.Names.nodeUniqueId(cluster.node)}-KubectlProvider`;
        const stack = core_1.Stack.of(scope);
        let provider = stack.node.tryFindChild(uid);
        if (!provider) {
            provider = new KubectlProvider(stack, uid, { cluster });
        }
        return provider;
    }
}
exports.KubectlProvider = KubectlProvider;
/**
 * Gets or create a singleton instance of KubectlLayer.
 *
 * (exported for unit tests).
 */
function getOrCreateKubectlLayer(scope, props = {}) {
    const stack = core_1.Stack.of(scope);
    const id = 'kubectl-layer-' + (props.version ? props.version : '8C2542BC-BF2B-4DFE-B765-E181FD30A9A0');
    const exists = stack.node.tryFindChild(id);
    if (exists) {
        return exists;
    }
    return new kubectl_layer_1.KubectlLayer(stack, id, props);
}
exports.getOrCreateKubectlLayer = getOrCreateKubectlLayer;
//# sourceMappingURL=data:application/json;base64,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