"""Number tests."""
from __future__ import annotations

import typing

import pytest

import humanize
from humanize import number


@pytest.mark.parametrize(
    "test_input, expected",
    [
        ("1", "1st"),
        ("2", "2nd"),
        ("3", "3rd"),
        ("4", "4th"),
        ("11", "11th"),
        ("12", "12th"),
        ("13", "13th"),
        ("101", "101st"),
        ("102", "102nd"),
        ("103", "103rd"),
        ("111", "111th"),
        ("something else", "something else"),
        (None, "None"),
    ],
)
def test_ordinal(test_input: str, expected: str) -> None:
    assert humanize.ordinal(test_input) == expected


@pytest.mark.parametrize(
    "test_args, expected",
    [
        ([100], "100"),
        ([1000], "1,000"),
        ([10123], "10,123"),
        ([10311], "10,311"),
        ([1_000_000], "1,000,000"),
        ([1_234_567.25], "1,234,567.25"),
        (["100"], "100"),
        (["1000"], "1,000"),
        (["10123"], "10,123"),
        (["10311"], "10,311"),
        (["1000000"], "1,000,000"),
        (["1234567.1234567"], "1,234,567.1234567"),
        ([None], "None"),
        ([14308.40], "14,308.4"),
        ([14308.40, None], "14,308.4"),
        ([14308.40, 1], "14,308.4"),
        ([14308.40, 2], "14,308.40"),
        ([14308.40, 3], "14,308.400"),
        ([1234.5454545], "1,234.5454545"),
        ([1234.5454545, None], "1,234.5454545"),
        ([1234.5454545, 1], "1,234.5"),
        ([1234.5454545, 2], "1,234.55"),
        ([1234.5454545, 3], "1,234.545"),
        ([1234.5454545, 10], "1,234.5454545000"),
    ],
)
def test_intcomma(
    test_args: list[int] | list[float] | list[str], expected: str
) -> None:
    assert humanize.intcomma(*test_args) == expected


def test_intword_powers() -> None:
    # make sure that powers & human_powers have the same number of items
    assert len(number.powers) == len(number.human_powers)


@pytest.mark.parametrize(
    "test_args, expected",
    [
        (["100"], "100"),
        (["1000"], "1.0 thousand"),
        (["12400"], "12.4 thousand"),
        (["12490"], "12.5 thousand"),
        (["1000000"], "1.0 million"),
        (["1200000"], "1.2 million"),
        (["1290000"], "1.3 million"),
        (["999999999"], "1.0 billion"),
        (["1000000000"], "1.0 billion"),
        (["2000000000"], "2.0 billion"),
        (["999999999999"], "1.0 trillion"),
        (["1000000000000"], "1.0 trillion"),
        (["6000000000000"], "6.0 trillion"),
        (["999999999999999"], "1.0 quadrillion"),
        (["1000000000000000"], "1.0 quadrillion"),
        (["1300000000000000"], "1.3 quadrillion"),
        (["3500000000000000000000"], "3.5 sextillion"),
        (["8100000000000000000000000000000000"], "8.1 decillion"),
        ([None], "None"),
        (["1230000", "%0.2f"], "1.23 million"),
        ([10**101], "1" + "0" * 101),
    ],
)
def test_intword(test_args: list[str], expected: str) -> None:
    assert humanize.intword(*test_args) == expected


@pytest.mark.parametrize(
    "test_input, expected",
    [
        (0, "zero"),
        (1, "one"),
        (2, "two"),
        (4, "four"),
        (5, "five"),
        (9, "nine"),
        (10, "10"),
        ("7", "seven"),
        (None, "None"),
    ],
)
def test_apnumber(test_input: int | str, expected: str) -> None:
    assert humanize.apnumber(test_input) == expected


@pytest.mark.parametrize(
    "test_input, expected",
    [
        (1, "1"),
        (2.0, "2"),
        (4.0 / 3.0, "1 1/3"),
        (5.0 / 6.0, "5/6"),
        ("7", "7"),
        ("8.9", "8 9/10"),
        ("ten", "ten"),
        (None, "None"),
        (1 / 3, "1/3"),
        (1.5, "1 1/2"),
        (0.3, "3/10"),
        (0.333, "333/1000"),
    ],
)
def test_fractional(test_input: float | str, expected: str) -> None:
    assert humanize.fractional(test_input) == expected


@pytest.mark.parametrize(
    "test_args, expected",
    [
        ([1000], "1.00 x 10³"),
        ([-1000], "-1.00 x 10³"),
        ([5.5], "5.50 x 10⁰"),
        ([5781651000], "5.78 x 10⁹"),
        (["1000"], "1.00 x 10³"),
        (["99"], "9.90 x 10¹"),
        ([float(0.3)], "3.00 x 10⁻¹"),
        (["foo"], "foo"),
        ([None], "None"),
        ([1000, 1], "1.0 x 10³"),
        ([float(0.3), 1], "3.0 x 10⁻¹"),
        ([1000, 0], "1 x 10³"),
        ([float(0.3), 0], "3 x 10⁻¹"),
        ([float(1e20)], "1.00 x 10²⁰"),
        ([float(2e-20)], "2.00 x 10⁻²⁰"),
        ([float(-3e20)], "-3.00 x 10²⁰"),
        ([float(-4e-20)], "-4.00 x 10⁻²⁰"),
    ],
)
def test_scientific(test_args: list[typing.Any], expected: str) -> None:
    assert humanize.scientific(*test_args) == expected


@pytest.mark.parametrize(
    "test_args, expected",
    [
        ([1], "1"),
        ([None], None),
        ([0.0001, "{:.0%}"], "0%"),
        ([0.0001, "{:.0%}", 0.01], "<1%"),
        ([0.9999, "{:.0%}", None, 0.99], ">99%"),
        ([0.0001, "{:.0%}", 0.01, None, "under ", None], "under 1%"),
        ([0.9999, "{:.0%}", None, 0.99, None, "above "], "above 99%"),
        ([1, humanize.intword, 1e6, None, "under "], "under 1.0 million"),
    ],
)
def test_clamp(test_args: list[typing.Any], expected: str) -> None:
    assert humanize.clamp(*test_args) == expected


@pytest.mark.parametrize(
    "test_args, expected",
    [
        ([1, "Hz"], "1.00 Hz"),
        ([1.0, "W"], "1.00 W"),
        ([3, "C"], "3.00 C"),
        ([3, "W", 5], "3.0000 W"),
        ([1.23456], "1.23"),
        ([12.3456], "12.3"),
        ([123.456], "123"),
        ([1234.56], "1.23 k"),
        ([12345, "", 6], "12.3450 k"),
        ([200_000], "200 k"),
        ([1e25, "m"], "10.0 Ym"),
        ([1e26, "m"], "100 Ym"),
        ([1e27, "A"], "1.00 x 10²⁷A"),
        ([1.234e28, "A"], "1.23 x 10²⁸A"),
        ([-1500, "V"], "-1.50 kV"),
        ([0.12], "120 m"),
        ([0.012], "12.0 m"),
        ([0.0012], "1.20 m"),
        ([0.00012], "120 μ"),
        ([1e-23], "10.0 y"),
        ([1e-24], "1.00 y"),
        ([1e-25], "1.00 x 10⁻²⁵"),
        ([1e-26], "1.00 x 10⁻²⁶"),
        ([1, "°"], "1.00°"),
        ([0.1, "°"], "100m°"),
        ([100], "100"),
        ([0.1], "100 m"),
    ],
    ids=str,
)
def test_metric(test_args: list[typing.Any], expected: str) -> None:
    assert humanize.metric(*test_args) == expected
