import * as secretsmanager from 'aws-cdk-lib/aws-secretsmanager';
import { Construct, IConstruct } from 'constructs';
import { SlackAppManifestProps } from './manifest';
/**
 * Properties for a SlackApp.
 *
 * @stability stable
 */
export interface SlackAppProps {
    /**
     * The definition of the app manifest.
     *
     * @see https://api.slack.com/reference/manifests
     * @stability stable
     */
    readonly manifest: SlackAppManifestDefinition;
    /**
     * An AWS Secrets Manager secret containing the app configuration token.
     *
     * Must use the following JSON format:
     *
     * ```
     * {
     *    "refreshToken": "<token>"
     * }
     * ```
     *
     * @stability stable
     */
    readonly configurationTokenSecret: secretsmanager.ISecret;
    /**
     * The AWS Secrets Manager secret where to store the app credentials.
     *
     * @default - a new secret is created
     * @stability stable
     */
    readonly credentialsSecret?: secretsmanager.ISecret;
}
/**
 * A Slack app manifest definition.
 *
 * @stability stable
 */
export declare abstract class SlackAppManifestDefinition {
    /**
     * Create a Slack app manifest from a JSON app manifest encoded as a string.
     *
     * @stability stable
     */
    static fromString(manifest: string): SlackAppManifestDefinition;
    /**
     * Creates a Slack app manifest from a file containg a JSON app manifest.
     *
     * @stability stable
     */
    static fromFile(file: string): SlackAppManifestDefinition;
    /**
     * Creates a Slack app manifest by specifying properties.
     *
     * @stability stable
     */
    static fromManifest(props: SlackAppManifestProps): SlackAppManifestDefinition;
    /**
     * Renders the JSON app manifest encoded as a string.
     *
     * @stability stable
     */
    abstract render(construct: IConstruct): string;
}
/**
 * A Slack application deployed with a manifest.
 *
 * @see https://api.slack.com/reference/manifests
 * @stability stable
 */
export declare class SlackApp extends Construct {
    /**
     * The ID of the application.
     *
     * @stability stable
     */
    readonly appId: string;
    /**
     * An AWS Secrets Manager secret containing the credentials of the application.
     *
     * ```
     * {
     *    "appId": "...",
     *    "clientId": "...",
     *    "clientSecret": "...",
     *    "verificationToken": "...",
     *    "signingSecret": "..."
     * }
     * ```
     *
     * @stability stable
     */
    readonly credentials: secretsmanager.ISecret;
    /**
     * A dynamic reference to the client ID of the app.
     *
     * @stability stable
     */
    readonly clientId: string;
    /**
     * A dynamic reference to the client secret of the app.
     *
     * @stability stable
     */
    readonly clientSecret: string;
    /**
     * A dynamic reference to the verification token of the app.
     *
     * @stability stable
     */
    readonly verificationToken: string;
    /**
     * A dynamic reference to the signing secret of the app.
     *
     * @stability stable
     */
    readonly signingSecret: string;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: SlackAppProps);
}
