# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['py_tictoc_timer']

package_data = \
{'': ['*']}

install_requires = \
['typeguard>=2.13,<2.14']

setup_kwargs = {
    'name': 'py-tictoc-timer',
    'version': '1.2.1',
    'description': "Time the execution of Python code using syntax similar to MATLAB's tic and toc functions.",
    'long_description': '<div align="center">\n\n# `py-tictoc-timer`\n\n[![PyPI version](https://img.shields.io/pypi/v/py-tictoc-timer?label=version&logo=git)](https://pypi.org/project/py-tictoc-timer/)\n[![Python](https://img.shields.io/pypi/pyversions/py-tictoc-timer.svg?style=plastic&logo=python&logoColor=FFDE50)](https://pypi.org/project/py-tictoc-timer/)<br>\n[![Released](https://img.shields.io/github/release-date/chrimaho/py-tictoc-timer?label=released&logo=google-calendar&logoColor=FF7143)](https://pypi.org/project/py-tictoc-timer/#history)\n[![Unit Testing](https://img.shields.io/github/workflow/status/chrimaho/py-tictoc-timer/Unit%20Testing/main?label=testing&logo=pytest)](https://github.com/chrimaho/py-tictoc-timer/actions/workflows/unit-tests.yml)\n[![Publish Package](https://img.shields.io/github/workflow/status/chrimaho/py-tictoc-timer/Publish%20Package?label=build&logo=poetry)](https://github.com/chrimaho/py-tictoc-timer/actions/workflows/pypi-publish.yml)\n[![codecov](https://codecov.io/gh/chrimaho/py-tictoc-timer/branch/main/graph/badge.svg)](https://codecov.io/gh/chrimaho/py-tictoc-timer)<br>\n[![Vulnerabilities](https://img.shields.io/snyk/vulnerabilities/github/chrimaho/py-tictoc-timer?color=green)](https://github.com/chrimaho/py-tictoc-timer)\n[![License](https://img.shields.io/pypi/l/py-tictoc-timer)](https://github.com/chrimaho/py-tictoc-timer/blob/main/LICENSE)\n[![Downloads](https://img.shields.io/pypi/dw/py-tictoc-timer)](https://github.com/chrimaho/py-tictoc-timer)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n<!-- [![Stability](https://img.shields.io/pypi/status/py-tictoc-timer)](https://img.shields.io/pypi/status/py-tictoc-timer) -->\n\n</div>\n\nTime the execution of Python code using syntax similar to MATLAB\'s tic and toc functions.\n\n<table>\n<td>\n\n**Contents**\n\n- [`py-tictoc-timer`](#py-tictoc-timer)\n  - [Installation](#installation)\n  - [Usage](#usage)\n  - [Contribution](#contribution)\n  - [Tests](#tests)\n  - [Credit](#credit)\n  - [Maintainers](#maintainers)\n\n</td>\n</table>\n\n## Installation\n\n- Using [`pip`](https://pypi.org/project/pip):\n  ```sh\n  pip install py-tictoc-timer\n  ```\n\n- Using [`pipenv`](https://github.com/pypa/pipenv):\n  ```sh\n  pipenv install py-tictoc-timer\n  ```\n\n- Using [`poetry`](https://python-poetry.org):\n  1. In your `pyproject.toml` file, add:\n      ```toml\n      [tool.poetry.dependencies]\n      py-tictoc-timer = "*"\n      ```\n      Then in the terminal, run:\n      ```sh\n      poetry install\n      ```\n  3. Or run:\n      ```sh\n      poetry add py-tictoc-timer\n      ```\n\n- Using [`conda`](https://docs.conda.io):\n  ```sh\n  conda install py-tictoc-timer\n  ```\n\n## Usage\n\n- Basic usage:\n  ```python linenums="1"\n  >>> from py_tictoc_timer.tictoc import TicToc\n  >>> from time import sleep\n  >>> tt = TicToc()\n  >>> tt.tic()\n  >>> sleep(1.1)\n  >>> tt.toc()\n  Elapsed time: 1secs\n  ```\n\n- Within context manager:\n  ```python linenums="1"\n  >>> from py_tictoc_timer.tictoc import TicToc\n  >>> from time import sleep\n  >>> with TicToc():\n  ...     sleep(1.1)\n  Elapsed time: 1secs\n  ```\n\n- Within context manager using custom messages:\n  ```python linenums="1"\n  >>> from py_tictoc_timer.tictoc import TicToc\n  >>> from time import sleep\n  >>> with TicToc(begin_message="start", end_message="end"):\n  ...     sleep(1.1)\n  start\n  end: 1secs\n  ```\n\n- Particularly helpful when running loops:\n  ```python linenums="1"\n  >>> from py_tictoc_timer.tictoc import TicToc\n  >>> from time import sleep\n  >>> with TicToc(begin_message="Start loop", end_message="Time to run loop")\n  ...     for value in ["first", "second", "Third"]:\n  ...         with TicToc(f"- Time for {value}"):\n  ...             sleep(1.1)\n  Start loop\n  - Time for first: 1secs\n  - Time for second: 1secs\n  - Time for Third: 1secs\n  Time to run loop: 3secs\n  ```\n\n- Custom message:\n  ```python linenums="1"\n  >>> from py_tictoc_timer.tictoc import TicToc\n  >>> from time import sleep\n  >>> with TicToc("Total Time"):\n  ...     sleep(1.1)\n  Total time: 1secs\n  ```\n\n- With restart during `.tic()`:\n  ```python linenums="1"\n  >>> from py_tictoc_timer.tictoc import TicToc\n  >>> from time import sleep\n  >>> tt = TicToc()\n  >>> tt.tic(restart=True)\n  >>> sleep(1.1)\n  >>> toc()\n  Elapsed time: 1secs\n  >>> toc()\n  Elapsed time: 1secs\n  ```\n\n- With restart during `.toc()`:\n  ```python linenums="1"\n  >>> from py_tictoc_timer.tictoc import TicToc\n  >>> from time import sleep\n  >>> tt = TicToc()\n  >>> tt.tic()\n  >>> sleep(1.1)\n  >>> tt.toc(restart=True)\n  Elapsed time: 1secs\n  >>> tt.toc()\n  Elapsed time: 1secs\n  ```\n\n- With restart using `.rtoc()`:\n  ```python linenums="1"\n  >>> from py_tictoc_timer.tictoc import TicToc\n  >>> from time import sleep\n  >>> tt = TicToc()\n  >>> tt.tic()\n  >>> sleep(1.1)\n  >>> tt.rtoc()\n  Elapsed time: 1secs\n  >>> tt.toc()\n  Elapsed time: 1secs\n  ```\n\n- With time returned:\n  ```python linenums="1"\n  >>> from py_tictoc_timer.tictoc import TicToc\n  >>> from time import sleep\n  >>> tt = TicToc()\n  >>> tt.tic()\n  >>> sleep(1.1)\n  >>> value = tt.toc_value()\n  >>> print(round(value, 1))\n  1.1\n  ```\n\n## Contribution\nContribution is always welcome!\n1. First, either [fork](https://docs.github.com/en/get-started/quickstart/fork-a-repo) or [branch](https://docs.github.com/en/pull-requests/collaborating-with-pull-requests/proposing-changes-to-your-work-with-pull-requests/creating-and-deleting-branches-within-your-repository#creating-a-branch) the [main repo](https://github.com/chrimaho/py-tictoc-timer/tree/main).\n2. [Clone](https://docs.github.com/en/repositories/creating-and-managing-repositories/cloning-a-repository) your forked/branched repo.\n3. Build your environment with any of the below options:\n   1. With [`pipenv`](https://github.com/pypa/pipenv):\n        ```sh\n        if (-not (Test-Path .venv)) {mkdir .venv}\n        python -m pipenv install --requirements requirements.txt --ignore-pipfile --skip-lock --no-site-packages\n        python -m pipenv install --requirements requirements-dev.txt --dev --ignore-pipfile --skip-lock --no-site-packages\n        python -m pipenv run pre-commit install\n        ```\n   2. With [`poetry`](https://python-poetry.org/) on Windows:\n        ```sh\n        (Invoke-WebRequest -Uri https://raw.githubusercontent.com/python-poetry/poetry/master/get-poetry.py -UseBasicParsing).Content | python -\n        python -m poetry run pre-commit install\n        ```\n   3. With [`poetry`](https://python-poetry.org/) on Linux:\n        ```sh\n\t    curl -sSL https://raw.githubusercontent.com/python-poetry/poetry/master/install-poetry.py | python -\n        python -m poetry run pre-commit install\n        ```\n4. Start contributing.\n5. Ensure you add additional [Unit Test](https://docs.python.org/3/library/unittest.html)\'s to the [test library](https://github.com/chrimaho/py-tictoc-timer/blob/main/tests/test_tictoc.py) for each new feature/functionality.\n6. Ensure that all the [tests](#tests) are passing successfully.\n7. When you\'re happy with the changes, raise a [Pull Request](https://docs.github.com/en/pull-requests/collaborating-with-pull-requests/proposing-changes-to-your-work-with-pull-requests/creating-a-pull-request) to merge with the [main](https://github.com/chrimaho/py-tictoc-timer/tree/main) branch again.\n\n## Tests\n\n- Run [Black](https://black.readthedocs.io/):\n  ```sh\n  pipenv run python -m black --safe py_tictoc_timer tests\n  ```\n\n- Run [PyTests](https://docs.pytest.org):\n  ```sh\n  pipenv run python -m pytest --verbose --cov=py_tictoc_timer --cov-report=term --cov-report=html:cov-report/html --cov-report=xml:cov-report/xml/cov-report.xml\n  ```\n\n- Run [MyPy](http://www.mypy-lang.org) Tests:\n  ```sh\n  pipenv run mypy py_tictoc_timer --ignore-missing-imports --pretty --install-types --non-interactive\n  ```\n\n## Credit\n\nThis package was inspired by a few other packages:\n- [`pytictoc`](https://pypi.org/project/pytictoc/)\n- [`TicTocTimer`](https://pypi.org/project/tictoctimer/)\n- [`ttictoc`](https://pypi.org/project/ttictoc/)\n- [`easy-tictoc`](https://pypi.org/project/easy-tictoc/)\n- [`easy-tic-toc`](https://pypi.org/project/easy-tic-toc/)\n- [`tictoc-borisgorelik`](https://pypi.org/project/tictoc-borisgorelik/)\n\nWhy you should use [`py-tictoc-timer`](https://github.com/chrimaho/py-tictoc-timer) and not any of the others is because this package has:\n1. Better & more flexible restart to the timer\n2. Better custom messages during starting & ending the timer\n3. Enhanced usage within a context manager\n\n## Maintainers\n<a href="https://github.com/chrimaho/py-tictoc-timer/graphs/contributors">\n  <img src="https://contributors-img.web.app/image?repo=chrimaho/py-tictoc-timer" width=40/>\n</a>\n',
    'author': 'Chris Mahoney',
    'author_email': 'chrismahoney@hotmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/chrimaho/py-tictoc-timer.git',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4',
}


setup(**setup_kwargs)
