# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['arrow_pd_parser']

package_data = \
{'': ['*']}

install_requires = \
['pandas>=1.2,<2.0', 'pyarrow>=2.0.0,<4.0.0']

setup_kwargs = {
    'name': 'arrow-pd-parser',
    'version': '0.2.0',
    'description': 'MoJ arrow-pd-parser',
    'long_description': '# mojap-arrow-pd-parser\n\nUsing pyArrow to read CSV, JSONL and Parquet ensuring that you get the best representation of the column types in the resulting Pandas dataframe. Also ensures data type conformance by maintaining column types when reading and writing data back into Pandas (even when round tripping across multiple data types).\n\nThis package also can read in data given a pyArrow schema again to result in a Pandas dataframe that best represents the provided schema. Datatypes are returned as well as pandas writes to the these files correctly for \n\n## Installation\n\n```\npip install arrow-pd-parser\n```\n\nor via GitHub\n\n```\npip install arrow-pd-parser @ git+https://github.com/moj-analytical-services/mojap-arrow-pd-parser\n```\n\n## Usage\n\nThis package uses `pyArrow` to parse CSVs, JSONL and Parquet files and convert them to a Pandas Dataframe that are the best representation of those datatypes and ensure conformance between them.\n\n```python\nfrom arrow_pd_parser.parse import pa_read_csv_to_pandas\n\ndf = pa_read_csv_to_pandas("tests/data/all_types.csv")\ndf.dtypes()\n\n# i                     Int64\n# my_bool             boolean\n# my_nullable_bool    boolean\n#\xa0my_date              object\n# my_datetime          object\n# my_int                Int64\n# my_string            string\n```\n\nNote that the default behavior of this package is to utilse the new pandas datatypes for Integers, Booleans and Strings that represent Nulls as `pd.NA()`. Dates are returned as nullable objects of `datetime.date()` type and timestamps are `datetime.datetime()`. By default we enforce these types instead of the native pandas timestamp as the indexing for the Pandas timestamp is nanoseconds and can cause dates to be out of bounds. See the [timestamps](#timestamps) section for more details.\n\n### Advanced Usage\n\nThis package will read in tabular data using pyarrow and then convert it to a pandas dataset. The full order of operations are:\n- Read in data to an arrow dataset\n- Cast the arrow dataset to the specified schema (if provided)\n- Convert the arrow dataset to Pandas\n\n#### Reading and Schema Casting\n\nYou can split up the previous code example (which used `pa_read_csv_to_pandas`) into two parts to get the exact same result (in case you wanted to do some transformations to the arrow dataset first).\n\n```python\nfrom arrow_pd_parser.parse import pa_read_csv\nfrom arrow_pd_parser.pa_pd import arrow_to_pandas\n\n# Read in the data first then convert it to a pandas dataframe\ndf_arrow = pa_read_csv("tests/data/all_types.csv")\ndf = arrow_to_pandas(df_arrow)\n```\n\nYou can specify the additional parameters used by the pyarrow readers when reading in the data. \n\n```python\nfrom io import BytesIO\nfrom pyarrow import csv\nfrom arrow_pd_parser.parse import pa_read_csv_to_pandas\n\ncsv_data = b"""\na;b\n1;"This is text with a \nnew line"\n2;some more text\n"""\ntest_file = BytesIO(csv_data)\ncsv_parse_options = csv.ParseOptions(delimiter=";", newlines_in_values=True)\ndf = pa_read_csv_to_pandas(test_file, parse_options=csv_parse_options)\ndf.head()\n```\n\nYou can also provide an arrow schema to try and cast the data.\n\n```python\nfrom io import BytesIO\nimport pyarrow as pa\nfrom arrow_pd_parser.parse import pa_read_csv_to_pandas\n\n\ncsv_data = b"""\na,b\n1,1.24\n2,7.81\n"""\ntest_file = BytesIO(csv_data)\nfrom io import BytesIO\nimport pyarrow as pa\nfrom arrow_pd_parser.parse import pa_read_csv_to_pandas\n\ncsv_data = b"""\na,b\n1,1.24\n2,7.81\n"""\n\n# By default the data above would read as\n# a:Int64, b:float64 for a Pandas dataframe\n# Instead tell pa_pd_parser to treat these with the following schema\nschema = pa.schema([("a", pa.string()), ("b", pa.decimal128(3,2))])\ntest_file = BytesIO(csv_data)\ndf = pa_read_csv_to_pandas(test_file, schema=schema)\ndf.types # a: String, b:object (each value is a decimal.Decimal)\n```\n\nNote there are currently some issues around pyarrow not being able to cast timestamps to strings (see this see repo\'s issues for more details). For example:\n\n```python\nfrom io import BytesIO\nimport pyarrow as pa\nfrom pyarrow import csv\nfrom arrow_pd_parser.parse import pa_read_csv_to_pandas\n\ncsv_data = b"""\na,b\n1,2020-01-01 00:00:00\n2,2021-01-01 23:59:59\n"""\n\n# note can also provide partial schema and get package to infer a\'s type by also setting `expect_full_schema=False`\nschema = pa.schema([("b", pa.string())])\ntest_file = BytesIO(csv_data)\n\n# The following line will raise an ArrowNotImplementedError.\n# This is because there is currently no implementation to casting timestamps to str.\ndf = pa_read_csv_to_pandas(test_file, schema=schema, expect_full_schema=False)\n\n# By default Arrow will read in str representations of timestamps as\n# timestamps if they conform to ISO standard format.\n# Then you get the error when you try and cast that timestamp to str. To\n# get around this you can force pyarrow to read in the data as a string\n# when it parses it as a CSV (note that ConvertOptions is not currently\n# available for the JSON reader)\nco = csv.ConvertOptions(column_types=schema)\ndf = pa_read_csv_to_pandas(test_file, schema=schema, expect_full_schema=False, convert_options=co)\n```\n\n#### Exporting data to CSV/JSON\n\nYou can also use the export module of this package to write data back (to CSV and JSON to ensure the same datatype will be read back in). This is useful when having to constantly read/write data between different storage systems and/or pipelines.\n\n```python\nfrom arrow_pd_parser.parse import (\n    pa_read_csv_to_pandas,\n    pa_read_json_to_pandas,\n)\nimport pyarrow as pa\nfrom io import StringIO, BytesIO\nfrom arrow_pd_parser.export import pd_to_json\n\ns = pa.schema(\n    [\n        ("i", pa.int8()),\n        ("my_bool", pa.bool_()),\n        ("my_nullable_bool", pa.bool_()),\n        ("my_date", pa.date32()),\n        ("my_datetime", pa.timestamp("s")),\n        ("my_int", pa.uint8()),\n        ("my_string", pa.string()),\n    ]\n)\n\n# Read in original table\noriginal = pa_read_csv_to_pandas(\n        "tests/data/all_types.csv",\n        s,\n    )\n\n# Write the table back out to a JSONL file\nf = StringIO()\npd_to_json(original, f)\n\n# Read it back in and check it matches original\nnew_f = BytesIO(f.getvalue().encode("utf8"))\nnew = pa_read_json_to_pandas(new_f, s)\n\noriginal == new # note that the two False values are where datetime is None in both tables\n```\n\n#### Integration with mojap-metadata\n\nThis package can also be used alongside the `mojap-metadata` package. In the example below you will need to install the package with the arrow dependencies:\n\n```\npip install mojap-metadata[arrow]\n```\n\n\n```python\nfrom io import BytesIO\nfrom mojap_metadata import Metadata\nfrom mojap_metadata.converters.arrow_converter import (\n    ArrowConverter\n)\nfrom arrow_pd_parser.parse import pa_read_csv_to_pandas\n\ncsv_data = b"""\na,b\n1,2020-01-01 00:00:00\n2,2021-01-01 23:59:59\n"""\ntest_file = BytesIO(csv_data)\n\n# Define our metadata for the data\nmd = {\n    "name": "test_data",\n    "columns": [\n        {\n            "name": "a",\n            "type": "string"\n        },\n        {\n            "name": "b",\n            "type": "timestamp(ms)"\n        }\n    ] \n}\nmeta = Metadata.from_dict(md)\n\n# Convert our schema to an arrow schema\nac = ArrowConverter()\narrow_schema = ac.generate_from_meta(meta)\n\n# Use the arrow_schema with arrow_pd_parser\ndf = pa_read_csv_to_pandas(test_file, schema=arrow_schema)\n```\n\n## Data Type Conformance\n\n### Timestamps\n\nPandas timestamps (currently) only support nanosecond resolution which is not ideal for a lot of timestamps as the range can be often too small.\n\n```python\nimport pandas as pd\npd.Timestamp.min # Timestamp(\'1677-09-22 00:12:43.145225\')\n\npd.Timestamp.max # Timestamp(\'2262-04-11 23:47:16.854775807\')\n```\n\nWhereas Spark 3.0 (for example) allows timestamps from `0001-01-01 00:00:00` to `9999-12-31 23:59:59.999999` ([source](https://databricks.com/blog/2020/07/22/a-comprehensive-look-at-dates-and-timestamps-in-apache-spark-3-0.html)). By default we do not allow Timestamps for this reason instead we use the python native datetime class as default for our types (wrapped in a Pandas object column type). Users can specify other Pandas date/timestamp using the `pd_timestamp_type` parameter which can either be `object` (default), `pd_timestamp` or `pd_period`. \n\nWhen setting `pd_timestamp_type=pd_period` pd_arrow_parser will identify the correct pandas period resolution based on the arrow column type.\n\n```python\nfrom arrow_pd_parser.parse import pa_read_csv_to_pandas\nimport pyarrow as pa\n\n# The Pandas Period type resolution is determined by the arrow col type\'s resolution\nschema = pa.schema([("my_datetime", pa.timestamp("s"))])\ndf = pa_read_csv_to_pandas(\n    "tests/data/datetime_type.csv",\n    pd_timestamp_type="pd_period",\n)\ndf.my_datetime.dtype # "period[S]"\n\nschema = pa.schema([("my_datetime", pa.timestamp("ms"))])\ndf = pa_read_csv_to_pandas(\n    "tests/data/datetime_type.csv",\n    pd_timestamp_type="pd_period",\n)\ndf.my_datetime.dtype # "period[L]"\n\n\n# Using timestamp type\ndf = pa_read_csv_to_pandas(\n    "tests/data/datetime_type.csv",\n    pd_timestamp_type="pd_timestamp"\n)\ndf.my_datetime.dtype # dtype(\'<M8[ns]\')\n```\n\n\n',
    'author': 'Karik Isichei',
    'author_email': 'karik.isichei@digital.justice.gov.uk',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/moj-analytical-services/mojap-arrow-pd-parser',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7.1,<=3.9',
}


setup(**setup_kwargs)
