# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['sentinel']
setup_kwargs = {
    'name': 'sentinel',
    'version': '0.2.1a0',
    'description': 'Create sentinel objects, akin to None, NotImplemented, Ellipsis',
    'long_description': "*************************************************\nsentinel — create sentinel and singleton objects\n*************************************************\n\nCreates simple sentinel objects which are the only instance of their own\nanonymous class. As a singleton, there is a guarantee that there will only\never be one instance: they can be safely used with ``pickle`` and ``cPickle``\nalike, as well as being able to be used properly with ``copy.deepcopy()``. In\naddition, a self-documenting ``__repr__`` is provided for free!\n\nUsage\n=====\n\nSentinels_ are singleton_ objects that typically represent some end or\nterminating condition. Some singletons already exist in Python, like ``None``\n``NotImplemented``, and ``Ellipsis``.\n\n\nAll that's needed to create a sentinel is its name::\n\n    >>> import sentinel\n    >>> Nothing = sentinel.create('Nothing')\n    >>> Nothing\n    Nothing\n\nThis by itself is useful when other objects such as ``None``, ``False``,\n``0``, ``-1``, etc.  are entirely valid values. For example, setting default\nvalues when all other values are valid with: ``dict.setdefault()``::\n\n    >>> MissingEntry = sentinel.create('MissingEntry')\n    >>> d = {'stdout': None, 'stdin': 0, 'EOF': -1}\n    >>> [d.setdefault(key, MissingEntry) for key in ('stdin', 'stdout', 'stderr')]\n    [0, None, MissingEntry]\n\nAlternatively, using ``dict.get()`` when fetching values::\n\n    >>> d = {'stdout': None, 'stdin': 0, 'EOF': -1}\n    >>> d.get('stdout', MissingEntry)\n    None\n    >>> d.get('stdin', MissingEntry)\n    0\n    >>> d.get('stderr', MissingEntry)\n    MissingEntry\n\nIt's known immediately which value was missing from the dictionary in a\nself-documenting manner.\n\nAdding extra methods and class attributes\n-----------------------------------------\n\nSentinels may also inherit from base classes, or implement extra methods.\n\nConsider a binary search tree with two kinds of nodes: interior nodes\n(``Node``) which contain some payload and leaves (``Leaf``), which simply\nterminate traversal.\n\nTo create singleton leaf which implements a ``search`` method and an\n``is_leaf`` property, you may provide any extra class attributes in the\n``cls_dict`` keyword argument. The following is a full example of both\nthe singleton ``Leaf`` and its ``Node`` counterpart::\n\n    def _search_leaf(self, key):\n        raise KeyError(key)\n\n    Leaf = sentinel.create('Leaf', cls_dict={\n        'search': _search_leaf,\n        'is_leaf': property(lambda self: True)\n    })\n\n    class Node(object):\n        def __init__(self, key, payload, left=Leaf, right=Leaf):\n            self.left = left\n            self.right = right\n            self.key = key\n            self.payload = payload\n\n        def search(self, key):\n            if key < self.key:\n                return self.left.search(key)\n            elif key > self.key:\n                return self.right.search(key)\n            else:\n                return self.payload\n\n        is_leaf = property(lambda: false)\n\nExample usage::\n\n    >>> tree = Node(2, 'bar', Node(1, 'foo'), Node(3, 'baz'))\n    >>> tree.search(1)\n    'foo'\n    >>> tree.search(4)\n    Traceback (most recent call last):\n        ...\n    KeyError: 2\n\n\nContributing\n============\n\nThis project uses Poetry_. To contribute to the codebase, make sure to `install poetry`_,\nWith Poetry installed, clone then repo, then within the repo directory, install the developer dependencies::\n\n    $ poetry install\n\nNext, I recommend you do all development tasks within the ``poetry shell``::\n\n    $ poetry shell\n    (sentinel-nUnrocCf-py3.9) $ black .\n    (sentinel-nUnrocCf-py3.9) $ pytest\n\n.. _Sentinels: http://en.wikipedia.org/wiki/Sentinel_nodes\n.. _singleton: http://en.wikipedia.org/wiki/Singleton_pattern\n.. _Poetry: https://python-poetry.org/\n.. _install poetry: https://python-poetry.org/docs/#installation\n",
    'author': 'Eddie Antonio Santos',
    'author_email': 'easantos@ualberta.ca',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/eddieantonio/sentinel',
    'py_modules': modules,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
