import numpy as np
from math import pow, pi
import sys, os
import fsps
from operator import itemgetter
from astropy.io import fits
from scipy.interpolate import interp1d
from scipy.integrate import simps

from ..utils.redshifting import cosmo_redshifting
from ..utils.filtering import filtering, cwave_filters
from ..utils.igm_absorption import igm_att_madau, igm_att_inoue

# warning is not logged here. Perfect for clean unit test output
with np.errstate(divide='ignore'):
    np.float64(1.0) / 0.0


__all__ = ["tau_SFH", "delay_tau_SFH", "lognormal_SFH", "gaussian_SFH", "double_power_SFH", "grid_SFH", "grid_arbitrary_SFH","calc_mw_age", "construct_SFH", 
			"construct_arbitrary_SFH", "calc_cumul_SM_t", "t_mass_formed", "frac_mass_stars", "csp_spec_restframe_fit", "get_dust_mass_othSFH_fit", 
			"get_sfr_dust_mass_fagnbol_othSFH_fit", "get_sfr_dust_mass_othSFH_fit", "get_dust_mass_mainSFH_fit", "get_dust_mass_fagnbol_mainSFH_fit", "spec_given_SFH_ZH", 
			"spec_given_ages_mass_Z", "convert_unit_spec_from_ergscm2A", "get_no_nebem_wave_fit", "calc_bollum_from_spec", 
			"calc_bollum_from_spec_rest", "breakdown_CSP_into_SSP", "breakdown_CSP_into_SSP_restframe", "get_emlines_luminosities"]


def tau_SFH(tau,t):
	"""A function to calculate SFR at a time t in the tau-model SFH
	"""    
	np.seterr(all='ignore', divide='ignore', over='ignore')
	SFR = 1.0/np.exp(t/tau)    # in unit of M_sun/Gyr
	return SFR

def delay_tau_SFH(tau,t):
	"""A function to calculate SFR at a time t in the delayed-tau SFH model 
	"""
	np.seterr(all='ignore', divide='ignore', over='ignore')
	SFR = t/np.exp(t/tau)      # in unit of M_sun/Gyr
	return SFR

def lognormal_SFH(t0,tau,t):
	"""A function to calculate SFR at a time t in the log-normal SFH model
	"""
	np.seterr(all='ignore', divide='ignore', over='ignore')
	exp0 = (np.log(t)-t0)*(np.log(t)-t0)/2.0/tau/tau
	SFR = 1.0/t/np.exp(exp0)
	return SFR                 # in unit of M_sun/Gyr
 
def gaussian_SFH(t0,tau,t):
	"""A function to calculate SFR at a time t in the Gaussian SFH model
	"""
	np.seterr(all='ignore', divide='ignore', over='ignore')
	exp0 = (t-t0)*(t-t0)/2.0/tau/tau
	SFR = 1.0/np.exp(exp0)
	return SFR                 # in unit of M_sun/Gyr

def double_power_SFH(tau,alpha,beta,t):
	"""A function to calculate SFR at a time t in the Double power law SFH model 
	"""
	np.seterr(all='ignore', divide='ignore', over='ignore')
	c1 = np.power(t/tau,alpha)
	c2 = 1.0/np.power(t/tau,beta)
	SFR = 1.0/(c1 + c2)
	return SFR                 # in unit of M_sun/Gyr


def grid_SFH(sfh_form=4,tau=0,t0=0,alpha=0,beta=0,age=0,formed_mass=0):
	np.seterr(all='ignore', divide='ignore', over='ignore')

	sfh_age = []
	sfh_sfr = []
	sfh_mass = []

	del_t = 0.001
	sfh_age = np.linspace(0,age,int(age/del_t)+1)

	t = age - sfh_age
	if sfh_form==0:
		sfh_sfr = tau_SFH(tau,t)
	elif sfh_form==1:
		sfh_sfr = delay_tau_SFH(tau,t)
	elif sfh_form==2:
		sfh_sfr = lognormal_SFH(t0,tau,t)
	elif sfh_form==3:
		sfh_sfr = gaussian_SFH(t0,tau,t)
	elif sfh_form==4:
		sfh_sfr = double_power_SFH(tau,alpha,beta,t)
	else:
		print ("SFH choice is not recognized!")
		sys.exit()

	sfh_mass = sfh_sfr*del_t
	intg_mass = simps(sfh_sfr[::-1],t[::-1])
	norm = formed_mass/intg_mass

	sfh_t = np.asarray(sfh_age)
	sfh_sfr = np.asarray(sfh_sfr)*norm/1e+9
	sfh_mass = np.asarray(sfh_mass)*norm
	return sfh_t, sfh_sfr, sfh_mass

 
def grid_arbitrary_SFH(age,sfh_t,sfh_sfr0):
	""" sfh_t goes forward from 0 to the age: linear
	"""
	del_t = 0.001
	sfh_age = np.linspace(del_t,int(age/del_t)*del_t,int(age/del_t))

	sfh_age0 = age - np.asarray(sfh_t)
	f = interp1d(sfh_age0, sfh_sfr0, fill_value="extrapolate")
	sfh_sfr = f(sfh_age)
	sfh_sfr[0] = sfh_sfr[1]
	sfh_sfr[len(sfh_age)-1] = sfh_sfr[len(sfh_age)-2]

	sfh_mass = sfh_sfr*del_t

	return sfh_age, sfh_sfr, sfh_mass


def calc_mw_age(sfh_form=4,tau=0,t0=0,alpha=0,beta=0,age=0,formed_mass=0,sfh_t=[],sfh_sfr=[]):
	"""A function for calculating mass-weighted age of a given model SFH
	"""
	
	if sfh_form==0 or sfh_form==1 or sfh_form==2 or sfh_form==3 or sfh_form==4:
		sfh_age, sfh_sfr0, sfh_mass = grid_SFH(sfh_form=sfh_form,tau=tau,t0=t0,alpha=alpha,beta=beta,age=age,formed_mass=formed_mass)
	elif sfh_form=='arbitrary_sfh':
		sfh_age, sfh_sfr0, sfh_mass = grid_arbitrary_SFH(age,sfh_t,sfh_sfr)
	else:
		print ("SFH choice is not recognized!")
		sys.exit()

	mod_mw_age = np.sum(sfh_mass*sfh_age)/np.sum(sfh_mass)

	if np.isnan(mod_mw_age)==True or np.isinf(mod_mw_age)==True or mod_mw_age==0.0:
		return age

	return mod_mw_age


def construct_SFH(sfh_form=4,t0=0.0,tau=0.0,alpha=0.0,beta=0.0,age=0.0,formed_mass=1.0,del_t=0.001):
	"""A function to construct a parametric SFH model

	:param sfh_form:
		A choice of parametric SFH model. Options are: ['tau_sfh', 'delayed_tau_sfh', 'log_normal_sfh', 'gaussian_sfh', 'double_power_sfh']
	"""
	np.seterr(all='ignore', divide='ignore', over='ignore')

	t = np.linspace(0,age,int(age/del_t)+1)

	if sfh_form==0:
		SFR_t = tau_SFH(tau,t)
	elif sfh_form==1:
		SFR_t = delay_tau_SFH(tau,t)
	elif sfh_form==2:
		SFR_t = lognormal_SFH(t0,tau,t)
	elif sfh_form==3:
		SFR_t = gaussian_SFH(t0,tau,t)
	elif sfh_form==4:
		SFR_t = double_power_SFH(tau,alpha,beta,t)
	else:
		print ("SFH choice is not recognized!")
		sys.exit()

	idx = np.where((np.isnan(SFR_t)==True) | (np.isinf(SFR_t)==True) | (SFR_t<0))
	if len(idx[0])>0:
		SFR_t[idx[0]] = 0.0
	intg_mass = simps(SFR_t,t)
	SFR_t = SFR_t*formed_mass/intg_mass/1e+9 

	return t,SFR_t

def construct_arbitrary_SFH(age,sfh_t,sfh_sfr):
	"""A function to construct arbitrary (i.e., non-functional form) SFH model
	"""
	npoints = len(sfh_t)

	del_t = 0.001
	t = np.linspace(del_t,age,int((age-del_t)/del_t)+1)

	f = interp1d(sfh_t,sfh_sfr, fill_value="extrapolate")
	SFR_t = f(t)
	SFR_t[0] = SFR_t[1]
	SFR_t[len(t)-1] = SFR_t[len(t)-2]

	return t, SFR_t


def calc_cumul_SM_t(sfh_form=4,tau=0,t0=0,alpha=0,beta=0,age=None,formed_mass=0,sfh_t=[],sfh_sfr=[],del_t=0.001):
	"""A function to calculate history of stellar mass growth
	"""
	if age == None:
		age = max(sfh_t)

	# make grid of SFH with time
	if sfh_form==0 or sfh_form==1 or sfh_form==2 or sfh_form==3 or sfh_form==4:
		t,SFR_t = construct_SFH(sfh_form=sfh_form,t0=t0,tau=tau,alpha=alpha,beta=beta,age=age,formed_mass=formed_mass)
	elif sfh_form=='arbitrary_sfh':
		t,SFR_t = construct_arbitrary_SFH(age=age,sfh_t=sfh_t,sfh_sfr=sfh_sfr)
	else:
		print ("SFH choice is not recognized!")
		sys.exit()

	ntimes = len(t)

	SM_t = SFR_t*del_t*1.0e+9

	cumul_SM_t = np.cumsum(SM_t)
	sSFR_t = SFR_t*1.0e+9/cumul_SM_t
	 
	return (t,SFR_t,SM_t,sSFR_t,cumul_SM_t)


def t_mass_formed(t_fwd=[],cumul_SM_t=[],age=None,perc=50.0):
	"""A function for calculating time at which a given fraction of current M* formed.
	"""
	if age == None:
		age = max(t_fwd)
	total_SM = max(cumul_SM_t)
	crit_SM = perc*total_SM/100.0

	del_SM = np.abs(crit_SM - cumul_SM_t)
	idx, min_val = min(enumerate(del_SM), key=itemgetter(1))

	t_fwd_form = t_fwd[idx]
	lbt_form = age - t_fwd_form
	
	return t_fwd_form, lbt_form

def frac_mass_stars(SFH_SFR,SFH_t,age_max):
	"""A function for calculating a fraction of mass associated with stars with ages below a certain value:
	## SFH_SFR and SFH_t is going backward in time, such that SFH_t is array of look-back time and SFH_SFR[0] is current SFR:
	## SFH_t should be in Gyr and SFH_SFR should be in M0/yr
	"""
	ntimes = len(SFH_t)
	tot_mass = 0
	tot_mass_below_age = 0
	for ii in range(0,ntimes-1):
		grid_mass = 0.5*(SFH_SFR[ii]+SFH_SFR[ii+1])*(SFH_t[ii+1]-SFH_t[ii])*1.0e+9
		if SFH_t[ii+1]<=age_max:
			tot_mass_below_age = tot_mass_below_age + grid_mass
		tot_mass = tot_mass + grid_mass

	frac_mass = tot_mass_below_age/tot_mass 
	return frac_mass

def rest_wave_fsps():
	wave = [91.0, 94.0, 96.0, 98.0, 100.0, 102.0, 104.0, 106.0, 108.0, 110.0, 114.0, 118.0, 121.0, 125.0, 127.0, 128.0, 131.0, 132.0, 134.0, 137.0, 140.0, 143.0, 147.0, 151.0, 155.0, 159.0, 162.0, 166.0, 170.0, 173.0, 177.0, 180.0, 182.0, 186.0, 191.0, 194.0, 198.0, 202.0, 205.0, 210.0, 216.0, 220.0, 223.0, 227.0, 230.0, 234.0, 240.0, 246.0, 252.0, 257.0, 260.0, 264.0, 269.0, 274.0, 279.0, 284.0, 290.0, 296.0, 301.0, 308.0, 318.0, 328.0, 338.0, 348.0, 357.0, 366.0, 375.0, 385.0, 395.0, 405.0, 414.0, 422.0, 430.0, 441.0, 451.0, 460.0, 470.0, 480.0, 490.0, 500.0, 506.0, 512.0, 520.0, 530.0, 540.0, 550.0, 560.0, 570.0, 580.0, 590.0, 600.0, 610.0, 620.0, 630.0, 640.0, 650.0, 658.0, 665.0, 675.0, 685.0, 695.0, 705.0, 716.0, 726.0, 735.0, 745.0, 755.0, 765.0, 775.0, 785.0, 795.0, 805.0, 815.0, 825.0, 835.0, 845.0, 855.0, 865.0, 875.0, 885.0, 895.0, 905.0, 915.0, 925.0, 935.0, 945.0, 955.0, 965.0, 975.0, 985.0, 995.0, 1005.0, 1015.0, 1025.0, 1035.0, 1045.0, 1055.0, 1065.0, 1075.0, 1085.0, 1095.0, 1105.0, 1115.0, 1125.0, 1135.0, 1145.0, 1155.0, 1165.0, 1175.0, 1185.0, 1195.0, 1205.0, 1215.0, 1225.0, 1235.0, 1245.0, 1255.0, 1265.0, 1275.0, 1285.0, 1295.0, 1305.0, 1315.0, 1325.0, 1335.0, 1345.0, 1355.0, 1365.0, 1375.0, 1385.0, 1395.0, 1405.0, 1415.0, 1425.0, 1435.0, 1442.0, 1447.0, 1455.0, 1465.0, 1475.0, 1485.0, 1495.0, 1505.0, 1512.0, 1517.0, 1525.0, 1535.0, 1545.0, 1555.0, 1565.0, 1575.0, 1585.0, 1595.0, 1605.0, 1615.0, 1625.0, 1635.0, 1645.0, 1655.0, 1665.0, 1672.0, 1677.0, 1685.0, 1695.0, 1705.0, 1715.0, 1725.0, 1735.0, 1745.0, 1755.0, 1765.0, 1775.0, 1785.0, 1795.0, 1805.0, 1815.0, 1825.0, 1835.0, 1845.0, 1855.0, 1865.0, 1875.0, 1885.0, 1895.0, 1905.0, 1915.0, 1925.0, 1935.0, 1945.0, 1955.0, 1967.0, 1976.0, 1984.0, 1995.0, 2005.0, 2015.0, 2025.0, 2035.0, 2045.0, 2055.0, 2065.0, 2074.0, 2078.0, 2085.0, 2095.0, 2105.0, 2115.0, 2125.0, 2135.0, 2145.0, 2155.0, 2165.0, 2175.0, 2185.0, 2195.0, 2205.0, 2215.0, 2225.0, 2235.0, 2245.0, 2255.0, 2265.0, 2275.0, 2285.0, 2295.0, 2305.0, 2315.0, 2325.0, 2335.0, 2345.0, 2355.0, 2365.0, 2375.0, 2385.0, 2395.0, 2405.0, 2415.0, 2425.0, 2435.0, 2445.0, 2455.0, 2465.0, 2475.0, 2485.0, 2495.0, 2505.0, 2513.0, 2518.0, 2525.0, 2535.0, 2545.0, 2555.0, 2565.0, 2575.0, 2585.0, 2595.0, 2605.0, 2615.0, 2625.0, 2635.0, 2645.0, 2655.0, 2665.0, 2675.0, 2685.0, 2695.0, 2705.0, 2715.0, 2725.0, 2735.0, 2745.0, 2755.0, 2765.0, 2775.0, 2785.0, 2795.0, 2805.0, 2815.0, 2825.0, 2835.0, 2845.0, 2855.0, 2865.0, 2875.0, 2885.0, 2895.0, 2910.0, 2930.0, 2950.0, 2970.0, 2990.0, 3010.0, 3030.0, 3050.0, 3070.0, 3090.0, 3110.0, 3130.0, 3150.0, 3170.0, 3190.0, 3210.0, 3230.0, 3250.0, 3270.0, 3290.0, 3310.0, 3330.0, 3350.0, 3370.0, 3390.0, 3410.0, 3430.0, 3450.0, 3470.0, 3490.0, 3510.0, 3530.0, 3550.0, 3570.0, 3590.0, 3601.8274, 3602.7275, 3603.6279, 3604.528, 3605.4282, 3606.3286, 3607.2287, 3608.1291, 3609.0292, 3609.9293, 3610.8297, 3611.7299, 3612.6302, 3613.5304, 3614.4305, 3615.3309, 3616.231, 3617.1314, 3618.0315, 3618.9317, 3619.832, 3620.7322, 3621.6326, 3622.5327, 3623.4328, 3624.3332, 3625.2333, 3626.1337, 3627.0339, 3627.934, 3628.8344, 3629.7345, 3630.6349, 3631.535, 3632.4351, 3633.3355, 3634.2357, 3635.136, 3636.0362, 3636.9363, 3637.8367, 3638.7368, 3639.6372, 3640.5373, 3641.4375, 3642.3379, 3643.238, 3644.1384, 3645.0385, 3645.9386, 3646.839, 3647.7391, 3648.6395, 3649.5397, 3650.4398, 3651.3402, 3652.2403, 3653.1407, 3654.0408, 3654.941, 3655.8413, 3656.7415, 3657.6419, 3658.542, 3659.4421, 3660.3425, 3661.2426, 3662.143, 3663.0432, 3663.9433, 3664.8437, 3665.7438, 3666.6442, 3667.5443, 3668.4445, 3669.3448, 3670.245, 3671.1453, 3672.0455, 3672.9456, 3673.846, 3674.7461, 3675.6465, 3676.5466, 3677.4468, 3678.3472, 3679.2473, 3680.1477, 3681.0478, 3681.9479, 3682.8483, 3683.7485, 3684.6488, 3685.549, 3686.4491, 3687.3495, 3688.2496, 3689.15, 3690.0501, 3690.9503, 3691.8507, 3692.7508, 3693.6512, 3694.5513, 3695.4514, 3696.3518, 3697.252, 3698.1523, 3699.0525, 3699.9526, 3700.853, 3701.7531, 3702.6535, 3703.5536, 3704.4538, 3705.3541, 3706.2543, 3707.1547, 3708.0548, 3708.9549, 3709.8553, 3710.7554, 3711.6558, 3712.556, 3713.4561, 3714.3565, 3715.2566, 3716.157, 3717.0571, 3717.9573, 3718.8576, 3719.7578, 3720.6582, 3721.5583, 3722.4584, 3723.3588, 3724.2589, 3725.1593, 3726.0595, 3726.9596, 3727.86, 3728.7601, 3729.6605, 3730.5606, 3731.4608, 3732.3611, 3733.2613, 3734.1617, 3735.0618, 3735.9619, 3736.8623, 3737.7624, 3738.6628, 3739.563, 3740.4631, 3741.3635, 3742.2636, 3743.164, 3744.0641, 3744.9643, 3745.8646, 3746.7648, 3747.6652, 3748.5653, 3749.4654, 3750.3658, 3751.2659, 3752.1663, 3753.0665, 3753.9666, 3754.867, 3755.7671, 3756.6675, 3757.5676, 3758.4678, 3759.3681, 3760.2683, 3761.1687, 3762.0688, 3762.9689, 3763.8693, 3764.7694, 3765.6698, 3766.57, 3767.4701, 3768.3705, 3769.2706, 3770.171, 3771.0711, 3771.9713, 3772.8716, 3773.7718, 3774.6722, 3775.5723, 3776.4724, 3777.3728, 3778.2729, 3779.1733, 3780.0735, 3780.9736, 3781.874, 3782.7741, 3783.6745, 3784.5746, 3785.4748, 3786.3751, 3787.2753, 3788.1757, 3789.0758, 3789.9759, 3790.8763, 3791.7765, 3792.6768, 3793.577, 3794.4771, 3795.3775, 3796.2776, 3797.178, 3798.0781, 3798.9783, 3799.8787, 3800.7788, 3801.6792, 3802.5793, 3803.4794, 3804.3798, 3805.28, 3806.1803, 3807.0805, 3807.9806, 3808.881, 3809.7811, 3810.6815, 3811.5816, 3812.4818, 3813.3822, 3814.2823, 3815.1827, 3816.0828, 3816.983, 3817.8833, 3818.7835, 3819.6838, 3820.584, 3821.4841, 3822.3845, 3823.2846, 3824.185, 3825.0852, 3825.9853, 3826.8857, 3827.7858, 3828.6862, 3829.5863, 3830.4865, 3831.3868, 3832.287, 3833.1874, 3834.0875, 3834.9876, 3835.888, 3836.7881, 3837.6885, 3838.5887, 3839.4888, 3840.3892, 3841.2893, 3842.1897, 3843.0898, 3843.99, 3844.8904, 3845.7905, 3846.6909, 3847.591, 3848.4911, 3849.3915, 3850.2917, 3851.192, 3852.0922, 3852.9923, 3853.8927, 3854.7928, 3855.6932, 3856.5933, 3857.4935, 3858.3939, 3859.294, 3860.1944, 3861.0945, 3861.9947, 3862.895, 3863.7952, 3864.6956, 3865.5957, 3866.4958, 3867.3962, 3868.2963, 3869.1967, 3870.0969, 3870.997, 3871.8974, 3872.7975, 3873.6979, 3874.598, 3875.4982, 3876.3985, 3877.2987, 3878.1991, 3879.0992, 3879.9993, 3880.8997, 3881.7999, 3882.7002, 3883.6004, 3884.5005, 3885.4009, 3886.301, 3887.2014, 3888.1015, 3889.0017, 3889.9021, 3890.8022, 3891.7026, 3892.6027, 3893.5029, 3894.4032, 3895.3034, 3896.2038, 3897.1039, 3898.004, 3898.9044, 3899.8045, 3900.7049, 3901.6051, 3902.5052, 3903.4056, 3904.3057, 3905.2061, 3906.1062, 3907.0064, 3907.9068, 3908.8069, 3909.7073, 3910.6074, 3911.5075, 3912.4079, 3913.3081, 3914.2084, 3915.1086, 3916.0087, 3916.9091, 3917.8092, 3918.7096, 3919.6098, 3920.5099, 3921.4103, 3922.3104, 3923.2108, 3924.1109, 3925.0111, 3925.9114, 3926.8116, 3927.712, 3928.6121, 3929.5122, 3930.4126, 3931.3128, 3932.2131, 3933.1133, 3934.0134, 3934.9138, 3935.8139, 3936.7143, 3937.6144, 3938.5146, 3939.415, 3940.3151, 3941.2155, 3942.1156, 3943.0158, 3943.9161, 3944.8163, 3945.7167, 3946.6168, 3947.5169, 3948.4173, 3949.3174, 3950.2178, 3951.118, 3952.0181, 3952.9185, 3953.8186, 3954.719, 3955.6191, 3956.5193, 3957.4197, 3958.3198, 3959.2202, 3960.1203, 3961.0205, 3961.9208, 3962.821, 3963.7214, 3964.6215, 3965.5216, 3966.422, 3967.3221, 3968.2225, 3969.1227, 3970.0228, 3970.9232, 3971.8233, 3972.7237, 3973.6238, 3974.524, 3975.4244, 3976.3245, 3977.2249, 3978.125, 3979.0252, 3979.9255, 3980.8257, 3981.7261, 3982.6262, 3983.5263, 3984.4267, 3985.3268, 3986.2272, 3987.1274, 3988.0275, 3988.9279, 3989.828, 3990.7284, 3991.6285, 3992.5287, 3993.4291, 3994.3292, 3995.2296, 3996.1297, 3997.0299, 3997.9302, 3998.8304, 3999.7308, 4000.6309, 4001.531, 4002.4314, 4003.3315, 4004.2319, 4005.1321, 4006.0322, 4006.9326, 4007.8327, 4008.7331, 4009.6332, 4010.5334, 4011.4338, 4012.3339, 4013.2343, 4014.1344, 4015.0346, 4015.9349, 4016.8351, 4017.7355, 4018.6356, 4019.5357, 4020.4361, 4021.3363, 4022.2366, 4023.1368, 4024.0369, 4024.9373, 4025.8374, 4026.7378, 4027.6379, 4028.5381, 4029.4385, 4030.3386, 4031.239, 4032.1391, 4033.0393, 4033.9396, 4034.8398, 4035.7402, 4036.6403, 4037.5404, 4038.4408, 4039.341, 4040.2413, 4041.1415, 4042.0416, 4042.942, 4043.8421, 4044.7425, 4045.6427, 4046.5428, 4047.4432, 4048.3433, 4049.2437, 4050.1438, 4051.044, 4051.9443, 4052.8445, 4053.7449, 4054.645, 4055.5451, 4056.4455, 4057.3457, 4058.246, 4059.1462, 4060.0463, 4060.9467, 4061.8468, 4062.7472, 4063.6474, 4064.5475, 4065.4479, 4066.348, 4067.2484, 4068.1485, 4069.0487, 4069.9491, 4070.8492, 4071.7496, 4072.6497, 4073.5499, 4074.4502, 4075.3504, 4076.2508, 4077.1509, 4078.051, 4078.9514, 4079.8516, 4080.7519, 4081.6521, 4082.5522, 4083.4526, 4084.3527, 4085.2531, 4086.1532, 4087.0534, 4087.9538, 4088.8539, 4089.7543, 4090.6544, 4091.5546, 4092.4549, 4093.3551, 4094.2555, 4095.1556, 4096.0557, 4096.9561, 4097.8565, 4098.7566, 4099.6568, 4100.5569, 4101.4571, 4102.3577, 4103.2578, 4104.158, 4105.0581, 4105.9582, 4106.8589, 4107.759, 4108.6591, 4109.5593, 4110.4594, 4111.36, 4112.2602, 4113.1603, 4114.0605, 4114.9606, 4115.8612, 4116.7614, 4117.6615, 4118.5616, 4119.4618, 4120.3624, 4121.2625, 4122.1627, 4123.0628, 4123.963, 4124.8636, 4125.7637, 4126.6639, 4127.564, 4128.4641, 4129.3648, 4130.2649, 4131.165, 4132.0652, 4132.9653, 4133.8659, 4134.7661, 4135.6662, 4136.5664, 4137.4665, 4138.3671, 4139.2673, 4140.1674, 4141.0675, 4141.9677, 4142.8683, 4143.7684, 4144.6686, 4145.5687, 4146.4689, 4147.3695, 4148.2696, 4149.1698, 4150.0699, 4150.97, 4151.8707, 4152.7708, 4153.6709, 4154.5711, 4155.4712, 4156.3718, 4157.272, 4158.1721, 4159.0723, 4159.9724, 4160.873, 4161.7732, 4162.6733, 4163.5734, 4164.4736, 4165.3742, 4166.2743, 4167.1745, 4168.0746, 4168.9748, 4169.8754, 4170.7755, 4171.6757, 4172.5758, 4173.4759, 4174.3766, 4175.2767, 4176.1768, 4177.077, 4177.9771, 4178.8777, 4179.7779, 4180.678, 4181.5782, 4182.4783, 4183.3789, 4184.2791, 4185.1792, 4186.0793, 4186.9795, 4187.8801, 4188.7802, 4189.6804, 4190.5805, 4191.4807, 4192.3813, 4193.2814, 4194.1816, 4195.0817, 4195.9818, 4196.8825, 4197.7826, 4198.6827, 4199.5829, 4200.483, 4201.3836, 4202.2838, 4203.1839, 4204.0841, 4204.9842, 4205.8848, 4206.785, 4207.6851, 4208.5852, 4209.4854, 4210.386, 4211.2861, 4212.1863, 4213.0864, 4213.9866, 4214.8872, 4215.7873, 4216.6875, 4217.5876, 4218.4877, 4219.3884, 4220.2885, 4221.1886, 4222.0888, 4222.9889, 4223.8895, 4224.7897, 4225.6898, 4226.59, 4227.4901, 4228.3907, 4229.2909, 4230.191, 4231.0911, 4231.9913, 4232.8919, 4233.7921, 4234.6922, 4235.5923, 4236.4925, 4237.3931, 4238.2932, 4239.1934, 4240.0935, 4240.9936, 4241.8943, 4242.7944, 4243.6946, 4244.5947, 4245.4948, 4246.3955, 4247.2956, 4248.1957, 4249.0959, 4249.996, 4250.8966, 4251.7968, 4252.6969, 4253.5971, 4254.4972, 4255.3978, 4256.298, 4257.1981, 4258.0982, 4258.9984, 4259.899, 4260.7991, 4261.6993, 4262.5994, 4263.4996, 4264.4002, 4265.3003, 4266.2005, 4267.1006, 4268.0007, 4268.9014, 4269.8015, 4270.7016, 4271.6018, 4272.5019, 4273.4026, 4274.3027, 4275.2028, 4276.103, 4277.0031, 4277.9037, 4278.8039, 4279.704, 4280.6042, 4281.5043, 4282.4049, 4283.3051, 4284.2052, 4285.1053, 4286.0055, 4286.9061, 4287.8062, 4288.7064, 4289.6065, 4290.5067, 4291.4073, 4292.3074, 4293.2076, 4294.1077, 4295.0078, 4295.9085, 4296.8086, 4297.7087, 4298.6089, 4299.509, 4300.4097, 4301.3098, 4302.2099, 4303.1101, 4304.0102, 4304.9108, 4305.811, 4306.7111, 4307.6113, 4308.5114, 4309.412, 4310.3122, 4311.2123, 4312.1124, 4313.0126, 4313.9132, 4314.8133, 4315.7135, 4316.6136, 4317.5138, 4318.4144, 4319.3145, 4320.2147, 4321.1148, 4322.0149, 4322.9156, 4323.8157, 4324.7158, 4325.616, 4326.5161, 4327.4168, 4328.3169, 4329.217, 4330.1172, 4331.0173, 4331.9179, 4332.8181, 4333.7182, 4334.6184, 4335.5185, 4336.4191, 4337.3193, 4338.2194, 4339.1195, 4340.0197, 4340.9203, 4341.8204, 4342.7206, 4343.6207, 4344.5209, 4345.4215, 4346.3216, 4347.2218, 4348.1219, 4349.022, 4349.9227, 4350.8228, 4351.723, 4352.6231, 4353.5232, 4354.4239, 4355.324, 4356.2241, 4357.1243, 4358.0244, 4358.925, 4359.8252, 4360.7253, 4361.6255, 4362.5256, 4363.4262, 4364.3264, 4365.2265, 4366.1266, 4367.0268, 4367.9274, 4368.8276, 4369.7277, 4370.6278, 4371.528, 4372.4286, 4373.3287, 4374.2289, 4375.129, 4376.0292, 4376.9298, 4377.8299, 4378.7301, 4379.6302, 4380.5303, 4381.431, 4382.3311, 4383.2312, 4384.1314, 4385.0315, 4385.9322, 4386.8323, 4387.7324, 4388.6326, 4389.5327, 4390.4333, 4391.3335, 4392.2336, 4393.1338, 4394.0339, 4394.9345, 4395.8347, 4396.7348, 4397.6349, 4398.5351, 4399.4357, 4400.3359, 4401.236, 4402.1361, 4403.0363, 4403.9369, 4404.837, 4405.7372, 4406.6373, 4407.5375, 4408.4381, 4409.3382, 4410.2384, 4411.1385, 4412.0386, 4412.9393, 4413.8394, 4414.7395, 4415.6397, 4416.5398, 4417.4405, 4418.3406, 4419.2407, 4420.1409, 4421.041, 4421.9416, 4422.8418, 4423.7419, 4424.6421, 4425.5422, 4426.4428, 4427.343, 4428.2431, 4429.1432, 4430.0434, 4430.944, 4431.8442, 4432.7443, 4433.6444, 4434.5446, 4435.4452, 4436.3453, 4437.2455, 4438.1456, 4439.0458, 4439.9464, 4440.8465, 4441.7467, 4442.6468, 4443.5469, 4444.4476, 4445.3477, 4446.2479, 4447.148, 4448.0481, 4448.9488, 4449.8489, 4450.749, 4451.6492, 4452.5493, 4453.4499, 4454.3501, 4455.2502, 4456.1504, 4457.0505, 4457.9511, 4458.8513, 4459.7514, 4460.6516, 4461.5517, 4462.4523, 4463.3525, 4464.2526, 4465.1527, 4466.0529, 4466.9535, 4467.8536, 4468.7538, 4469.6539, 4470.5541, 4471.4547, 4472.3548, 4473.255, 4474.1551, 4475.0553, 4475.9559, 4476.856, 4477.7562, 4478.6563, 4479.5564, 4480.4571, 4481.3572, 4482.2573, 4483.1575, 4484.0576, 4484.9583, 4485.8584, 4486.7585, 4487.6587, 4488.5588, 4489.4594, 4490.3596, 4491.2597, 4492.1599, 4493.06, 4493.9606, 4494.8608, 4495.7609, 4496.6611, 4497.5612, 4498.4618, 4499.362, 4500.2621, 4501.1622, 4502.0624, 4502.963, 4503.8631, 4504.7633, 4505.6634, 4506.5636, 4507.4642, 4508.3643, 4509.2645, 4510.1646, 4511.0648, 4511.9654, 4512.8655, 4513.7657, 4514.6658, 4515.5659, 4516.4666, 4517.3667, 4518.2669, 4519.167, 4520.0671, 4520.9678, 4521.8679, 4522.768, 4523.6682, 4524.5683, 4525.4689, 4526.3691, 4527.2692, 4528.1694, 4529.0695, 4529.9701, 4530.8703, 4531.7704, 4532.6706, 4533.5707, 4534.4713, 4535.3715, 4536.2716, 4537.1717, 4538.0719, 4538.9725, 4539.8727, 4540.7728, 4541.6729, 4542.5731, 4543.4737, 4544.3738, 4545.274, 4546.1741, 4547.0743, 4547.9749, 4548.875, 4549.7752, 4550.6753, 4551.5754, 4552.4761, 4553.3762, 4554.2764, 4555.1765, 4556.0766, 4556.9773, 4557.8774, 4558.7775, 4559.6777, 4560.5778, 4561.4785, 4562.3786, 4563.2787, 4564.1789, 4565.079, 4565.9796, 4566.8798, 4567.7799, 4568.6801, 4569.5802, 4570.4808, 4571.381, 4572.2811, 4573.1813, 4574.0814, 4574.982, 4575.8822, 4576.7823, 4577.6824, 4578.5826, 4579.4832, 4580.3834, 4581.2835, 4582.1836, 4583.0838, 4583.9844, 4584.8845, 4585.7847, 4586.6848, 4587.585, 4588.4856, 4589.3857, 4590.2859, 4591.186, 4592.0862, 4592.9868, 4593.8869, 4594.7871, 4595.6872, 4596.5873, 4597.488, 4598.3881, 4599.2883, 4600.1884, 4601.0885, 4601.9892, 4602.8893, 4603.7894, 4604.6896, 4605.5897, 4606.4903, 4607.3905, 4608.2906, 4609.1908, 4610.0909, 4610.9915, 4611.8917, 4612.7918, 4613.692, 4614.5921, 4615.4927, 4616.3929, 4617.293, 4618.1931, 4619.0933, 4619.9939, 4620.8941, 4621.7942, 4622.6943, 4623.5945, 4624.4951, 4625.3952, 4626.2954, 4627.1955, 4628.0957, 4628.9963, 4629.8964, 4630.7966, 4631.6967, 4632.5969, 4633.4975, 4634.3976, 4635.2978, 4636.1979, 4637.0981, 4637.9987, 4638.8988, 4639.799, 4640.6991, 4641.5992, 4642.4999, 4643.4, 4644.3002, 4645.2003, 4646.1004, 4647.0011, 4647.9012, 4648.8013, 4649.7015, 4650.6016, 4651.5023, 4652.4024, 4653.3025, 4654.2027, 4655.1028, 4656.0034, 4656.9036, 4657.8037, 4658.7039, 4659.604, 4660.5046, 4661.4048, 4662.3049, 4663.2051, 4664.1052, 4665.0058, 4665.906, 4666.8061, 4667.7062, 4668.6064, 4669.507, 4670.4072, 4671.3073, 4672.2074, 4673.1076, 4674.0082, 4674.9083, 4675.8085, 4676.7086, 4677.6088, 4678.5094, 4679.4095, 4680.3097, 4681.2098, 4682.11, 4683.0106, 4683.9107, 4684.8109, 4685.711, 4686.6111, 4687.5118, 4688.4119, 4689.3121, 4690.2122, 4691.1123, 4692.013, 4692.9131, 4693.8132, 4694.7134, 4695.6135, 4696.5142, 4697.4143, 4698.3144, 4699.2146, 4700.1147, 4701.0154, 4701.9155, 4702.8156, 4703.7158, 4704.6159, 4705.5165, 4706.4167, 4707.3168, 4708.217, 4709.1171, 4710.0177, 4710.9179, 4711.818, 4712.7182, 4713.6183, 4714.5189, 4715.4191, 4716.3192, 4717.2193, 4718.1195, 4719.0201, 4719.9203, 4720.8204, 4721.7205, 4722.6207, 4723.5213, 4724.4214, 4725.3216, 4726.2217, 4727.1219, 4728.0225, 4728.9226, 4729.8228, 4730.7229, 4731.6231, 4732.5237, 4733.4238, 4734.324, 4735.2241, 4736.1243, 4737.0249, 4737.925, 4738.8252, 4739.7253, 4740.6254, 4741.5261, 4742.4262, 4743.3264, 4744.2265, 4745.1266, 4746.0273, 4746.9274, 4747.8276, 4748.7277, 4749.6278, 4750.5285, 4751.4286, 4752.3287, 4753.2289, 4754.129, 4755.0297, 4755.9298, 4756.8299, 4757.7301, 4758.6302, 4759.5308, 4760.431, 4761.3311, 4762.2313, 4763.1314, 4764.032, 4764.9322, 4765.8323, 4766.7325, 4767.6326, 4768.5332, 4769.4334, 4770.3335, 4771.2337, 4772.1338, 4773.0344, 4773.9346, 4774.8347, 4775.7348, 4776.635, 4777.5356, 4778.4358, 4779.3359, 4780.236, 4781.1362, 4782.0368, 4782.9369, 4783.8371, 4784.7372, 4785.6374, 4786.538, 4787.4381, 4788.3383, 4789.2384, 4790.1386, 4791.0392, 4791.9393, 4792.8395, 4793.7396, 4794.6398, 4795.5404, 4796.4405, 4797.3407, 4798.2408, 4799.141, 4800.0416, 4800.9417, 4801.8419, 4802.742, 4803.6421, 4804.5428, 4805.4429, 4806.3431, 4807.2432, 4808.1433, 4809.044, 4809.9441, 4810.8442, 4811.7444, 4812.6445, 4813.5452, 4814.4453, 4815.3454, 4816.2456, 4817.1457, 4818.0464, 4818.9465, 4819.8466, 4820.7468, 4821.6469, 4822.5475, 4823.4477, 4824.3478, 4825.248, 4826.1481, 4827.0487, 4827.9489, 4828.849, 4829.7492, 4830.6493, 4831.5499, 4832.4501, 4833.3502, 4834.2504, 4835.1505, 4836.0511, 4836.9513, 4837.8514, 4838.7515, 4839.6517, 4840.5523, 4841.4525, 4842.3526, 4843.2527, 4844.1529, 4845.0535, 4845.9537, 4846.8538, 4847.7539, 4848.6541, 4849.5547, 4850.4548, 4851.355, 4852.2551, 4853.1553, 4854.0559, 4854.956, 4855.8562, 4856.7563, 4857.6565, 4858.5571, 4859.4572, 4860.3574, 4861.2575, 4862.1577, 4863.0583, 4863.9584, 4864.8586, 4865.7587, 4866.6589, 4867.5595, 4868.4596, 4869.3598, 4870.2599, 4871.16, 4872.0607, 4872.9608, 4873.861, 4874.7611, 4875.6612, 4876.5619, 4877.462, 4878.3622, 4879.2623, 4880.1624, 4881.0631, 4881.9632, 4882.8633, 4883.7635, 4884.6636, 4885.5643, 4886.4644, 4887.3645, 4888.2647, 4889.1648, 4890.0655, 4890.9656, 4891.8657, 4892.7659, 4893.666, 4894.5666, 4895.4668, 4896.3669, 4897.2671, 4898.1672, 4899.0678, 4899.968, 4900.8681, 4901.7683, 4902.6684, 4903.569, 4904.4692, 4905.3693, 4906.2695, 4907.1696, 4908.0702, 4908.9704, 4909.8705, 4910.7707, 4911.6708, 4912.5714, 4913.4716, 4914.3717, 4915.2719, 4916.172, 4917.0726, 4917.9728, 4918.8729, 4919.773, 4920.6732, 4921.5738, 4922.474, 4923.3741, 4924.2742, 4925.1744, 4926.075, 4926.9752, 4927.8753, 4928.7754, 4929.6756, 4930.5762, 4931.4763, 4932.3765, 4933.2766, 4934.1768, 4935.0774, 4935.9775, 4936.8777, 4937.7778, 4938.678, 4939.5786, 4940.4787, 4941.3789, 4942.279, 4943.1792, 4944.0798, 4944.9799, 4945.8801, 4946.7802, 4947.6804, 4948.581, 4949.4811, 4950.3813, 4951.2814, 4952.1816, 4953.0822, 4953.9823, 4954.8825, 4955.7826, 4956.6827, 4957.5834, 4958.4835, 4959.3837, 4960.2838, 4961.1839, 4962.0846, 4962.9847, 4963.8849, 4964.785, 4965.6851, 4966.5858, 4967.4859, 4968.3861, 4969.2862, 4970.1863, 4971.087, 4971.9871, 4972.8872, 4973.7874, 4974.6875, 4975.5882, 4976.4883, 4977.3884, 4978.2886, 4979.1887, 4980.0894, 4980.9895, 4981.8896, 4982.7898, 4983.6899, 4984.5905, 4985.4907, 4986.3908, 4987.291, 4988.1911, 4989.0917, 4989.9919, 4990.892, 4991.7922, 4992.6923, 4993.5929, 4994.4931, 4995.3932, 4996.2934, 4997.1935, 4998.0941, 4998.9943, 4999.8944, 5000.7946, 5001.6947, 5002.5953, 5003.4955, 5004.3956, 5005.2958, 5006.1959, 5007.0965, 5007.9967, 5008.8968, 5009.797, 5010.6971, 5011.5977, 5012.4979, 5013.398, 5014.2981, 5015.1983, 5016.0989, 5016.9991, 5017.8992, 5018.7993, 5019.6995, 5020.6001, 5021.5003, 5022.4004, 5023.3005, 5024.2007, 5025.1013, 5026.0015, 5026.9016, 5027.8017, 5028.7019, 5029.6025, 5030.5027, 5031.4028, 5032.3029, 5033.2031, 5034.1037, 5035.0038, 5035.904, 5036.8041, 5037.7043, 5038.6049, 5039.505, 5040.4052, 5041.3053, 5042.2055, 5043.1061, 5044.0062, 5044.9064, 5045.8065, 5046.7067, 5047.6073, 5048.5074, 5049.4076, 5050.3077, 5051.2079, 5052.1085, 5053.0086, 5053.9088, 5054.8089, 5055.7091, 5056.6097, 5057.5098, 5058.41, 5059.3101, 5060.2103, 5061.1109, 5062.011, 5062.9112, 5063.8113, 5064.7115, 5065.6121, 5066.5122, 5067.4124, 5068.3125, 5069.2126, 5070.1133, 5071.0134, 5071.9136, 5072.8137, 5073.7138, 5074.6145, 5075.5146, 5076.4148, 5077.3149, 5078.215, 5079.1157, 5080.0158, 5080.916, 5081.8161, 5082.7162, 5083.6169, 5084.517, 5085.4172, 5086.3173, 5087.2174, 5088.1181, 5089.0182, 5089.9184, 5090.8185, 5091.7186, 5092.6193, 5093.5194, 5094.4195, 5095.3197, 5096.2198, 5097.1205, 5098.0206, 5098.9207, 5099.8209, 5100.721, 5101.6217, 5102.5218, 5103.4219, 5104.3221, 5105.2222, 5106.1229, 5107.023, 5107.9231, 5108.8233, 5109.7234, 5110.6241, 5111.5242, 5112.4243, 5113.3245, 5114.2246, 5115.1253, 5116.0254, 5116.9255, 5117.8257, 5118.7258, 5119.6264, 5120.5266, 5121.4267, 5122.3269, 5123.227, 5124.1276, 5125.0278, 5125.9279, 5126.8281, 5127.7282, 5128.6288, 5129.529, 5130.4291, 5131.3293, 5132.2294, 5133.13, 5134.0302, 5134.9303, 5135.8305, 5136.7306, 5137.6312, 5138.5314, 5139.4315, 5140.3317, 5141.2318, 5142.1324, 5143.0326, 5143.9327, 5144.8329, 5145.733, 5146.6336, 5147.5338, 5148.4339, 5149.3341, 5150.2342, 5151.1348, 5152.035, 5152.9351, 5153.8353, 5154.7354, 5155.636, 5156.5362, 5157.4363, 5158.3365, 5159.2366, 5160.1372, 5161.0374, 5161.9375, 5162.8377, 5163.7378, 5164.6384, 5165.5386, 5166.4387, 5167.3389, 5168.239, 5169.1396, 5170.0398, 5170.9399, 5171.8401, 5172.7402, 5173.6408, 5174.541, 5175.4411, 5176.3412, 5177.2414, 5178.142, 5179.0422, 5179.9423, 5180.8424, 5181.7426, 5182.6432, 5183.5434, 5184.4435, 5185.3436, 5186.2438, 5187.1444, 5188.0446, 5188.9447, 5189.8448, 5190.745, 5191.6456, 5192.5458, 5193.4459, 5194.346, 5195.2462, 5196.1468, 5197.047, 5197.9471, 5198.8472, 5199.7474, 5200.648, 5201.5482, 5202.4483, 5203.3484, 5204.2486, 5205.1492, 5206.0494, 5206.9495, 5207.8496, 5208.7498, 5209.6504, 5210.5506, 5211.4507, 5212.3508, 5213.251, 5214.1516, 5215.0518, 5215.9519, 5216.852, 5217.7522, 5218.6528, 5219.5529, 5220.4531, 5221.3532, 5222.2534, 5223.154, 5224.0541, 5224.9543, 5225.8544, 5226.7546, 5227.6552, 5228.5553, 5229.4555, 5230.3556, 5231.2558, 5232.1564, 5233.0565, 5233.9567, 5234.8568, 5235.757, 5236.6576, 5237.5577, 5238.4579, 5239.358, 5240.2582, 5241.1588, 5242.0589, 5242.9591, 5243.8592, 5244.7594, 5245.66, 5246.5601, 5247.4603, 5248.3604, 5249.2606, 5250.1612, 5251.0613, 5251.9615, 5252.8616, 5253.7618, 5254.6624, 5255.5625, 5256.4627, 5257.3628, 5258.263, 5259.1636, 5260.0637, 5260.9639, 5261.864, 5262.7642, 5263.6648, 5264.5649, 5265.4651, 5266.3652, 5267.2654, 5268.166, 5269.0661, 5269.9663, 5270.8664, 5271.7666, 5272.6672, 5273.5673, 5274.4675, 5275.3676, 5276.2678, 5277.1684, 5278.0685, 5278.9687, 5279.8688, 5280.769, 5281.6696, 5282.5697, 5283.4699, 5284.37, 5285.2702, 5286.1708, 5287.0709, 5287.9711, 5288.8712, 5289.7714, 5290.672, 5291.5721, 5292.4723, 5293.3724, 5294.2726, 5295.1732, 5296.0733, 5296.9735, 5297.8736, 5298.7738, 5299.6744, 5300.5745, 5301.4747, 5302.3748, 5303.275, 5304.1756, 5305.0757, 5305.9759, 5306.876, 5307.7762, 5308.6768, 5309.5769, 5310.4771, 5311.3772, 5312.2774, 5313.178, 5314.0781, 5314.9783, 5315.8784, 5316.7786, 5317.6792, 5318.5793, 5319.4795, 5320.3796, 5321.2798, 5322.1804, 5323.0805, 5323.9807, 5324.8808, 5325.781, 5326.6816, 5327.5817, 5328.4819, 5329.382, 5330.2822, 5331.1828, 5332.0829, 5332.9831, 5333.8832, 5334.7834, 5335.684, 5336.5841, 5337.4843, 5338.3844, 5339.2846, 5340.1852, 5341.0853, 5341.9855, 5342.8856, 5343.7858, 5344.6864, 5345.5865, 5346.4867, 5347.3868, 5348.287, 5349.1876, 5350.0877, 5350.9879, 5351.888, 5352.7882, 5353.6888, 5354.5889, 5355.4891, 5356.3892, 5357.2894, 5358.19, 5359.0901, 5359.9903, 5360.8904, 5361.7906, 5362.6912, 5363.5913, 5364.4915, 5365.3916, 5366.2918, 5367.1924, 5368.0925, 5368.9927, 5369.8928, 5370.793, 5371.6936, 5372.5937, 5373.4939, 5374.394, 5375.2942, 5376.1948, 5377.0949, 5377.9951, 5378.8952, 5379.7954, 5380.696, 5381.5961, 5382.4963, 5383.3964, 5384.2966, 5385.1972, 5386.0973, 5386.9975, 5387.8976, 5388.7978, 5389.6984, 5390.5985, 5391.4987, 5392.3988, 5393.299, 5394.1996, 5395.0997, 5395.9999, 5396.9, 5397.8002, 5398.7008, 5399.6009, 5400.5011, 5401.4012, 5402.3014, 5403.202, 5404.1021, 5405.0023, 5405.9024, 5406.8026, 5407.7032, 5408.6033, 5409.5035, 5410.4036, 5411.3038, 5412.2044, 5413.1045, 5414.0047, 5414.9048, 5415.805, 5416.7056, 5417.6057, 5418.5059, 5419.406, 5420.3062, 5421.2068, 5422.1069, 5423.0071, 5423.9072, 5424.8074, 5425.708, 5426.6081, 5427.5083, 5428.4084, 5429.3086, 5430.2092, 5431.1093, 5432.0095, 5432.9096, 5433.8098, 5434.7104, 5435.6105, 5436.5107, 5437.4108, 5438.311, 5439.2116, 5440.1117, 5441.0119, 5441.912, 5442.8122, 5443.7128, 5444.6129, 5445.5131, 5446.4132, 5447.3134, 5448.214, 5449.1142, 5450.0143, 5450.9144, 5451.8146, 5452.7152, 5453.6154, 5454.5155, 5455.4156, 5456.3158, 5457.2164, 5458.1166, 5459.0167, 5459.9168, 5460.817, 5461.7176, 5462.6178, 5463.5179, 5464.418, 5465.3182, 5466.2188, 5467.119, 5468.0191, 5468.9192, 5469.8194, 5470.72, 5471.6202, 5472.5203, 5473.4204, 5474.3206, 5475.2212, 5476.1214, 5477.0215, 5477.9216, 5478.8218, 5479.7224, 5480.6226, 5481.5227, 5482.4228, 5483.323, 5484.2236, 5485.1238, 5486.0239, 5486.924, 5487.8242, 5488.7248, 5489.625, 5490.5251, 5491.4252, 5492.3254, 5493.226, 5494.1262, 5495.0263, 5495.9265, 5496.8266, 5497.7272, 5498.6274, 5499.5275, 5500.4277, 5501.3278, 5502.2284, 5503.1286, 5504.0287, 5504.9289, 5505.829, 5506.7296, 5507.6298, 5508.5299, 5509.4301, 5510.3302, 5511.2308, 5512.131, 5513.0311, 5513.9313, 5514.8314, 5515.732, 5516.6322, 5517.5323, 5518.4325, 5519.3326, 5520.2332, 5521.1334, 5522.0335, 5522.9337, 5523.8338, 5524.7344, 5525.6346, 5526.5347, 5527.4349, 5528.335, 5529.2356, 5530.1358, 5531.0359, 5531.9361, 5532.8362, 5533.7368, 5534.637, 5535.5371, 5536.4373, 5537.3374, 5538.238, 5539.1382, 5540.0383, 5540.9385, 5541.8386, 5542.7392, 5543.6394, 5544.5395, 5545.4397, 5546.3398, 5547.2404, 5548.1406, 5549.0407, 5549.9409, 5550.841, 5551.7416, 5552.6418, 5553.5419, 5554.4421, 5555.3422, 5556.2428, 5557.143, 5558.0431, 5558.9433, 5559.8434, 5560.7441, 5561.6442, 5562.5443, 5563.4445, 5564.3446, 5565.2453, 5566.1454, 5567.0455, 5567.9457, 5568.8458, 5569.7465, 5570.6466, 5571.5467, 5572.4469, 5573.347, 5574.2477, 5575.1478, 5576.0479, 5576.9481, 5577.8482, 5578.7489, 5579.649, 5580.5491, 5581.4493, 5582.3494, 5583.2501, 5584.1502, 5585.0504, 5585.9505, 5586.8506, 5587.7513, 5588.6514, 5589.5516, 5590.4517, 5591.3518, 5592.2525, 5593.1526, 5594.0528, 5594.9529, 5595.853, 5596.7537, 5597.6538, 5598.554, 5599.4541, 5600.3542, 5601.2549, 5602.155, 5603.0552, 5603.9553, 5604.8554, 5605.7561, 5606.6562, 5607.5564, 5608.4565, 5609.3566, 5610.2573, 5611.1574, 5612.0576, 5612.9577, 5613.8579, 5614.7585, 5615.6586, 5616.5588, 5617.4589, 5618.3591, 5619.2597, 5620.1598, 5621.06, 5621.9601, 5622.8603, 5623.7609, 5624.661, 5625.5612, 5626.4613, 5627.3615, 5628.2621, 5629.1622, 5630.0624, 5630.9625, 5631.8627, 5632.7633, 5633.6634, 5634.5636, 5635.4637, 5636.3639, 5637.2645, 5638.1646, 5639.0648, 5639.9649, 5640.8651, 5641.7657, 5642.6658, 5643.566, 5644.4661, 5645.3663, 5646.2669, 5647.167, 5648.0672, 5648.9673, 5649.8675, 5650.7681, 5651.6683, 5652.5684, 5653.4685, 5654.3687, 5655.2693, 5656.1695, 5657.0696, 5657.9697, 5658.8699, 5659.7705, 5660.6707, 5661.5708, 5662.4709, 5663.3711, 5664.2717, 5665.1719, 5666.072, 5666.9721, 5667.8723, 5668.7729, 5669.6731, 5670.5732, 5671.4733, 5672.3735, 5673.2741, 5674.1743, 5675.0744, 5675.9746, 5676.8747, 5677.7753, 5678.6755, 5679.5756, 5680.4758, 5681.3759, 5682.2765, 5683.1767, 5684.0768, 5684.977, 5685.8771, 5686.7777, 5687.6779, 5688.578, 5689.4782, 5690.3783, 5691.2789, 5692.1791, 5693.0792, 5693.9794, 5694.8795, 5695.7801, 5696.6803, 5697.5804, 5698.4806, 5699.3807, 5700.2813, 5701.1815, 5702.0816, 5702.9818, 5703.8819, 5704.7825, 5705.6827, 5706.5828, 5707.483, 5708.3831, 5709.2838, 5710.1839, 5711.084, 5711.9842, 5712.8843, 5713.785, 5714.6851, 5715.5852, 5716.4854, 5717.3855, 5718.2862, 5719.1863, 5720.0864, 5720.9866, 5721.8867, 5722.7874, 5723.6875, 5724.5876, 5725.4878, 5726.3879, 5727.2886, 5728.1887, 5729.0889, 5729.989, 5730.8891, 5731.7898, 5732.6899, 5733.5901, 5734.4902, 5735.3903, 5736.291, 5737.1911, 5738.0913, 5738.9914, 5739.8915, 5740.7922, 5741.6923, 5742.5925, 5743.4926, 5744.3928, 5745.2934, 5746.1935, 5747.0937, 5747.9938, 5748.894, 5749.7946, 5750.6947, 5751.5949, 5752.495, 5753.3952, 5754.2958, 5755.1959, 5756.0961, 5756.9962, 5757.8964, 5758.797, 5759.6971, 5760.5973, 5761.4974, 5762.3976, 5763.2982, 5764.1983, 5765.0985, 5765.9986, 5766.8988, 5767.7994, 5768.6995, 5769.5997, 5770.4998, 5771.4, 5772.3006, 5773.2008, 5774.1009, 5775.001, 5775.9012, 5776.8018, 5777.702, 5778.6021, 5779.5022, 5780.4024, 5781.303, 5782.2032, 5783.1033, 5784.0034, 5784.9036, 5785.8042, 5786.7044, 5787.6045, 5788.5047, 5789.4048, 5790.3054, 5791.2056, 5792.1057, 5793.0059, 5793.906, 5794.8066, 5795.7068, 5796.6069, 5797.5071, 5798.4072, 5799.3078, 5800.208, 5801.1081, 5802.0083, 5802.9084, 5803.809, 5804.7092, 5805.6093, 5806.5095, 5807.4096, 5808.3102, 5809.2104, 5810.1105, 5811.0107, 5811.9108, 5812.8114, 5813.7116, 5814.6117, 5815.5119, 5816.412, 5817.3127, 5818.2128, 5819.1129, 5820.0131, 5820.9132, 5821.8139, 5822.714, 5823.6141, 5824.5143, 5825.4144, 5826.3151, 5827.2152, 5828.1153, 5829.0155, 5829.9156, 5830.8163, 5831.7164, 5832.6166, 5833.5167, 5834.4168, 5835.3175, 5836.2176, 5837.1178, 5838.0179, 5838.918, 5839.8187, 5840.7188, 5841.619, 5842.5191, 5843.4192, 5844.3199, 5845.22, 5846.1202, 5847.0203, 5847.9205, 5848.8211, 5849.7212, 5850.6214, 5851.5215, 5852.4217, 5853.3223, 5854.2224, 5855.1226, 5856.0227, 5856.9229, 5857.8235, 5858.7236, 5859.6238, 5860.5239, 5861.4241, 5862.3247, 5863.2248, 5864.125, 5865.0251, 5865.9253, 5866.8259, 5867.726, 5868.6262, 5869.5263, 5870.4265, 5871.3271, 5872.2273, 5873.1274, 5874.0275, 5874.9277, 5875.8283, 5876.7285, 5877.6286, 5878.5287, 5879.4289, 5880.3295, 5881.2297, 5882.1298, 5883.03, 5883.9301, 5884.8307, 5885.7309, 5886.631, 5887.5312, 5888.4313, 5889.3319, 5890.2321, 5891.1322, 5892.0324, 5892.9325, 5893.8331, 5894.7333, 5895.6334, 5896.5336, 5897.4337, 5898.3343, 5899.2345, 5900.1346, 5901.0348, 5901.9349, 5902.8355, 5903.7357, 5904.6358, 5905.536, 5906.4361, 5907.3368, 5908.2369, 5909.137, 5910.0372, 5910.9373, 5911.838, 5912.7381, 5913.6382, 5914.5384, 5915.4385, 5916.3392, 5917.2393, 5918.1394, 5919.0396, 5919.9397, 5920.8404, 5921.7405, 5922.6407, 5923.5408, 5924.4409, 5925.3416, 5926.2417, 5927.1419, 5928.042, 5928.9421, 5929.8428, 5930.7429, 5931.6431, 5932.5432, 5933.4434, 5934.344, 5935.2441, 5936.1443, 5937.0444, 5937.9446, 5938.8452, 5939.7453, 5940.6455, 5941.5456, 5942.4458, 5943.3464, 5944.2465, 5945.1467, 5946.0468, 5946.947, 5947.8476, 5948.7477, 5949.6479, 5950.548, 5951.4482, 5952.3488, 5953.2489, 5954.1491, 5955.0492, 5955.9494, 5956.85, 5957.7502, 5958.6503, 5959.5504, 5960.4506, 5961.3512, 5962.2514, 5963.1515, 5964.0516, 5964.9518, 5965.8524, 5966.7526, 5967.6527, 5968.5529, 5969.453, 5970.3536, 5971.2538, 5972.1539, 5973.0541, 5973.9542, 5974.8548, 5975.755, 5976.6551, 5977.5553, 5978.4554, 5979.356, 5980.2562, 5981.1563, 5982.0565, 5982.9566, 5983.8572, 5984.7574, 5985.6575, 5986.5577, 5987.4578, 5988.3585, 5989.2586, 5990.1587, 5991.0589, 5991.959, 5992.8597, 5993.7598, 5994.6599, 5995.5601, 5996.4602, 5997.3609, 5998.261, 5999.1612, 6000.0613, 6000.9614, 6001.8621, 6002.7622, 6003.6624, 6004.5625, 6005.4626, 6006.3633, 6007.2634, 6008.1636, 6009.0637, 6009.9639, 6010.8645, 6011.7646, 6012.6648, 6013.5649, 6014.4651, 6015.3657, 6016.2658, 6017.166, 6018.0661, 6018.9663, 6019.8669, 6020.767, 6021.6672, 6022.5673, 6023.4675, 6024.3681, 6025.2682, 6026.1684, 6027.0685, 6027.9687, 6028.8693, 6029.7695, 6030.6696, 6031.5697, 6032.4699, 6033.3705, 6034.2707, 6035.1708, 6036.0709, 6036.9711, 6037.8717, 6038.7719, 6039.672, 6040.5722, 6041.4723, 6042.3729, 6043.2731, 6044.1732, 6045.0734, 6045.9735, 6046.8741, 6047.7743, 6048.6744, 6049.5746, 6050.4747, 6051.3753, 6052.2755, 6053.1756, 6054.0758, 6054.9759, 6055.8765, 6056.7767, 6057.6768, 6058.577, 6059.4771, 6060.3778, 6061.2779, 6062.178, 6063.0782, 6063.9783, 6064.879, 6065.7791, 6066.6792, 6067.5794, 6068.4795, 6069.3802, 6070.2803, 6071.1805, 6072.0806, 6072.9807, 6073.8814, 6074.7815, 6075.6817, 6076.5818, 6077.4819, 6078.3826, 6079.2827, 6080.1829, 6081.083, 6081.9832, 6082.8838, 6083.7839, 6084.6841, 6085.5842, 6086.4844, 6087.385, 6088.2851, 6089.1853, 6090.0854, 6090.9856, 6091.8862, 6092.7863, 6093.6865, 6094.5866, 6095.4868, 6096.3874, 6097.2875, 6098.1877, 6099.0878, 6099.988, 6100.8886, 6101.7888, 6102.6889, 6103.589, 6104.4892, 6105.3898, 6106.29, 6107.1901, 6108.0902, 6108.9904, 6109.891, 6110.7912, 6111.6913, 6112.5915, 6113.4916, 6114.3922, 6115.2924, 6116.1925, 6117.0927, 6117.9928, 6118.8934, 6119.7936, 6120.6937, 6121.5939, 6122.494, 6123.3946, 6124.2948, 6125.1949, 6126.0951, 6126.9952, 6127.8959, 6128.796, 6129.6961, 6130.5963, 6131.4964, 6132.3971, 6133.2972, 6134.1973, 6135.0975, 6135.9976, 6136.8983, 6137.7984, 6138.6986, 6139.5987, 6140.4988, 6141.3995, 6142.2996, 6143.1998, 6144.0999, 6145.0, 6145.9007, 6146.8008, 6147.701, 6148.6011, 6149.5013, 6150.4019, 6151.302, 6152.2022, 6153.1023, 6154.0025, 6154.9031, 6155.8032, 6156.7034, 6157.6035, 6158.5037, 6159.4043, 6160.3044, 6161.2046, 6162.1047, 6163.0049, 6163.9055, 6164.8057, 6165.7058, 6166.6059, 6167.5061, 6168.4067, 6169.3069, 6170.207, 6171.1071, 6172.0073, 6172.9079, 6173.8081, 6174.7082, 6175.6084, 6176.5085, 6177.4091, 6178.3093, 6179.2094, 6180.1096, 6181.0097, 6181.9103, 6182.8105, 6183.7106, 6184.6108, 6185.5109, 6186.4115, 6187.3117, 6188.2118, 6189.112, 6190.0121, 6190.9128, 6191.8129, 6192.713, 6193.6132, 6194.5133, 6195.414, 6196.3141, 6197.2142, 6198.1144, 6199.0145, 6199.9152, 6200.8153, 6201.7155, 6202.6156, 6203.5157, 6204.4164, 6205.3165, 6206.2167, 6207.1168, 6208.017, 6208.9176, 6209.8177, 6210.7179, 6211.618, 6212.5182, 6213.4188, 6214.3189, 6215.2191, 6216.1192, 6217.0194, 6217.92, 6218.8201, 6219.7203, 6220.6204, 6221.5206, 6222.4212, 6223.3213, 6224.2215, 6225.1216, 6226.0218, 6226.9224, 6227.8226, 6228.7227, 6229.6228, 6230.523, 6231.4236, 6232.3238, 6233.2239, 6234.1241, 6235.0242, 6235.9248, 6236.825, 6237.7251, 6238.6253, 6239.5254, 6240.426, 6241.3262, 6242.2263, 6243.1265, 6244.0266, 6244.9272, 6245.8274, 6246.7275, 6247.6277, 6248.5278, 6249.4285, 6250.3286, 6251.2287, 6252.1289, 6253.029, 6253.9297, 6254.8298, 6255.7299, 6256.6301, 6257.5302, 6258.4309, 6259.331, 6260.2312, 6261.1313, 6262.0314, 6262.9321, 6263.8322, 6264.7324, 6265.6325, 6266.5327, 6267.4333, 6268.3334, 6269.2336, 6270.1337, 6271.0339, 6271.9345, 6272.8346, 6273.7348, 6274.6349, 6275.5351, 6276.4357, 6277.3358, 6278.236, 6279.1361, 6280.0363, 6280.9369, 6281.8371, 6282.7372, 6283.6373, 6284.5375, 6285.4381, 6286.3383, 6287.2384, 6288.1385, 6289.0387, 6289.9393, 6290.8395, 6291.7396, 6292.6398, 6293.5399, 6294.4405, 6295.3407, 6296.2408, 6297.141, 6298.0411, 6298.9417, 6299.8419, 6300.742, 6301.6422, 6302.5423, 6303.4429, 6304.3431, 6305.2432, 6306.1434, 6307.0435, 6307.9442, 6308.8443, 6309.7444, 6310.6446, 6311.5447, 6312.4454, 6313.3455, 6314.2457, 6315.1458, 6316.0459, 6316.9466, 6317.8467, 6318.7469, 6319.647, 6320.5471, 6321.4478, 6322.3479, 6323.2481, 6324.1482, 6325.0484, 6325.949, 6326.8491, 6327.7493, 6328.6494, 6329.5496, 6330.4502, 6331.3503, 6332.2505, 6333.1506, 6334.0508, 6334.9514, 6335.8515, 6336.7517, 6337.6518, 6338.552, 6339.4526, 6340.3528, 6341.2529, 6342.153, 6343.0532, 6343.9538, 6344.854, 6345.7541, 6346.6543, 6347.5544, 6348.455, 6349.3552, 6350.2553, 6351.1555, 6352.0556, 6352.9562, 6353.8564, 6354.7565, 6355.6567, 6356.5568, 6357.4574, 6358.3576, 6359.2577, 6360.1579, 6361.058, 6361.9587, 6362.8588, 6363.7589, 6364.6591, 6365.5592, 6366.4599, 6367.36, 6368.2602, 6369.1603, 6370.0604, 6370.9611, 6371.8612, 6372.7614, 6373.6615, 6374.5616, 6375.4623, 6376.3624, 6377.2626, 6378.1627, 6379.0629, 6379.9635, 6380.8636, 6381.7638, 6382.6639, 6383.5641, 6384.4647, 6385.3648, 6386.265, 6387.1651, 6388.0653, 6388.9659, 6389.866, 6390.7662, 6391.6663, 6392.5665, 6393.4671, 6394.3673, 6395.2674, 6396.1675, 6397.0677, 6397.9683, 6398.8685, 6399.7686, 6400.6688, 6401.5689, 6402.4695, 6403.3697, 6404.2698, 6405.17, 6406.0701, 6406.9707, 6407.8709, 6408.771, 6409.6712, 6410.5713, 6411.4719, 6412.3721, 6413.2722, 6414.1724, 6415.0725, 6415.9732, 6416.8733, 6417.7734, 6418.6736, 6419.5737, 6420.4744, 6421.3745, 6422.2747, 6423.1748, 6424.0749, 6424.9756, 6425.8757, 6426.7759, 6427.676, 6428.5762, 6429.4768, 6430.3769, 6431.2771, 6432.1772, 6433.0774, 6433.978, 6434.8781, 6435.7783, 6436.6784, 6437.5786, 6438.4792, 6439.3793, 6440.2795, 6441.1796, 6442.0798, 6442.9804, 6443.8806, 6444.7807, 6445.6808, 6446.581, 6447.4816, 6448.3818, 6449.2819, 6450.1821, 6451.0822, 6451.9828, 6452.883, 6453.7831, 6454.6833, 6455.5834, 6456.484, 6457.3842, 6458.2843, 6459.1845, 6460.0846, 6460.9852, 6461.8854, 6462.7855, 6463.6857, 6464.5858, 6465.4865, 6466.3866, 6467.2867, 6468.1869, 6469.087, 6469.9877, 6470.8878, 6471.788, 6472.6881, 6473.5882, 6474.4889, 6475.389, 6476.2892, 6477.1893, 6478.0895, 6478.9901, 6479.8902, 6480.7904, 6481.6905, 6482.5907, 6483.4913, 6484.3914, 6485.2916, 6486.1917, 6487.0919, 6487.9925, 6488.8926, 6489.7928, 6490.6929, 6491.5931, 6492.4937, 6493.3939, 6494.294, 6495.1941, 6496.0943, 6496.9949, 6497.8951, 6498.7952, 6499.6954, 6500.5955, 6501.4961, 6502.3963, 6503.2964, 6504.1966, 6505.0967, 6505.9973, 6506.8975, 6507.7976, 6508.6978, 6509.5979, 6510.4985, 6511.3987, 6512.2988, 6513.199, 6514.0991, 6514.9998, 6515.8999, 6516.8, 6517.7002, 6518.6003, 6519.501, 6520.4011, 6521.3013, 6522.2014, 6523.1015, 6524.0022, 6524.9023, 6525.8025, 6526.7026, 6527.6028, 6528.5034, 6529.4035, 6530.3037, 6531.2038, 6532.104, 6533.0046, 6533.9047, 6534.8049, 6535.705, 6536.6052, 6537.5058, 6538.4059, 6539.3061, 6540.2062, 6541.1064, 6542.007, 6542.9072, 6543.8073, 6544.7074, 6545.6076, 6546.5082, 6547.4084, 6548.3085, 6549.2087, 6550.1088, 6551.0094, 6551.9096, 6552.8097, 6553.7099, 6554.61, 6555.5106, 6556.4108, 6557.3109, 6558.2111, 6559.1112, 6560.0119, 6560.912, 6561.8121, 6562.7123, 6563.6124, 6564.5131, 6565.4132, 6566.3134, 6567.2135, 6568.1136, 6569.0143, 6569.9144, 6570.8146, 6571.7147, 6572.6148, 6573.5155, 6574.4156, 6575.3158, 6576.2159, 6577.1161, 6578.0167, 6578.9168, 6579.817, 6580.7171, 6581.6173, 6582.5179, 6583.418, 6584.3182, 6585.2183, 6586.1185, 6587.0191, 6587.9193, 6588.8194, 6589.7195, 6590.6197, 6591.5203, 6592.4205, 6593.3206, 6594.2208, 6595.1209, 6596.0215, 6596.9217, 6597.8218, 6598.722, 6599.6221, 6600.5227, 6601.4229, 6602.323, 6603.2232, 6604.1233, 6605.024, 6605.9241, 6606.8242, 6607.7244, 6608.6245, 6609.5252, 6610.4253, 6611.3254, 6612.2256, 6613.1257, 6614.0264, 6614.9265, 6615.8267, 6616.7268, 6617.6269, 6618.5276, 6619.4277, 6620.3279, 6621.228, 6622.1282, 6623.0288, 6623.9289, 6624.8291, 6625.7292, 6626.6294, 6627.53, 6628.4301, 6629.3303, 6630.2304, 6631.1306, 6632.0312, 6632.9314, 6633.8315, 6634.7316, 6635.6318, 6636.5324, 6637.4326, 6638.3327, 6639.2329, 6640.133, 6641.0336, 6641.9338, 6642.8339, 6643.7341, 6644.6342, 6645.5348, 6646.435, 6647.3351, 6648.2353, 6649.1354, 6650.0361, 6650.9362, 6651.8363, 6652.7365, 6653.6366, 6654.5373, 6655.4374, 6656.3375, 6657.2377, 6658.1378, 6659.0385, 6659.9386, 6660.8388, 6661.7389, 6662.639, 6663.5397, 6664.4398, 6665.34, 6666.2401, 6667.1403, 6668.0409, 6668.941, 6669.8412, 6670.7413, 6671.6415, 6672.5421, 6673.4422, 6674.3424, 6675.2425, 6676.1427, 6677.0433, 6677.9435, 6678.8436, 6679.7437, 6680.6439, 6681.5445, 6682.4447, 6683.3448, 6684.245, 6685.1451, 6686.0457, 6686.9459, 6687.846, 6688.7462, 6689.6463, 6690.5469, 6691.4471, 6692.3472, 6693.2474, 6694.1475, 6695.0482, 6695.9483, 6696.8484, 6697.7486, 6698.6487, 6699.5494, 6700.4495, 6701.3497, 6702.2498, 6703.1499, 6704.0506, 6704.9507, 6705.8509, 6706.751, 6707.6512, 6708.5518, 6709.4519, 6710.3521, 6711.2522, 6712.1524, 6713.053, 6713.9531, 6714.8533, 6715.7534, 6716.6536, 6717.5542, 6718.4543, 6719.3545, 6720.2546, 6721.1548, 6722.0554, 6722.9556, 6723.8557, 6724.7558, 6725.656, 6726.5566, 6727.4568, 6728.3569, 6729.2571, 6730.1572, 6731.0578, 6731.958, 6732.8581, 6733.7583, 6734.6584, 6735.559, 6736.4592, 6737.3593, 6738.2595, 6739.1596, 6740.0603, 6740.9604, 6741.8605, 6742.7607, 6743.6608, 6744.5615, 6745.4616, 6746.3618, 6747.2619, 6748.162, 6749.0627, 6749.9628, 6750.863, 6751.7631, 6752.6633, 6753.5639, 6754.464, 6755.3642, 6756.2643, 6757.1645, 6758.0651, 6758.9652, 6759.8654, 6760.7655, 6761.6657, 6762.5663, 6763.4665, 6764.3666, 6765.2667, 6766.1669, 6767.0675, 6767.9677, 6768.8678, 6769.768, 6770.6681, 6771.5687, 6772.4689, 6773.369, 6774.2692, 6775.1693, 6776.0699, 6776.9701, 6777.8702, 6778.7704, 6779.6705, 6780.5712, 6781.4713, 6782.3714, 6783.2716, 6784.1717, 6785.0724, 6785.9725, 6786.8727, 6787.7728, 6788.6729, 6789.5736, 6790.4737, 6791.3739, 6792.274, 6793.1742, 6794.0748, 6794.9749, 6795.8751, 6796.7752, 6797.6754, 6798.576, 6799.4761, 6800.3763, 6801.2764, 6802.1766, 6803.0772, 6803.9774, 6804.8775, 6805.7776, 6806.6778, 6807.5784, 6808.4786, 6809.3787, 6810.2789, 6811.179, 6812.0796, 6812.9798, 6813.8799, 6814.7801, 6815.6802, 6816.5808, 6817.481, 6818.3811, 6819.2813, 6820.1814, 6821.0821, 6821.9822, 6822.8823, 6823.7825, 6824.6826, 6825.5833, 6826.4834, 6827.3836, 6828.2837, 6829.1838, 6830.0845, 6830.9846, 6831.8848, 6832.7849, 6833.6851, 6834.5857, 6835.4858, 6836.386, 6837.2861, 6838.1863, 6839.0869, 6839.987, 6840.8872, 6841.7873, 6842.6875, 6843.5881, 6844.4883, 6845.3884, 6846.2885, 6847.1887, 6848.0893, 6848.9895, 6849.8896, 6850.7898, 6851.6899, 6852.5905, 6853.4907, 6854.3908, 6855.291, 6856.1911, 6857.0917, 6857.9919, 6858.892, 6859.7922, 6860.6923, 6861.593, 6862.4931, 6863.3932, 6864.2934, 6865.1935, 6866.0942, 6866.9943, 6867.8945, 6868.7946, 6869.6947, 6870.5954, 6871.4955, 6872.3957, 6873.2958, 6874.196, 6875.0966, 6875.9967, 6876.8969, 6877.797, 6878.6972, 6879.5978, 6880.4979, 6881.3981, 6882.2982, 6883.1984, 6884.099, 6884.9992, 6885.8993, 6886.7994, 6887.6996, 6888.6002, 6889.5004, 6890.4005, 6891.3007, 6892.2008, 6893.1014, 6894.0016, 6894.9017, 6895.8019, 6896.702, 6897.6026, 6898.5028, 6899.4029, 6900.3031, 6901.2032, 6902.1039, 6903.004, 6903.9041, 6904.8043, 6905.7044, 6906.6051, 6907.5052, 6908.4054, 6909.3055, 6910.2056, 6911.1063, 6912.0064, 6912.9066, 6913.8067, 6914.7069, 6915.6075, 6916.5076, 6917.4078, 6918.3079, 6919.2081, 6920.1087, 6921.0088, 6921.909, 6922.8091, 6923.7093, 6924.6099, 6925.5101, 6926.4102, 6927.3103, 6928.2105, 6929.1111, 6930.0113, 6930.9114, 6931.8116, 6932.7117, 6933.6123, 6934.5125, 6935.4126, 6936.3128, 6937.2129, 6938.1135, 6939.0137, 6939.9138, 6940.814, 6941.7141, 6942.6148, 6943.5149, 6944.415, 6945.3152, 6946.2153, 6947.116, 6948.0161, 6948.9163, 6949.8164, 6950.7165, 6951.6172, 6952.5173, 6953.4175, 6954.3176, 6955.2178, 6956.1184, 6957.0185, 6957.9187, 6958.8188, 6959.719, 6960.6196, 6961.5197, 6962.4199, 6963.32, 6964.2202, 6965.1208, 6966.021, 6966.9211, 6967.8212, 6968.7214, 6969.622, 6970.5222, 6971.4223, 6972.3225, 6973.2226, 6974.1232, 6975.0234, 6975.9235, 6976.8237, 6977.7238, 6978.6244, 6979.5246, 6980.4247, 6981.3249, 6982.225, 6983.1257, 6984.0258, 6984.9259, 6985.8261, 6986.7262, 6987.6269, 6988.527, 6989.4272, 6990.3273, 6991.2275, 6992.1281, 6993.0282, 6993.9284, 6994.8285, 6995.7287, 6996.6293, 6997.5294, 6998.4296, 6999.3297, 7000.2299, 7001.1305, 7002.0307, 7002.9308, 7003.8309, 7004.7311, 7005.6317, 7006.5319, 7007.432, 7008.3322, 7009.2323, 7010.1329, 7011.0331, 7011.9332, 7012.8334, 7013.7335, 7014.6341, 7015.5343, 7016.4344, 7017.3346, 7018.2347, 7019.1354, 7020.0355, 7020.9356, 7021.8358, 7022.7359, 7023.6366, 7024.5367, 7025.4369, 7026.337, 7027.2371, 7028.1378, 7029.0379, 7029.9381, 7030.8382, 7031.7384, 7032.639, 7033.5391, 7034.4393, 7035.3394, 7036.2396, 7037.1402, 7038.0403, 7038.9405, 7039.8406, 7040.7408, 7041.6414, 7042.5416, 7043.4417, 7044.3418, 7045.242, 7046.1426, 7047.0428, 7047.9429, 7048.8431, 7049.7432, 7050.6438, 7051.544, 7052.4441, 7053.3443, 7054.2444, 7055.1451, 7056.0452, 7056.9453, 7057.8455, 7058.7456, 7059.6463, 7060.5464, 7061.4466, 7062.3467, 7063.2468, 7064.1475, 7065.0476, 7065.9478, 7066.8479, 7067.7481, 7068.6487, 7069.5488, 7070.449, 7071.3491, 7072.2493, 7073.1499, 7074.05, 7074.9502, 7075.8503, 7076.7505, 7077.6511, 7078.5513, 7079.4514, 7080.3515, 7081.2517, 7082.1523, 7083.0525, 7083.9526, 7084.8528, 7085.7529, 7086.6535, 7087.5537, 7088.4538, 7089.354, 7090.2541, 7091.1547, 7092.0549, 7092.955, 7093.8552, 7094.7553, 7095.656, 7096.5561, 7097.4562, 7098.3564, 7099.2565, 7100.1572, 7101.0573, 7101.9575, 7102.8576, 7103.7578, 7104.6584, 7105.5585, 7106.4587, 7107.3588, 7108.259, 7109.1596, 7110.0597, 7110.9599, 7111.86, 7112.7602, 7113.6608, 7114.561, 7115.4611, 7116.3612, 7117.2614, 7118.162, 7119.0622, 7119.9623, 7120.8625, 7121.7626, 7122.6632, 7123.5634, 7124.4635, 7125.3637, 7126.2638, 7127.1644, 7128.0646, 7128.9647, 7129.8649, 7130.765, 7131.6657, 7132.5658, 7133.4659, 7134.3661, 7135.2662, 7136.1669, 7137.067, 7137.9672, 7138.8673, 7139.7674, 7140.6681, 7141.5682, 7142.4684, 7143.3685, 7144.2687, 7145.1693, 7146.0694, 7146.9696, 7147.8697, 7148.7699, 7149.6705, 7150.5707, 7151.4708, 7152.3709, 7153.2711, 7154.1717, 7155.0719, 7155.972, 7156.8722, 7157.7723, 7158.6729, 7159.5731, 7160.4732, 7161.3734, 7162.2735, 7163.1741, 7164.0743, 7164.9744, 7165.8746, 7166.7747, 7167.6754, 7168.5755, 7169.4756, 7170.3758, 7171.2759, 7172.1766, 7173.0767, 7173.9769, 7174.877, 7175.7771, 7176.6778, 7177.5779, 7178.4781, 7179.3782, 7180.2784, 7181.179, 7182.0791, 7182.9793, 7183.8794, 7184.7796, 7185.6802, 7186.5804, 7187.4805, 7188.3806, 7189.2808, 7190.1814, 7191.0816, 7191.9817, 7192.8819, 7193.782, 7194.6826, 7195.5828, 7196.4829, 7197.3831, 7198.2832, 7199.1838, 7200.084, 7200.9841, 7201.8843, 7202.7844, 7203.6851, 7204.5852, 7205.4853, 7206.3855, 7207.2856, 7208.1863, 7209.0864, 7209.9866, 7210.8867, 7211.7869, 7212.6875, 7213.5876, 7214.4878, 7215.3879, 7216.2881, 7217.1887, 7218.0888, 7218.989, 7219.8891, 7220.7893, 7221.6899, 7222.5901, 7223.4902, 7224.3903, 7225.2905, 7226.1911, 7227.0913, 7227.9914, 7228.8916, 7229.7917, 7230.6923, 7231.5925, 7232.4926, 7233.3928, 7234.2929, 7235.1935, 7236.0937, 7236.9938, 7237.894, 7238.7941, 7239.6948, 7240.5949, 7241.4951, 7242.3952, 7243.2953, 7244.196, 7245.0961, 7245.9963, 7246.8964, 7247.7966, 7248.6972, 7249.5973, 7250.4975, 7251.3976, 7252.2978, 7253.1984, 7254.0985, 7254.9987, 7255.8988, 7256.799, 7257.6996, 7258.5998, 7259.4999, 7260.4, 7261.3002, 7262.2008, 7263.101, 7264.0011, 7264.9013, 7265.8014, 7266.702, 7267.6022, 7268.5023, 7269.4025, 7270.3026, 7271.2033, 7272.1034, 7273.0035, 7273.9037, 7274.8038, 7275.7045, 7276.6046, 7277.5048, 7278.4049, 7279.305, 7280.2057, 7281.1058, 7282.006, 7282.9061, 7283.8063, 7284.7069, 7285.607, 7286.5072, 7287.4073, 7288.3075, 7289.2081, 7290.1082, 7291.0084, 7291.9085, 7292.8087, 7293.7093, 7294.6095, 7295.5096, 7296.4098, 7297.3099, 7298.2105, 7299.1107, 7300.0108, 7300.911, 7301.8111, 7302.7117, 7303.6119, 7304.512, 7305.4122, 7306.3123, 7307.213, 7308.1131, 7309.0132, 7309.9134, 7310.8135, 7311.7142, 7312.6143, 7313.5145, 7314.4146, 7315.3147, 7316.2154, 7317.1155, 7318.0157, 7318.9158, 7319.816, 7320.7166, 7321.6167, 7322.5169, 7323.417, 7324.3172, 7325.2178, 7326.118, 7327.0181, 7327.9182, 7328.8184, 7329.719, 7330.6192, 7331.5193, 7332.4195, 7333.3196, 7334.2202, 7335.1204, 7336.0205, 7336.9207, 7337.8208, 7338.7214, 7339.6216, 7340.5217, 7341.4219, 7342.322, 7343.2227, 7344.1228, 7345.023, 7345.9231, 7346.8232, 7347.7239, 7348.624, 7349.5242, 7350.4243, 7351.3245, 7352.2251, 7353.1252, 7354.0254, 7354.9255, 7355.8257, 7356.7263, 7357.6264, 7358.5266, 7359.4267, 7360.3269, 7361.2275, 7362.1277, 7363.0278, 7363.9279, 7364.8281, 7365.7287, 7366.6289, 7367.529, 7368.4292, 7369.3293, 7370.2299, 7371.1301, 7372.0302, 7372.9304, 7373.8305, 7374.7312, 7375.6313, 7376.5314, 7377.4316, 7378.3317, 7379.2324, 7380.1325, 7381.0327, 7381.9328, 7382.8329, 7383.7336, 7384.6337, 7385.5339, 7386.434, 7387.3342, 7388.2348, 7389.1349, 7390.0351, 7390.9352, 7391.8354, 7392.736, 7393.6362, 7394.5363, 7395.4364, 7396.3366, 7397.2372, 7398.1374, 7399.0375, 7399.9377, 7400.8378, 7401.7384, 7410.0, 7430.0, 7450.0, 7470.0, 7490.0, 7510.0, 7530.0, 7550.0, 7570.0, 7590.0, 7610.0, 7630.0, 7650.0, 7670.0, 7690.0, 7710.0, 7730.0, 7750.0, 7770.0, 7790.0, 7810.0, 7830.0, 7850.0, 7870.0, 7890.0, 7910.0, 7930.0, 7950.0, 7970.0, 7990.0, 8010.0, 8030.0, 8050.0, 8070.0, 8090.0, 8110.0, 8130.0, 8150.0, 8170.0, 8190.0, 8210.0, 8230.0, 8250.0, 8270.0, 8290.0, 8310.0, 8330.0, 8350.0, 8370.0, 8390.0, 8410.0, 8430.0, 8450.0, 8470.0, 8490.0, 8510.0, 8530.0, 8550.0, 8570.0, 8590.0, 8610.0, 8630.0, 8650.0, 8670.0, 8690.0, 8710.0, 8730.0, 8750.0, 8770.0, 8790.0, 8810.0, 8830.0, 8850.0, 8870.0, 8890.0, 8910.0, 8930.0, 8950.0, 8970.0, 8990.0, 9010.0, 9030.0, 9050.0, 9070.0, 9090.0, 9110.0, 9130.0, 9150.0, 9170.0, 9190.0, 9210.0, 9230.0, 9250.0, 9270.0, 9290.0, 9310.0, 9330.0, 9350.0, 9370.0, 9390.0, 9410.0, 9430.0, 9450.0, 9470.0, 9490.0, 9510.0, 9530.0, 9550.0, 9570.0, 9590.0, 9610.0, 9630.0, 9650.0, 9670.0, 9690.0, 9710.0, 9730.0, 9750.0, 9770.0, 9790.0, 9810.0, 9830.0, 9850.0, 9870.0, 9890.0, 9910.0, 9930.0, 9950.0, 9970.0, 9990.0, 10025.0, 10075.0, 10125.0, 10175.0, 10225.0, 10275.0, 10325.0, 10375.0, 10425.0, 10475.0, 10525.0, 10575.0, 10625.0, 10675.0, 10725.0, 10775.0, 10825.0, 10875.0, 10925.0, 10975.0, 11025.0, 11075.0, 11125.0, 11175.0, 11225.0, 11275.0, 11325.0, 11375.0, 11425.0, 11475.0, 11525.0, 11575.0, 11625.0, 11675.0, 11725.0, 11775.0, 11825.0, 11875.0, 11925.0, 11975.0, 12025.0, 12075.0, 12125.0, 12175.0, 12225.0, 12275.0, 12325.0, 12375.0, 12425.0, 12475.0, 12525.0, 12575.0, 12625.0, 12675.0, 12725.0, 12775.0, 12825.0, 12875.0, 12925.0, 12975.0, 13025.0, 13075.0, 13125.0, 13175.0, 13225.0, 13275.0, 13325.0, 13375.0, 13425.0, 13475.0, 13525.0, 13575.0, 13625.0, 13675.0, 13725.0, 13775.0, 13825.0, 13875.0, 13925.0, 13975.0, 14025.0, 14075.0, 14125.0, 14175.0, 14225.0, 14275.0, 14325.0, 14375.0, 14425.0, 14475.0, 14525.0, 14570.0, 14620.0, 14675.0, 14725.0, 14775.0, 14825.0, 14875.0, 14925.0, 14975.0, 15025.0, 15075.0, 15125.0, 15175.0, 15225.0, 15275.0, 15325.0, 15375.0, 15425.0, 15475.0, 15525.0, 15575.0, 15625.0, 15675.0, 15725.0, 15775.0, 15825.0, 15875.0, 15925.0, 15975.0, 16050.0, 16150.0, 16250.0, 16350.0, 16450.0, 16550.0, 16650.0, 16750.0, 16850.0, 16950.0, 17050.0, 17150.0, 17250.0, 17350.0, 17450.0, 17550.0, 17650.0, 17750.0, 17850.0, 17950.0, 18050.0, 18150.0, 18250.0, 18350.0, 18450.0, 18550.0, 18650.0, 18750.0, 18850.0, 18950.0, 19050.0, 19150.0, 19250.0, 19350.0, 19450.0, 19550.0, 19650.0, 19750.0, 19850.0, 19950.0, 20050.0, 20150.0, 20250.0, 20350.0, 20450.0, 20550.0, 20650.0, 20750.0, 20850.0, 20950.0, 21050.0, 21150.0, 21250.0, 21350.0, 21450.0, 21550.0, 21650.0, 21750.0, 21850.0, 21950.0, 22050.0, 22150.0, 22250.0, 22350.0, 22450.0, 22550.0, 22650.0, 22750.0, 22850.0, 22950.0, 23050.0, 23150.0, 23250.0, 23350.0, 23450.0, 23550.0, 23650.0, 23750.0, 23850.0, 23950.0, 24050.0, 24150.0, 24250.0, 24350.0, 24450.0, 24550.0, 24650.0, 24750.0, 24850.0, 24950.0, 25050.0, 25150.0, 25250.0, 25350.0, 25450.0, 25550.0, 25650.0, 25750.0, 25850.0, 25950.0, 26050.0, 26150.0, 26250.0, 26350.0, 26450.0, 26550.0, 26650.0, 26750.0, 26850.0, 26950.0, 27050.0, 27150.0, 27250.0, 27350.0, 27450.0, 27550.0, 27650.0, 27750.0, 27850.0, 27950.0, 28050.0, 28150.0, 28250.0, 28350.0, 28450.0, 28550.0, 28650.0, 28750.0, 28850.0, 28950.0, 29050.0, 29150.0, 29250.0, 29350.0, 29450.0, 29550.0, 29650.0, 29750.0, 29850.0, 29950.0, 30050.0, 30150.0, 30250.0, 30350.0, 30450.0, 30550.0, 30650.0, 30750.0, 30850.0, 30950.0, 31050.0, 31150.0, 31250.0, 31350.0, 31450.0, 31550.0, 31650.0, 31750.0, 31850.0, 31950.0, 32100.0, 32300.0, 32500.0, 32700.0, 32900.0, 33100.0, 33300.0, 33500.0, 33700.0, 33900.0, 34100.0, 34300.0, 34500.0, 34700.0, 34900.0, 35100.0, 35300.0, 35500.0, 35700.0, 35900.0, 36100.0, 36300.0, 36500.0, 36700.0, 36900.0, 37100.0, 37300.0, 37500.0, 37700.0, 37900.0, 38100.0, 38300.0, 38500.0, 38700.0, 38900.0, 39100.0, 39300.0, 39500.0, 39700.0, 39900.0, 40100.0, 40300.0, 40500.0, 40700.0, 40900.0, 41100.0, 41300.0, 41500.0, 41700.0, 41900.0, 42100.0, 42300.0, 42500.0, 42700.0, 42900.0, 43100.0, 43300.0, 43500.0, 43700.0, 43900.0, 44100.0, 44300.0, 44500.0, 44700.0, 44900.0, 45100.0, 45300.0, 45500.0, 45700.0, 45900.0, 46100.0, 46300.0, 46500.0, 46700.0, 46900.0, 47100.0, 47300.0, 47500.0, 47700.0, 47900.0, 48100.0, 48300.0, 48500.0, 48700.0, 48900.0, 49100.0, 49300.0, 49500.0, 49700.0, 49900.0, 50100.0, 50300.0, 50500.0, 50700.0, 50900.0, 51100.0, 51300.0, 51500.0, 51700.0, 51900.0, 52100.0, 52300.0, 52500.0, 52700.0, 52900.0, 53100.0, 53300.0, 53500.0, 53700.0, 53900.0, 54100.0, 54300.0, 54500.0, 54700.0, 54900.0, 55100.0, 55300.0, 55500.0, 55700.0, 55900.0, 56100.0, 56300.0, 56500.0, 56700.0, 56900.0, 57100.0, 57300.0, 57500.0, 57700.0, 57900.0, 58100.0, 58300.0, 58500.0, 58700.0, 58900.0, 59100.0, 59300.0, 59500.0, 59700.0, 59900.0, 60100.0, 60300.0, 60500.0, 60700.0, 60900.0, 61100.0, 61300.0, 61500.0, 61700.0, 61900.0, 62100.0, 62300.0, 62500.0, 62700.0, 62900.0, 63100.0, 63300.0, 63500.0, 63700.0, 63900.0, 64200.0, 64600.0, 65000.0, 65400.0, 65800.0, 66200.0, 66600.0, 67000.0, 67400.0, 67800.0, 68200.0, 68600.0, 69000.0, 69400.0, 69800.0, 70200.0, 70600.0, 71000.0, 71400.0, 71800.0, 72200.0, 72600.0, 73000.0, 73400.0, 73800.0, 74200.0, 74600.0, 75000.0, 75400.0, 75800.0, 76200.0, 76600.0, 77000.0, 77400.0, 77800.0, 78200.0, 78600.0, 79000.0, 79400.0, 79800.0, 80200.0, 80600.0, 81000.0, 81400.0, 81800.0, 82200.0, 82600.0, 83000.0, 83400.0, 83800.0, 84200.0, 84600.0, 85000.0, 85400.0, 85800.0, 86200.0, 86600.0, 87000.0, 87400.0, 87800.0, 88200.0, 88600.0, 89000.0, 89400.0, 89800.0, 90200.0, 90600.0, 91000.0, 91400.0, 91800.0, 92200.0, 92600.0, 93000.0, 93400.0, 93800.0, 94200.0, 94600.0, 95000.0, 95400.0, 95800.0, 96200.0, 96600.0, 97000.0, 97400.0, 97800.0, 98200.0, 98600.0, 99000.0, 99400.0, 99800.0, 100000.0, 100900.0, 101900.0, 102800.0, 103800.0, 104700.0, 105700.0, 106700.0, 107600.0, 108600.0, 109600.0, 110700.0, 111700.0, 112700.0, 113800.0, 114800.0, 115900.0, 116900.0, 118000.0, 119100.0, 120200.0, 121300.0, 122500.0, 123600.0, 124700.0, 125900.0, 127100.0, 128200.0, 129400.0, 130600.0, 131800.0, 133000.0, 134300.0, 135500.0, 136800.0, 138000.0, 139300.0, 140600.0, 141900.0, 143200.0, 144500.0, 145900.0, 147200.0, 148600.0, 150000.0, 151400.0, 152800.0, 154200.0, 155600.0, 157000.0, 158500.0, 160000.0, 161400.0, 162900.0, 164400.0, 166000.0, 167500.0, 169000.0, 170600.0, 172200.0, 173800.0, 175400.0, 177000.0, 178600.0, 180300.0, 182000.0, 183700.0, 185400.0, 187100.0, 188800.0, 190500.0, 192300.0, 194100.0, 195900.0, 197700.0, 199500.0, 201400.0, 203200.0, 205100.0, 207000.0, 208900.0, 210900.0, 212800.0, 214800.0, 216800.0, 218800.0, 220800.0, 222800.0, 224900.0, 227000.0, 229100.0, 231200.0, 233300.0, 235500.0, 237700.0, 239900.0, 242100.0, 244300.0, 246600.0, 248900.0, 251200.0, 253500.0, 255900.0, 258200.0, 260600.0, 263000.0, 265500.0, 267900.0, 270400.0, 272900.0, 275400.0, 278000.0, 280500.0, 283100.0, 285800.0, 288400.0, 291100.0, 293800.0, 296500.0, 299200.0, 302000.0, 304800.0, 307600.0, 310500.0, 313300.0, 316200.0, 319200.0, 322100.0, 325100.0, 328100.0, 331100.0, 334200.0, 337300.0, 340400.0, 343600.0, 346700.0, 349900.0, 353200.0, 356500.0, 359700.0, 363100.0, 366400.0, 369800.0, 373300.0, 376700.0, 380200.0, 383700.0, 387300.0, 390800.0, 394500.0, 398100.0, 401800.0, 405500.0, 409300.0, 413000.0, 416900.0, 420700.0, 424600.0, 428500.0, 432500.0, 436500.0, 440600.0, 444600.0, 448700.0, 452900.0, 457100.0, 461300.0, 465600.0, 469900.0, 474200.0, 478600.0, 483100.0, 487500.0, 492000.0, 496600.0, 501200.0, 505800.0, 510500.0, 515200.0, 520000.0, 524800.0, 529700.0, 534600.0, 539500.0, 544500.0, 549500.0, 554600.0, 559800.0, 564900.0, 570200.0, 575400.0, 580800.0, 586100.0, 591600.0, 597000.0, 602600.0, 608100.0, 613800.0, 619400.0, 625200.0, 631000.0, 636800.0, 642700.0, 648600.0, 654600.0, 660700.0, 666800.0, 673000.0, 679200.0, 685500.0, 691800.0, 698200.0, 704700.0, 711200.0, 717800.0, 724400.0, 731100.0, 737900.0, 744700.0, 751600.0, 758600.0, 765600.0, 772700.0, 779800.0, 787000.0, 794300.0, 801700.0, 809100.0, 816600.0, 824100.0, 831800.0, 839500.0, 847200.0, 855100.0, 863000.0, 871000.0, 879000.0, 887200.0, 895400.0, 903600.0, 912000.0, 920400.0, 929000.0, 937600.0, 946200.0, 955000.0, 963800.0, 972700.0, 981700.0, 990800.0, 1000000.0, 1009000.0, 1019000.0, 1028000.0, 1038000.0, 1047000.0, 1057000.0, 1067000.0, 1076000.0, 1086000.0, 1096000.0, 1107000.0, 1117000.0, 1127000.0, 1138000.0, 1148000.0, 1159000.0, 1169000.0, 1180000.0, 1191000.0, 1202000.0, 1213000.0, 1225000.0, 1236000.0, 1247000.0, 1259000.0, 1271000.0, 1282000.0, 1294000.0, 1306000.0, 1318000.0, 1330000.0, 1343000.0, 1355000.0, 1368000.0, 1380000.0, 1393000.0, 1406000.0, 1419000.0, 1432000.0, 1445000.0, 1459000.0, 1472000.0, 1486000.0, 1500000.0, 1514000.0, 1528000.0, 1542000.0, 1556000.0, 1570000.0, 1585000.0, 1600000.0, 1614000.0, 1629000.0, 1644000.0, 1660000.0, 1675000.0, 1690000.0, 1706000.0, 1722000.0, 1738000.0, 1754000.0, 1770000.0, 1786000.0, 1803000.0, 1820000.0, 1837000.0, 1854000.0, 1871000.0, 1888000.0, 1905000.0, 1923000.0, 1941000.0, 1959000.0, 1977000.0, 1995000.0, 2014000.0, 2032000.0, 2051000.0, 2070000.0, 2089000.0, 2109000.0, 2128000.0, 2148000.0, 2168000.0, 2188000.0, 2208000.0, 2228000.0, 2249000.0, 2270000.0, 2291000.0, 2312000.0, 2333000.0, 2355000.0, 2377000.0, 2399000.0, 2421000.0, 2443000.0, 2466000.0, 2489000.0, 2512000.0, 2535000.0, 2559000.0, 2582000.0, 2606000.0, 2630000.0, 2655000.0, 2679000.0, 2704000.0, 2729000.0, 2754000.0, 2780000.0, 2805000.0, 2831000.0, 2858000.0, 2884000.0, 2911000.0, 2938000.0, 2965000.0, 2992000.0, 3020000.0, 3048000.0, 3076000.0, 3105000.0, 3133000.0, 3162000.0, 3192000.0, 3221000.0, 3251000.0, 3281000.0, 3311000.0, 3342000.0, 3373000.0, 3404000.0, 3436000.0, 3467000.0, 3499000.0, 3532000.0, 3565000.0, 3597000.0, 3631000.0, 3664000.0, 3698000.0, 3733000.0, 3767000.0, 3802000.0, 3837000.0, 3873000.0, 3908000.0, 3945000.0, 3981000.0, 4018000.0, 4055000.0, 4093000.0, 4130000.0, 4169000.0, 4207000.0, 4246000.0, 4285000.0, 4325000.0, 4365000.0, 4406000.0, 4446000.0, 4487000.0, 4529000.0, 4571000.0, 4613000.0, 4656000.0, 4699000.0, 4742000.0, 4786000.0, 4831000.0, 4875000.0, 4920000.0, 4966000.0, 5012000.0, 5058000.0, 5105000.0, 5152000.0, 5200000.0, 5248000.0, 5297000.0, 5346000.0, 5395000.0, 5445000.0, 5495000.0, 5546000.0, 5598000.0, 5649000.0, 5702000.0, 5754000.0, 5808000.0, 5861000.0, 5916000.0, 5970000.0, 6026000.0, 6081000.0, 6138000.0, 6194000.0, 6252000.0, 6310000.0, 6368000.0, 6427000.0, 6486000.0, 6546000.0, 6607000.0, 6668000.0, 6730000.0, 6792000.0, 6855000.0, 6918000.0, 6982000.0, 7047000.0, 7112000.0, 7178000.0, 7244000.0, 7311000.0, 7379000.0, 7447000.0, 7516000.0, 7586000.0, 7656000.0, 7727000.0, 7798000.0, 7870000.0, 7943000.0, 8017000.0, 8091000.0, 8166000.0, 8241000.0, 8318000.0, 8395000.0, 8472000.0, 8551000.0, 8630000.0, 8710000.0, 8790000.0, 8872000.0, 8954000.0, 9036000.0, 9120000.0, 9204000.0, 9290000.0, 9376000.0, 9462000.0, 9550000.0, 9638000.0, 9727000.0, 9817000.0, 9908000.0, 10000000.0, 10090000.0, 10190000.0, 10280000.0, 10380000.0, 10470000.0, 10570000.0, 10670000.0, 10760000.0, 10860000.0, 10960000.0, 11070000.0, 11170000.0, 11270000.0, 11380000.0, 11480000.0, 11590000.0, 11690000.0, 11800000.0, 11910000.0, 12020000.0, 12130000.0, 12250000.0, 12360000.0, 12470000.0, 12590000.0, 12710000.0, 12820000.0, 12940000.0, 13060000.0, 13180000.0, 13300000.0, 13430000.0, 13550000.0, 13680000.0, 13800000.0, 13930000.0, 14060000.0, 14190000.0, 14320000.0, 14450000.0, 14590000.0, 14720000.0, 14860000.0, 15000000.0, 15140000.0, 15280000.0, 15420000.0, 15560000.0, 15700000.0, 15850000.0, 16000000.0, 16140000.0, 16290000.0, 16440000.0, 16600000.0, 16750000.0, 16900000.0, 17060000.0, 17220000.0, 17380000.0, 17540000.0, 17700000.0, 17860000.0, 18030000.0, 18200000.0, 18370000.0, 18540000.0, 18710000.0, 18880000.0, 19050000.0, 19230000.0, 19410000.0, 19590000.0, 19770000.0, 19950000.0, 20140000.0, 20320000.0, 20510000.0, 20700000.0, 20890000.0, 21090000.0, 21280000.0, 21480000.0, 21680000.0, 21880000.0, 22080000.0, 22280000.0, 22490000.0, 22700000.0, 22910000.0, 23120000.0, 23330000.0, 23550000.0, 23770000.0, 23990000.0, 24210000.0, 24430000.0, 24660000.0, 24890000.0, 25120000.0, 25350000.0, 25590000.0, 25820000.0, 26060000.0, 26300000.0, 26550000.0, 26790000.0, 27040000.0, 27290000.0, 27540000.0, 27800000.0, 28050000.0, 28310000.0, 28580000.0, 28840000.0, 29110000.0, 29380000.0, 29650000.0, 29920000.0, 30200000.0, 30480000.0, 30760000.0, 31050000.0, 31330000.0, 31620000.0, 31920000.0, 32210000.0, 32510000.0, 32810000.0, 33110000.0, 33420000.0, 33730000.0, 34040000.0, 34360000.0, 34670000.0, 34990000.0, 35320000.0, 35650000.0, 35970000.0, 36310000.0, 36640000.0, 36980000.0, 37330000.0, 37670000.0, 38020000.0, 38370000.0, 38730000.0, 39080000.0, 39450000.0, 39810000.0, 40180000.0, 40550000.0, 40930000.0, 41300000.0, 41690000.0, 42070000.0, 42460000.0, 42850000.0, 43250000.0, 43650000.0, 44060000.0, 44460000.0, 44870000.0, 45290000.0, 45710000.0, 46130000.0, 46560000.0, 46990000.0, 47420000.0, 47860000.0, 48310000.0, 48750000.0, 49200000.0, 49660000.0, 50120000.0, 50580000.0, 51050000.0, 51520000.0, 52000000.0, 52480000.0, 52970000.0, 53460000.0, 53950000.0, 54450000.0, 54950000.0, 55460000.0, 55980000.0, 56490000.0, 57020000.0, 57540000.0, 58080000.0, 58610000.0, 59160000.0, 59700000.0, 60260000.0, 60810000.0, 61380000.0, 61940000.0, 62520000.0, 63100000.0, 63680000.0, 64270000.0, 64860000.0, 65460000.0, 66070000.0, 66680000.0, 67300000.0, 67920000.0, 68550000.0, 69180000.0, 69820000.0, 70470000.0, 71120000.0, 71780000.0, 72440000.0, 73110000.0, 73790000.0, 74470000.0, 75160000.0, 75860000.0, 76560000.0, 77270000.0, 77980000.0, 78700000.0, 79430000.0, 80170000.0, 80910000.0, 81660000.0, 82410000.0, 83180000.0, 83950000.0, 84720000.0, 85510000.0, 86300000.0, 87100000.0, 87900000.0, 88720000.0, 89540000.0, 90360000.0, 91200000.0, 92040000.0, 92900000.0, 93760000.0, 94620000.0, 95500000.0, 96380000.0, 97270000.0, 98170000.0, 99080000.0, 100000000.0]
	return wave


def csp_spec_restframe_fit(sp=None,sfh_form=4,formed_mass=1.0,age=0.0,
	tau=0.0,t0=0.0,alpha=0.0,beta=0.0):
	"""A function for generating model spectrum of an CSP

	:param sp:
		Initialization of FSPS, such as sp=fsps.StellarPopulation()

	:param sfh_form: (default: 4)
		Choice for the parametric SFH model. Options are: (a) 0 for exponentially declining or tau model, (b) 1 for delayed tau model, (c) 2 for log normal model, (d) 3 for Gaussian form, (e) 4 for double power-law model.

	:param formed_mass:
		The total stellar mass formed.
	"""

	# make grid of SFH:
	sfh_t, sfh_sfr = construct_SFH(sfh_form=sfh_form,t0=t0,tau=tau,alpha=alpha,beta=beta,age=age,formed_mass=formed_mass)

	# exclude nan or inf
	idx = np.where((np.isnan(sfh_sfr)==True) | (np.isinf(sfh_sfr)==True) | (sfh_sfr<0))
	if len(idx[0]) == len(sfh_sfr):
		SFR = 1.0e-35
		mass = formed_mass
		wave = rest_wave_fsps()
		spec = np.zeros(len(wave))
		dust_mass = 1.0e-35

	else:
		sfh_sfr[idx[0]] = 0

		if np.any(sfh_sfr > 1.0e-33) == False:
			SFR = 1.0e-35
			mass = formed_mass
			wave = rest_wave_fsps()
			spec = np.zeros(len(wave))
			dust_mass = 1.0e-35
		else:
			sp.params["sfh"] = 3
			sp.set_tabular_sfh(sfh_t, sfh_sfr)
			wave, spec0 = sp.get_spectrum(peraa=True,tage=age) ## spectrum in L_sun/AA
			
			mass0 = sp.stellar_mass
			dust_mass0 = sp.dust_mass

			norm = formed_mass/mass0
			spec = spec0*norm
			dust_mass = dust_mass0*norm

			SFR = sp.sfr
			mass = formed_mass

	return SFR,mass,wave,spec,dust_mass


def get_dust_mass_othSFH_fit(sp=None,imf_type=1,sfh_form=4,params_fsps=['logzsol', 'log_tau', 'log_age', 
	'dust_index', 'dust1', 'dust2', 'log_gamma', 'log_umin', 'log_qpah','log_fagn', 'log_tauagn'], params_val={'log_mass':0.0,
	'z':0.001,'log_fagn':-3.0,'log_tauagn':1.0,'log_qpah':0.54,'log_umin':0.0,'log_gamma':-2.0,'dust1':0.5,'dust2':0.5,
	'dust_index':-0.7,'log_age':1.0,'log_alpha':0.1,'log_beta':0.1,'log_t0':0.4,'log_tau':0.4,'logzsol':0.0}):
	
	params_assoc_fsps = {'logzsol':"logzsol", 'log_tau':"tau", 'log_age':"tage", 
					'dust_index':"dust_index", 'dust1':"dust1", 'dust2':"dust2",
					'log_gamma':"duste_gamma", 'log_umin':"duste_umin", 
					'log_qpah':"duste_qpah",'log_fagn':"fagn", 'log_tauagn':"agn_tau"}
	status_log = {'logzsol':0, 'log_tau':1, 'log_age':1, 'dust_index':0, 'dust1':0, 'dust2':0,
				'log_gamma':1, 'log_umin':1, 'log_qpah':1,'log_fagn':1, 'log_tauagn':1}

	# get stellar mass:
	formed_mass = pow(10.0,params_val['log_mass'])
	t0 = pow(10.0,params_val['log_t0'])
	tau = pow(10.0,params_val['log_tau'])
	age = pow(10.0,params_val['log_age'])
	alpha = pow(10.0,params_val['log_alpha'])
	beta = pow(10.0,params_val['log_beta'])

	# input model parameters to FSPS:
	nparams_fsps = len(params_fsps)
	for pp in range(0,nparams_fsps):
		str_temp = params_assoc_fsps[params_fsps[pp]]
		if status_log[params_fsps[pp]] == 0:
			sp.params[str_temp] = params_val[params_fsps[pp]]
		elif status_log[params_fsps[pp]] == 1:
			sp.params[str_temp] = pow(10.0,params_val[params_fsps[pp]])

	sp.params['imf_type'] = imf_type
	sp.params['gas_logz'] = params_val['logzsol']

	# make grid of SFH:
	sfh_t, sfh_sfr = construct_SFH(sfh_form=sfh_form,t0=t0,tau=tau,alpha=alpha,
										beta=beta,age=age,formed_mass=formed_mass)

	# exclude nan or inf
	idx = np.where((np.isnan(sfh_sfr)==True) | (np.isinf(sfh_sfr)==True) | (sfh_sfr<0))
	if len(idx[0]) == len(sfh_sfr):
		dust_mass = 1.0e-35
	else:
		sfh_sfr[idx[0]] = 0
		if np.any(sfh_sfr > 1.0e-33) == False:
			dust_mass = 1.0e-35
		else:
			sp.params["sfh"] = 3
			sp.set_tabular_sfh(sfh_t, sfh_sfr)
			#wave, spec0 = sp.get_spectrum(peraa=True,tage=age) ## spectrum in L_sun/AA
			mass0 = sp.stellar_mass
			dust_mass0 = sp.dust_mass
			norm = formed_mass/mass0
			dust_mass = dust_mass0*norm

	return dust_mass


def get_sfr_dust_mass_fagnbol_othSFH_fit(sp=None,imf_type=1,sfh_form=4,params_fsps=['logzsol', 'log_tau', 'log_age', 
	'dust_index', 'dust1', 'dust2', 'log_gamma', 'log_umin', 'log_qpah','log_fagn', 'log_tauagn'],params_val={'log_mass':0.0,
	'z':0.001,'log_fagn':-3.0,'log_tauagn':1.0,'log_qpah':0.54,'log_umin':0.0,'log_gamma':-2.0,'dust1':0.5,'dust2':0.5,
	'dust_index':-0.7,'log_age':1.0,'log_alpha':0.1,'log_beta':0.1,'log_t0':0.4,'log_tau':0.4,'logzsol':0.0}):
	
	params_assoc_fsps = {'logzsol':"logzsol", 'log_tau':"tau", 'log_age':"tage", 
					'dust_index':"dust_index", 'dust1':"dust1", 'dust2':"dust2",
					'log_gamma':"duste_gamma", 'log_umin':"duste_umin", 
					'log_qpah':"duste_qpah",'log_fagn':"fagn", 'log_tauagn':"agn_tau"}
	status_log = {'logzsol':0, 'log_tau':1, 'log_age':1, 'dust_index':0, 'dust1':0, 'dust2':0,
				'log_gamma':1, 'log_umin':1, 'log_qpah':1,'log_fagn':1, 'log_tauagn':1}

	# get stellar mass:
	formed_mass = pow(10.0,params_val['log_mass'])
	t0 = pow(10.0,params_val['log_t0'])
	tau = pow(10.0,params_val['log_tau'])
	age = pow(10.0,params_val['log_age'])
	alpha = pow(10.0,params_val['log_alpha'])
	beta = pow(10.0,params_val['log_beta'])

	# input model parameters to FSPS:
	nparams_fsps = len(params_fsps)
	for pp in range(0,nparams_fsps):
		str_temp = params_assoc_fsps[params_fsps[pp]]
		if status_log[params_fsps[pp]] == 0:
			sp.params[str_temp] = params_val[params_fsps[pp]]
		elif status_log[params_fsps[pp]] == 1:
			sp.params[str_temp] = pow(10.0,params_val[params_fsps[pp]])

	sp.params['imf_type'] = imf_type
	# gas phase metallicity:
	sp.params['gas_logz'] = params_val['logzsol']

	# make grid of SFH:
	sfh_t, sfh_sfr = construct_SFH(sfh_form=sfh_form,t0=t0,tau=tau,alpha=alpha,
										beta=beta,age=age,formed_mass=formed_mass)

	# exclude nan or inf
	idx = np.where((np.isnan(sfh_sfr)==True) | (np.isinf(sfh_sfr)==True) | (sfh_sfr<0))
	if len(idx[0]) == len(sfh_sfr):
		SFR = 1.0e-35
		dust_mass = 1.0e-35
		log_fagn_bol = 1.0e-33
	else:
		sfh_sfr[idx[0]] = 0
		if np.any(sfh_sfr > 1.0e-33) == False:
			SFR = 1.0e-35
			dust_mass = 1.0e-35
			log_fagn_bol = 1.0e-33
		else:
			sp.params["sfh"] = 3
			sp.set_tabular_sfh(sfh_t, sfh_sfr)
			mass0 = sp.stellar_mass
			dust_mass0 = sp.dust_mass
			norm = formed_mass/mass0
			dust_mass = dust_mass0*norm
			SFR = sp.sfr
			# calculate AGN lum.
			wave, spec = sp.get_spectrum(peraa=True,tage=age)
			lbol_agn = calc_bollum_from_spec_rest(spec_wave=wave,spec_lum=spec)
			sp.params["fagn"] = 0.0
			wave, spec = sp.get_spectrum(peraa=True,tage=age)
			lbol_noagn = calc_bollum_from_spec_rest(spec_wave=wave,spec_lum=spec)

			log_fagn_bol = np.log10((lbol_agn-lbol_noagn)/lbol_agn)

	return SFR, dust_mass, log_fagn_bol


def get_sfr_dust_mass_othSFH_fit(sp=None,imf_type=1,sfh_form=4,params_fsps=['logzsol', 'log_tau', 'log_age', 
	'dust_index', 'dust1', 'dust2', 'log_gamma', 'log_umin', 'log_qpah','log_fagn', 'log_tauagn'], params_val={'log_mass':0.0,
	'z':0.001,'log_fagn':-3.0,'log_tauagn':1.0,'log_qpah':0.54,'log_umin':0.0,'log_gamma':-2.0,'dust1':0.5,'dust2':0.5,
	'dust_index':-0.7,'log_age':1.0,'log_alpha':0.1,'log_beta':0.1,'log_t0':0.4,'log_tau':0.4,'logzsol':0.0}):
	
	params_assoc_fsps = {'logzsol':"logzsol", 'log_tau':"tau", 'log_age':"tage", 
					'dust_index':"dust_index", 'dust1':"dust1", 'dust2':"dust2",
					'log_gamma':"duste_gamma", 'log_umin':"duste_umin", 
					'log_qpah':"duste_qpah",'log_fagn':"fagn", 'log_tauagn':"agn_tau"}
	status_log = {'logzsol':0, 'log_tau':1, 'log_age':1, 'dust_index':0, 'dust1':0, 'dust2':0,
				'log_gamma':1, 'log_umin':1, 'log_qpah':1,'log_fagn':1, 'log_tauagn':1}

	# get stellar mass:
	formed_mass = pow(10.0,params_val['log_mass'])
	t0 = pow(10.0,params_val['log_t0'])
	tau = pow(10.0,params_val['log_tau'])
	age = pow(10.0,params_val['log_age'])
	alpha = pow(10.0,params_val['log_alpha'])
	beta = pow(10.0,params_val['log_beta'])

	# input model parameters to FSPS:
	nparams_fsps = len(params_fsps)
	for pp in range(0,nparams_fsps):
		str_temp = params_assoc_fsps[params_fsps[pp]]
		if status_log[params_fsps[pp]] == 0:
			sp.params[str_temp] = params_val[params_fsps[pp]]
		elif status_log[params_fsps[pp]] == 1:
			sp.params[str_temp] = pow(10.0,params_val[params_fsps[pp]])

	sp.params['imf_type'] = imf_type
	# gas phase metallicity:
	sp.params['gas_logz'] = params_val['logzsol']

	# make grid of SFH:
	sfh_t, sfh_sfr = construct_SFH(sfh_form=sfh_form,t0=t0,tau=tau,alpha=alpha,beta=beta,age=age,formed_mass=formed_mass)

	# exclude nan or inf
	idx = np.where((np.isnan(sfh_sfr)==True) | (np.isinf(sfh_sfr)==True) | (sfh_sfr<0))
	if len(idx[0]) == len(sfh_sfr):
		SFR = 1.0e-35
		dust_mass = 1.0e-35
	else:
		sfh_sfr[idx[0]] = 0
		if np.any(sfh_sfr > 1.0e-33) == False:
			SFR = 1.0e-35
			dust_mass = 1.0e-35
		else:
			sp.params["sfh"] = 3
			sp.set_tabular_sfh(sfh_t, sfh_sfr)
			#wave, spec0 = sp.get_spectrum(peraa=True,tage=age) ## spectrum in L_sun/AA
			mass0 = sp.stellar_mass
			dust_mass0 = sp.dust_mass
			norm = formed_mass/mass0
			spec = spec0*norm
			dust_mass = dust_mass0*norm
			SFR = sp.sfr

	return SFR, dust_mass


def get_dust_mass_mainSFH_fit(sp=None,imf_type=1,sfh_form=4,params_fsps=['logzsol', 'log_tau', 'log_age', 
	'dust_index', 'dust1', 'dust2', 'log_gamma', 'log_umin', 'log_qpah','log_fagn', 'log_tauagn'], params_val={'log_mass':0.0,
	'z':0.001,'log_fagn':-3.0,'log_tauagn':1.0,'log_qpah':0.54,'log_umin':0.0,'log_gamma':-2.0,'dust1':0.5,'dust2':0.5,
	'dust_index':-0.7,'log_age':1.0,'log_alpha':0.1,'log_beta':0.1,'log_t0':0.4,'log_tau':0.4,'logzsol':0.0}):

	params_assoc_fsps = {'logzsol':"logzsol", 'log_tau':"tau", 'log_age':"tage", 
					'dust_index':"dust_index", 'dust1':"dust1", 'dust2':"dust2",
					'log_gamma':"duste_gamma", 'log_umin':"duste_umin", 
					'log_qpah':"duste_qpah",'log_fagn':"fagn", 'log_tauagn':"agn_tau"}
	status_log = {'logzsol':0, 'log_tau':1, 'log_age':1, 'dust_index':0, 'dust1':0, 'dust2':0,
				'log_gamma':1, 'log_umin':1, 'log_qpah':1,'log_fagn':1, 'log_tauagn':1}

	# get stellar mass:
	formed_mass = pow(10.0,params_val['log_mass'])
	tau = pow(10.0,params_val['log_tau'])
	age = pow(10.0,params_val['log_age'])

	# input model parameters to FSPS:
	nparams_fsps = len(params_fsps)
	for pp in range(0,nparams_fsps):
		str_temp = params_assoc_fsps[params_fsps[pp]]
		if status_log[params_fsps[pp]] == 0:
			sp.params[str_temp] = params_val[params_fsps[pp]]
		elif status_log[params_fsps[pp]] == 1:
			sp.params[str_temp] = pow(10.0,params_val[params_fsps[pp]])

	sp.params['imf_type'] = imf_type
	sp.params['gas_logz'] = params_val['logzsol']

	# generate the SED:
	mass = sp.stellar_mass
	dust_mass0 = sp.dust_mass   ## in solar mass/norm

	# normalize:
	norm0 = formed_mass/mass
	dust_mass = dust_mass0*norm0

	#print ("mass0=%e  dust_mass0=%e formed_mass=%e  norm=%e  dust_mass=%e" % (mass,dust_mass0,formed_mass,norm0,dust_mass))

	return dust_mass


def get_dust_mass_fagnbol_mainSFH_fit(sp=None,imf_type=1,sfh_form=4,params_fsps=['logzsol', 'log_tau', 'log_age', 
	'dust_index', 'dust1', 'dust2', 'log_gamma', 'log_umin', 'log_qpah','log_fagn', 'log_tauagn'], params_val={'log_mass':0.0,
	'z':0.001,'log_fagn':-3.0,'log_tauagn':1.0,'log_qpah':0.54,'log_umin':0.0,'log_gamma':-2.0,'dust1':0.5,'dust2':0.5,
	'dust_index':-0.7,'log_age':1.0,'log_alpha':0.1,'log_beta':0.1,'log_t0':0.4,'log_tau':0.4,'logzsol':0.0}):

	params_assoc_fsps = {'logzsol':"logzsol", 'log_tau':"tau", 'log_age':"tage", 
					'dust_index':"dust_index", 'dust1':"dust1", 'dust2':"dust2",
					'log_gamma':"duste_gamma", 'log_umin':"duste_umin", 
					'log_qpah':"duste_qpah",'log_fagn':"fagn", 'log_tauagn':"agn_tau"}
	status_log = {'logzsol':0, 'log_tau':1, 'log_age':1, 'dust_index':0, 'dust1':0, 'dust2':0,
				'log_gamma':1, 'log_umin':1, 'log_qpah':1,'log_fagn':1, 'log_tauagn':1}

	# get stellar mass:
	formed_mass = pow(10.0,params_val['log_mass'])
	tau = pow(10.0,params_val['log_tau'])
	age = pow(10.0,params_val['log_age'])

	# input model parameters to FSPS:
	nparams_fsps = len(params_fsps)
	for pp in range(0,nparams_fsps):
		str_temp = params_assoc_fsps[params_fsps[pp]]
		if status_log[params_fsps[pp]] == 0:
			sp.params[str_temp] = params_val[params_fsps[pp]]
		elif status_log[params_fsps[pp]] == 1:
			sp.params[str_temp] = pow(10.0,params_val[params_fsps[pp]])

	sp.params['imf_type'] = imf_type
	# gas phase metallicity:
	sp.params['gas_logz'] = params_val['logzsol']

	# generate the SED and get AGN luminosity
	wave, spec = sp.get_spectrum(peraa=True,tage=age) ## spectrum in L_sun/AA
	lbol_agn = calc_bollum_from_spec_rest(spec_wave=wave,spec_lum=spec)

	sp.params["fagn"] = 0.0
	wave, spec = sp.get_spectrum(peraa=True,tage=age) ## spectrum in L_sun/AA
	lbol_noagn = calc_bollum_from_spec_rest(spec_wave=wave,spec_lum=spec)

	fagn_bol = (lbol_agn-lbol_noagn)/lbol_agn

	# get dust mass:
	mass = sp.stellar_mass
	dust_mass0 = sp.dust_mass   ## in solar mass/norm

	# normalize:
	norm0 = formed_mass/mass
	dust_mass = dust_mass0*norm0

	#print ("mass0=%e  dust_mass0=%e formed_mass=%e  norm=%e  dust_mass=%e" % (mass,dust_mass0,formed_mass,norm0,dust_mass))

	return dust_mass, fagn_bol


def spec_given_SFH_ZH(lbt=[],SFH_sfr=[],ZH_logzsol=[],z=0.001,cosmo='flat_LCDM',H0=70.0,Om0=0.3,imf=1,duste_switch=0,
	add_neb_emission=1,dust_law=1,add_agn=0,add_igm_absorption=0,dust1=0.0,dust2=0.0,dust_index=-99.0,
	gas_logu=-2.0,log_gamma=-99.0,log_umin=-99.0,log_qpah=-99.0,log_fagn=-99.0,log_tauagn=-99.0):
	"""A function to calculate spectra of a galaxy given SFH and metal enrichment history. SFH_sfr is in unit of M_solar/yr.
	lbt should be in Gyr. 
	"""
	
	# calling FSPS:
	sp = fsps.StellarPopulation(zcontinuous=1, imf_type=imf)

	sp.params['imf_type'] = imf

	# dust emission:
	if duste_switch == 1:
		sp.params["add_dust_emission"] = True
		sp.params["duste_gamma"] = pow(10.0,log_gamma) 
		sp.params["duste_umin"] = pow(10.0,log_umin)
		sp.params["duste_qpah"] = pow(10.0,log_qpah)
	elif duste_switch == 0:
		sp.params["add_dust_emission"] = False
	# nebular emission:
	if add_neb_emission == 1:
		sp.params["add_neb_emission"] = True
		sp.params['gas_logu'] = -2.0
	elif add_neb_emission == 0:
		sp.params["add_neb_emission"] = False
	# AGN:
	if add_agn == 1:
		sp.params["fagn"] = pow(10.0,log_fagn)
		sp.params["agn_tau"] = pow(10.0,log_tauagn)
	elif add_agn == 0:
		sp.params["fagn"] = 0
	# SSP
	sp.params["sfh"] = 0
	# dust attenuation:
	if dust_law==0:
		sp.params["dust_type"] = 0  
		sp.params["dust_tesc"] = 7.0
		sp.params["dust_index"] = dust_index
		dust1_index = -1.0
		sp.params["dust1_index"] = dust1_index
		sp.params["dust1"] = dust1
		sp.params["dust2"] = dust2
	elif dust_law==1:
		sp.params["dust_type"] = 2
		sp.params["dust1"] = 0
		sp.params["dust2"] = dust2

	# iteration:
	ntimes = len(lbt)
	spec_array = []
	survive_mass = 0
	for tt in range(0,ntimes-1):
		# calculate mass formed:
		formed_mass = abs(0.5*(lbt[tt]-lbt[tt+1])*(SFH_sfr[tt]+SFH_sfr[tt+1]))*1.0e+9
		age0 = 0.5*(lbt[tt]+lbt[tt+1])

		ave_Z = 0.5*(ZH_logzsol[tt]+ZH_logzsol[tt+1])
		sp.params["logzsol"] = ave_Z
		sp.params['gas_logz'] = ave_Z
		sp.params['tage'] = age0

		wave, spec0 = sp.get_spectrum(peraa=True,tage=age0) ## spectrum in L_sun/AA
		mass0 = sp.stellar_mass
		
		spec_array.append(spec0*formed_mass)
		survive_mass = survive_mass + (mass0*formed_mass)

		# end of for tt: ntimes

	spec = np.sum(spec_array, axis=0)

	# redshifting:
	#spec_wave,spec_flux = redshifting.cosmo_redshifting(cosmo=cosmo,H0=H0,Om0=Om0,z=z,wave=wave,spec=spec) ### in erg/s/cm^2/Ang.
	spec_wave,spec_flux = cosmo_redshifting(cosmo=cosmo,H0=H0,Om0=Om0,z=z,wave=wave,spec=spec) ### in erg/s/cm^2/Ang.

	return spec_wave,spec_flux,survive_mass


def spec_given_ages_mass_Z(grid_age=[],grid_mass=[],grid_logzsol=[],z=0.001,cosmo='flat_LCDM',H0=70.0,Om0=0.3,imf=1,duste_switch=0,
	add_neb_emission=1,dust_law=1,add_agn=0,add_igm_absorption=0,dust1=0.0,dust2=0.0,dust_index=-99.0,gas_logu=-2.0,
	log_gamma=-99.0,log_umin=-99.0,log_qpah=-99.0,log_fagn=-99.0,log_tauagn=-99.0):
	"""A function for generating model spectrum of a galaxy given input of ages, mass, and Z
	"""

	# calling FSPS:
	sp = fsps.StellarPopulation(zcontinuous=1, imf_type=imf)

	sp.params['imf_type'] = imf

	# dust emission switch:
	if duste_switch == 1:
		sp.params["add_dust_emission"] = True
		sp.params["duste_gamma"] = pow(10.0,log_gamma) 
		sp.params["duste_umin"] = pow(10.0,log_umin)
		sp.params["duste_qpah"] = pow(10.0,log_qpah)
	elif duste_switch == 0:
		sp.params["add_dust_emission"] = False
	# nebular emission switch:
	if add_neb_emission == 1:
		sp.params["add_neb_emission"] = True
		sp.params['gas_logu'] = -2.0
	elif add_neb_emission == 0:
		sp.params["add_neb_emission"] = False
	# AGN:
	if add_agn == 1:
		sp.params["fagn"] = pow(10.0,log_fagn)
		sp.params["agn_tau"] = pow(10.0,log_tauagn)
	elif add_agn == 0:
		sp.params["fagn"] = 0
	# SSP
	sp.params["sfh"] = 0
	# dust attenuation:
	if dust_law==0:
		sp.params["dust_type"] = 0  
		sp.params["dust_tesc"] = 7.0
		sp.params["dust_index"] = dust_index
		dust1_index = -1.0
		sp.params["dust1_index"] = dust1_index
		sp.params["dust1"] = dust1
		sp.params["dust2"] = dust2
	elif dust_law==1:
		sp.params["dust_type"] = 2
		sp.params["dust1"] = 0
		sp.params["dust2"] = dust2

	# iteration:
	ntimes = len(grid_age)
	spec_array = []
	for tt in range(0,ntimes):
		sp.params["logzsol"] = grid_logzsol[tt]
		sp.params['gas_logz'] = grid_logzsol[tt]
		sp.params['tage'] = grid_age[tt]

		wave, spec0 = sp.get_spectrum(peraa=True,tage=grid_age[tt]) ## spectrum in L_sun/AA
		mass0 = sp.stellar_mass

		norm = grid_mass[tt]/mass0
		spec_array.append(spec0*norm)

		# end of for tt: ntimes

	spec = np.sum(spec_array, axis=0)

	# redshifting:
	spec_wave,spec_flux = cosmo_redshifting(cosmo=cosmo,H0=H0,Om0=Om0,z=z,wave=wave,spec=spec) ### in erg/s/cm^2/Ang.

	return spec_wave,spec_flux


def convert_unit_spec_from_ergscm2A(wave,spec,funit='Jy'):
	"""A function to convert unit of flux from 'erg/s/cm2/A' --> 'erg/s/cm2' or 'Jy'
	"""
	if funit=='erg/s/cm2/A' or funit==0:
		spec_new = spec
	elif funit=='erg/s/cm2' or funit==1:
		spec_new = np.asarray(spec)*np.asarray(wave)
	elif funit=='Jy' or funit==2:
		spec_new = np.asarray(spec)*np.asarray(wave)*np.asarray(wave)/1.0e-23/2.998e+18
	else:
		print ("The input funit is not recognized!")
		sys.exit()

	return spec_new


def get_no_nebem_wave_fit(z,wave,del_wave):
	"""A function to get list of wavelengths (from the input array of wavelengths) that don't have emission line.

	:param z:
		Redshift

	:param wave:
		List of wavelengths

	:param del_wave:
		Assumption for the half of the width of the emission lines. del_wave is in Ang. 
	"""

	nebem_wave0 = np.asarray([9.231500e+02, 9.262500e+02, 9.307500e+02, 9.378100e+02, 9.497400e+02,
		 9.725200e+02, 1.025730e+03, 1.215670e+03, 1.640420e+03, 1.661240e+03,
		 1.666150e+03, 1.812200e+03, 1.854720e+03, 1.862790e+03, 1.906680e+03,
		 1.908730e+03, 2.142300e+03, 2.321660e+03, 2.324210e+03, 2.325400e+03,
		 2.326110e+03, 2.327640e+03, 2.328830e+03, 2.471090e+03, 2.661150e+03,
		 2.669950e+03, 2.796350e+03, 2.803530e+03, 3.109980e+03, 3.343500e+03,
		 3.722750e+03, 3.727100e+03, 3.729860e+03, 3.798990e+03, 3.836490e+03,
		 3.869860e+03, 3.889750e+03, 3.890170e+03, 3.968590e+03, 3.971200e+03,
		 4.069750e+03, 4.077500e+03, 4.102890e+03, 4.341690e+03, 4.364440e+03,
		 4.472730e+03, 4.622860e+03, 4.725470e+03, 4.862710e+03, 4.960300e+03,
		 5.008240e+03, 5.193270e+03, 5.201700e+03, 5.519240e+03, 5.539410e+03,
		 5.578890e+03, 5.756190e+03, 5.877250e+03, 6.302050e+03, 6.313810e+03,
		 6.365530e+03, 6.549860e+03, 6.564600e+03, 6.585270e+03, 6.679990e+03,
		 6.718290e+03, 6.732670e+03, 7.067140e+03, 7.137770e+03, 7.321940e+03,
		 7.332210e+03, 7.334170e+03, 7.753190e+03, 8.581060e+03, 8.729530e+03,
		 9.017800e+03, 9.071100e+03, 9.126100e+03, 9.232200e+03, 9.533200e+03,
		 9.548800e+03, 9.852960e+03, 1.005260e+04, 1.032332e+04, 1.083206e+04,
		 1.083331e+04, 1.094117e+04, 1.257021e+04, 1.282158e+04, 1.736688e+04,
		 1.817920e+04, 1.875640e+04, 1.945089e+04, 2.166118e+04, 2.625871e+04,
		 3.039202e+04, 3.296980e+04, 3.740576e+04, 4.052279e+04, 4.653780e+04,
		 5.128650e+04, 5.908220e+04, 6.985274e+04, 7.459900e+04, 7.502440e+04,
		 8.991380e+04, 1.051050e+05, 1.237191e+05, 1.281355e+05, 1.436780e+05,
		 1.555510e+05, 1.871300e+05, 2.183020e+05, 3.287090e+05, 3.348000e+05,
		 3.481400e+05, 3.601350e+05, 5.181450e+05, 5.733000e+05, 6.064200e+05,
		 6.318520e+05, 8.835640e+05, 1.218000e+06, 1.455350e+06, 1.576430e+06,
		 2.053000e+06, 3.703700e+06, 6.097000e+06])

	nebem_wave = (1.0+z)*nebem_wave0

	min_wave = min(np.asarray(wave))
	max_wave = max(np.asarray(wave))

	idx = np.where((nebem_wave>min_wave-del_wave) & (nebem_wave<max_wave+del_wave))
	nebem_wave = nebem_wave[idx[0]]

	nwave = len(wave)
	flag_excld = np.zeros(nwave)
	for ii in range(0,len(nebem_wave)):
		idx_excld = np.where((wave>=nebem_wave[ii]-del_wave) & (wave<=nebem_wave[ii]+del_wave))
		flag_excld[idx_excld[0]] = 1

	idx_excld = np.where(flag_excld==1)
	wave_clean = np.delete(wave, idx_excld[0])
	waveid_excld = idx_excld[0]

	return wave_clean,waveid_excld


def calc_bollum_from_spec_rest(spec_wave=[],spec_lum=[]):
	""" Function for calculating bolometric luminosity of rest-frame model spectrum in L_sun/A.
	"""

	spec_wave = np.asarray(spec_wave)
	spec_lum = np.asarray(spec_lum)

	# integrate
	wave_left = spec_wave[0:len(spec_wave)-1]
	lum_left = spec_lum[0:len(spec_wave)-1]

	wave_right = spec_wave[1:len(spec_wave)]
	lum_right = spec_lum[1:len(spec_wave)]

	areas = 0.5*(lum_left+lum_right)*(wave_right-wave_left)

	bol_lum = np.sum(areas)										# in L_sun

	l_sun = 3.826e+33      										# in erg/s
	bol_lum = bol_lum*l_sun

	return bol_lum


### calculate bolometric luminosity from a given spectrum in flux per unit wavelength:
### flux_lambda in unit erg/s/cm^2/Ang. and bolometric luminosity in erg/s
def calc_bollum_from_spec(spec_wave=[],spec_flux=[],wave_min=1000,wave_max=10000,gal_z=0.01,cosmo='flat_LCDM',H0=70.0,Om0=0.3):
	"""A function for calculating bolometric luminosity from a given spectrum in flux per unit wavelength: flux_lambda in unit erg/s/cm^2/Ang.
	The output bolometric luminosity is in erg/s.
	"""

	from astropy.cosmology import FlatLambdaCDM, WMAP5, WMAP7, WMAP9, Planck13, Planck15  
	from astropy import units as u

	# get luminosity distance:
	if cosmo=='flat_LCDM' or cosmo==0:
		cosmo1 = FlatLambdaCDM(H0=H0, Om0=Om0)
		DL0 = cosmo1.luminosity_distance(gal_z)      # in unit of Mpc
	elif cosmo=='WMAP5' or cosmo==1:
		DL0 = WMAP5.luminosity_distance(gal_z)
	elif cosmo=='WMAP7' or cosmo==2:
		DL0 = WMAP7.luminosity_distance(gal_z)
	elif cosmo=='WMAP9' or cosmo==3:
		DL0 = WMAP9.luminosity_distance(gal_z)
	elif cosmo=='Planck13' or cosmo==4:
		DL0 = Planck13.luminosity_distance(gal_z)
	elif cosmo=='Planck15' or cosmo==5:
		DL0 = Planck15.luminosity_distance(gal_z)
	#elif cosmo=='Planck18' or cosmo==6:
	#	DL0 = Planck18.luminosity_distance(gal_z)
     
	DL = DL0.value
	#print ("DL=%e Mpc" % DL)
	DL_temp = DL*1.0e+6*u.parsec
	DL_cm0 = DL_temp.to(u.cm)
	DL_cm = DL_cm0.value                    # in unit cm

	spec_lum_lambda = 4.0*pi*DL_cm*DL_cm*spec_flux  # in unit of erg/s/Ang.

	# cut spectrum:
	idx_incld = np.where((spec_wave >= wave_min) & (spec_wave <= wave_max))
	id_min = min(idx_incld[0])
	id_max = max(idx_incld[0])
	cut_spec_wave = spec_wave[id_min:id_max]
	cut_spec_lum_lambda = spec_lum_lambda[id_min:id_max]

	# integrate:
	bol_lum = 0
	for ii in range(0,len(cut_spec_wave)-1):
		area_bin = 0.5*(cut_spec_lum_lambda[ii]+cut_spec_lum_lambda[ii+1])*(cut_spec_wave[ii+1]-cut_spec_wave[ii])
		bol_lum = bol_lum + area_bin

	return bol_lum


def breakdown_CSP_into_SSP(sp=None,imf_type=1,logzsol=0.0,CSP_age=None,SFH_lbt=[],SFH_SFR=[],del_t=0.001,dust_law=1,
	dust_index=-0.7,duste_switch=0,add_neb_emission=1,add_agn=0,dust1=0,dust2=0,log_umin=0,log_qpah=0,log_gamma=0,
	log_fagn=0,log_tauagn=0,gal_z=0.001,cosmo='flat_LCDM',H0=70.0,Om0=0.3):
	"""A function to break down a CSP into its SSP components
	"""

	# break down the SFH:
	sfh_age = np.linspace(del_t,int(CSP_age/del_t)*del_t,int(CSP_age/del_t))
	f = interp1d(SFH_lbt, SFH_SFR, fill_value="extrapolate")
	sfh_sfr = f(sfh_age)
	sfh_sfr[0] = sfh_sfr[1]
	sfh_sfr[len(sfh_age)-1] = sfh_sfr[len(sfh_age)-2]

	# get CSP age:
	if CSP_age == None:
		CSP_age = max(SFH_lbt)

	#### ====================================================== #####
	# calling FSPS:
	if sp == None:
		sp = fsps.StellarPopulation(zcontinuous=1, imf_type=imf_type)

	# dust emission:
	if duste_switch == 1:
		sp.params["add_dust_emission"] = True
		sp.params["duste_gamma"] = pow(10.0,log_gamma) 
		sp.params["duste_umin"] = pow(10.0,log_umin)
		sp.params["duste_qpah"] = pow(10.0,log_qpah)
	elif duste_switch == 0:
		sp.params["add_dust_emission"] = False
	# nebular emission:
	if add_neb_emission == 1:
		sp.params["add_neb_emission"] = True
		sp.params['gas_logu'] = -2.0
	elif add_neb_emission == 0:
		sp.params["add_neb_emission"] = False
	# AGN:
	if add_agn == 1:
		sp.params["fagn"] = pow(10.0,log_fagn)
		sp.params["agn_tau"] = pow(10.0,log_tauagn)
	elif add_agn == 0:
		sp.params["fagn"] = 0

	# SSP
	sp.params["sfh"] = 0
	# dust attenuation:
	if dust_law==0:
		sp.params["dust_type"] = 0  
		sp.params["dust_tesc"] = 7.0
		sp.params["dust_index"] = dust_index
		dust1_index = -1.0
		sp.params["dust1_index"] = dust1_index
		sp.params["dust1"] = dust1
		sp.params["dust2"] = dust2
	elif dust_law==1:
		sp.params["dust_type"] = 2
		sp.params["dust1"] = 0
		sp.params["dust2"] = dust2
	#### ====================================================== #####

	# get number of wavelength:
	sp.params["logzsol"] = logzsol
	sp.params["gas_logz"] = logzsol
	sp.params["tage"] = CSP_age
	wave, spec = sp.get_spectrum(peraa=True,tage=CSP_age) ## spectrum in L_sun/AA
	nwaves = len(wave)

	# get the spectra of SSPs:
	nSSPs = len(sfh_age)
	SSP_spectra = np.zeros((nSSPs,nwaves))
	SSP_wave = np.zeros(nwaves)
	for tt in range(0,nSSPs):
		sp.params["logzsol"] = logzsol
		sp.params["gas_logz"] = logzsol
		sp.params["tage"] = sfh_age[tt]
		wave, spec = sp.get_spectrum(peraa=True,tage=sfh_age[tt]) ## spectrum in L_sun/AA
		mass = sp.stellar_mass
		norm = sfh_sfr[tt]*del_t*1.0e+9/mass

		# redshifting:
		#spec_wave,spec_flux = redshifting.cosmo_redshifting(cosmo=cosmo,H0=H0,Om0=Om0,z=gal_z,wave=wave,spec=spec*norm) ### in erg/s/cm^2/Ang.
		spec_wave,spec_flux = cosmo_redshifting(cosmo=cosmo,H0=H0,Om0=Om0,z=gal_z,wave=wave,spec=spec*norm) ### in erg/s/cm^2/Ang.

		SSP_spectra[tt] = spec_flux
		SSP_wave = spec_wave
		# end of for tt: nSSPs

	return SSP_wave,SSP_spectra,sfh_age,sfh_sfr



### define a function to break down a CSP into its SSP components:
def breakdown_CSP_into_SSP_restframe(sp=None,imf_type=1,logzsol=0.0,CSP_age=None,SFH_lbt=[],SFH_SFR=[],del_t=0.001,dust_law=1,
	dust_index=-0.7,duste_switch=0,add_neb_emission=1,add_agn=0,dust1=0,dust2=0,log_umin=0,log_qpah=0,log_gamma=0,log_fagn=0,
	log_tauagn=0):
	"""A function to break down a CSP into its SSP components (in rest-frame)
	"""

	# break down the SFH:
	sfh_age = np.linspace(del_t,int(CSP_age/del_t)*del_t,int(CSP_age/del_t))
	f = interp1d(SFH_lbt, SFH_SFR, fill_value="extrapolate")
	sfh_sfr = f(sfh_age)
	sfh_sfr[0] = sfh_sfr[1]
	sfh_sfr[len(sfh_age)-1] = sfh_sfr[len(sfh_age)-2]

	# get CSP age:
	if CSP_age == None:
		CSP_age = max(SFH_lbt)

	#### ====================================================== #####
	# calling FSPS:
	if sp == None:
		sp = fsps.StellarPopulation(zcontinuous=1, imf_type=imf_type)

	# dust emission switch:
	if duste_switch == 1:
		sp.params["add_dust_emission"] = True
		sp.params["duste_gamma"] = pow(10.0,log_gamma) 
		sp.params["duste_umin"] = pow(10.0,log_umin)
		sp.params["duste_qpah"] = pow(10.0,log_qpah)
	elif duste_switch == 0:
		sp.params["add_dust_emission"] = False
	# nebular emission switch:
	if add_neb_emission == 1:
		sp.params["add_neb_emission"] = True
		sp.params['gas_logu'] = -2.0
	elif add_neb_emission == 0:
		sp.params["add_neb_emission"] = False
	# AGN:
	if add_agn == 1:
		sp.params["fagn"] = pow(10.0,log_fagn)
		sp.params["agn_tau"] = pow(10.0,log_tauagn)
	elif add_agn == 0:
		sp.params["fagn"] = 0

	# SSP
	sp.params["sfh"] = 0
	# dust attenuation:
	if dust_law==0:
		sp.params["dust_type"] = 0  
		sp.params["dust_tesc"] = 7.0
		sp.params["dust_index"] = dust_index
		dust1_index = -1.0
		sp.params["dust1_index"] = dust1_index
		sp.params["dust1"] = dust1
		sp.params["dust2"] = dust2
	elif dust_law==1:
		sp.params["dust_type"] = 2
		sp.params["dust1"] = 0
		sp.params["dust2"] = dust2
	#### ====================================================== #####

	# get number of wavelength:
	sp.params["logzsol"] = logzsol
	sp.params["gas_logz"] = logzsol
	sp.params["tage"] = CSP_age
	wave, spec = sp.get_spectrum(peraa=True,tage=CSP_age) ## spectrum in L_sun/AA
	nwaves = len(wave)

	# get the spectra of SSPs:
	nSSPs = len(sfh_age)
	SSP_spectra = np.zeros((nSSPs,nwaves))
	SSP_wave = np.zeros(nwaves)
	for tt in range(0,nSSPs):
		sp.params["logzsol"] = logzsol
		sp.params["gas_logz"] = logzsol
		sp.params["tage"] = sfh_age[tt]
		wave, spec = sp.get_spectrum(peraa=True,tage=sfh_age[tt]) ## spectrum in L_sun/AA
		mass = sp.stellar_mass
		norm = sfh_sfr[tt]*del_t*1.0e+9/mass

		SSP_spectra[tt] = spec*norm
		SSP_wave = wave
		# end of for tt: nSSPs

	return SSP_wave,SSP_spectra,sfh_age,sfh_sfr


def get_emlines_luminosities(sp=None,emlines_rest_waves=[],imf_type=1,duste_switch=1,add_neb_emission=1,dust_law=1,
	sfh_form=4,add_agn=1,sfh_t=[],sfh_sfr=[],params_val={'log_mass':0.0,'z':0.001,'log_fagn':-3.0,'log_tauagn':1.0,
	'log_qpah':0.54,'log_umin':0.0,'log_gamma':-2.0,'dust1':0.5,'dust2':0.5,'dust_index':-0.7,'log_age':1.0,'log_alpha':0.1,
	'log_beta':0.1,'log_t0':0.4,'log_tau':0.4,'logzsol':0.0}):
	"""A function to get luminosities of some emission lines.
	"""

	formed_mass = pow(10.0,params_val['log_mass'])

	# get some input properties:
	age = pow(10.0,params_val['log_age'])
	t0 = pow(10.0,params_val['log_t0'])
	tau = pow(10.0,params_val['log_tau'])
	alpha = pow(10.0,params_val['log_alpha'])
	beta = pow(10.0,params_val['log_beta'])

	sp.params['imf_type'] = imf_type

	# dust emission:
	if duste_switch == 1:
		sp.params["add_dust_emission"] = True
		sp.params["duste_gamma"] = pow(10.0,params_val['log_gamma']) 
		sp.params["duste_umin"] = pow(10.0,params_val['log_umin'])
		sp.params["duste_qpah"] = pow(10.0,params_val['log_qpah'])
	elif duste_switch == 0:
		sp.params["add_dust_emission"] = False
	# nebular emission:
	if add_neb_emission == 1:
		sp.params["add_neb_emission"] = True
		sp.params['gas_logu'] = -2.0
	elif add_neb_emission == 0:
		sp.params["add_neb_emission"] = False
	# AGN:
	if add_agn == 1:
		sp.params["fagn"] = pow(10.0,params_val['log_fagn'])
		sp.params["agn_tau"] = pow(10.0,params_val['log_tauagn'])
	elif add_agn == 0:
		sp.params["fagn"] = 0

	# CSP
	#sp.params["sfh"] = 3
	# dust attenuation:
	if dust_law==0:
		sp.params["dust_type"] = 0  
		sp.params["dust_tesc"] = 7.0
		sp.params["dust_index"] = params_val['dust_index']
		dust1_index = -1.0
		sp.params["dust1_index"] = dust1_index
		sp.params["dust1"] = params_val['dust1']
		sp.params["dust2"] = params_val['dust2']
	elif dust_law==1:
		sp.params["dust_type"] = 2
		sp.params["dust1"] = 0
		sp.params["dust2"] = params_val['dust2']
	# other parameters:
	sp.params["logzsol"] = params_val['logzsol'] 
	sp.params['gas_logz'] = params_val['logzsol'] 
	sp.params['tage'] = age

	# make grid of SFH:
	if sfh_form==0 or sfh_form==1 or sfh_form==2 or sfh_form==3 or sfh_form==4:
		sfh_t, sfh_sfr = construct_SFH(sfh_form=sfh_form,t0=t0,tau=tau,alpha=alpha,
										beta=beta,age=age,formed_mass=formed_mass)
	elif sfh_form=='arbitrary_sfh':
		sfh_t0 = sfh_t
		sfh_sfr0 = sfh_sfr
		del_t = 0.001
		sfh_t = np.linspace(0,age,int(age/del_t)+1)
		f = interp1d(sfh_t0, sfh_sfr0, fill_value="extrapolate")
		sfh_sfr = f(sfh_t)
		idx_neg = np.where(sfh_sfr < 0)
		for ii in range(0,len(idx_neg[0])):
			sfh_sfr[idx_neg[0][ii]] = 0.0
	else:
		print ("SFH choice is not recognized!")
		sys.exit()

	# exclude nan or inf
	idx = np.where((np.isnan(sfh_sfr)==True) | (np.isinf(sfh_sfr)==True) | (sfh_sfr<0))
	if len(idx[0]) == len(sfh_sfr):
		emline_lum = 0.0
		emline_wave = 0.0
	else:
		sfh_sfr[idx[0]] = 0
		if np.any(sfh_sfr > 1.0e-33) == False:
			emline_lum = 0.0
			emline_wave = 0.0
		else:
			sp.params["sfh"] = 3
			sp.set_tabular_sfh(sfh_t, sfh_sfr)
			wave, spec0 = sp.get_spectrum(peraa=True,tage=age) ## spectrum in L_sun/AA
			# get emission lines:
			emline_lum = sp.emline_luminosity
			emline_wave = sp.emline_wavelengths

	# select among the list of the emission lines:
	nlines = len(emlines_rest_waves)

	emline_waves = np.zeros(nlines)
	emline_luminosities = np.zeros(nlines)
	for ii in range(0,nlines):
		del_waves = np.abs(emline_wave - emlines_rest_waves[ii])
		idx, min_val = min(enumerate(del_waves), key=itemgetter(1))
		emline_waves[ii] = emline_wave[idx]
		emline_luminosities[ii] = emline_lum[idx]*3.826e+33    ### in unit of erg/s

	return emline_waves,emline_luminosities



