# -*- coding: utf-8 -*-
"""Chemical Engineering Design Library (ChEDL). Utilities for process modeling.
Copyright (C) 2016, 2017, 2018, 2020 Caleb Bell <Caleb.Andrew.Bell@gmail.com>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

This module contains functions for looking up standard pipe sizes from
different schedules. Similarly, there is a converter between gauge number
thickness (and back).

For reporting bugs, adding feature requests, or submitting pull requests,
please use the `GitHub issue tracker <https://github.com/CalebBell/fluids/>`_
or contact the author at Caleb.Andrew.Bell@gmail.com.


.. contents:: :local:

Pipe Schedules
--------------
.. autofunction:: nearest_pipe

Wire Gauge
----------
.. autofunction:: gauge_from_t
.. autofunction:: t_from_gauge
.. autodata:: wire_schedules

"""

from __future__ import division
from math import pi
from fluids.constants import inch

__all__ = ['nearest_pipe', 'gauge_from_t', 't_from_gauge', 'wire_schedules']

# Schedules 5, 10, 20, 30, 40, 60, 80, 100, 120, 140, 160 from
# ASME B36.10M - Welded and Seamless Wrought Steel Pipe
# All schedule lists stored in mm, other than NPS.
# i = inner diameter, o = outer diameter, and t = wall thickness in variable names

NPS5 = [0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 30]
S5i = [18, 23.4, 30.1, 38.9, 45, 57, 68.78, 84.68, 97.38, 110.08, 135.76, 162.76, 213.56, 266.2, 315.88, 347.68, 398.02, 448.62, 498.44, 549.44, 598.92, 749.3]
S5o = [21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 101.6, 114.3, 141.3, 168.3, 219.1, 273, 323.8, 355.6, 406.4, 457, 508, 559, 610, 762]
S5t = [1.65, 1.65, 1.65, 1.65, 1.65, 1.65, 2.11, 2.11, 2.11, 2.11, 2.77, 2.77, 2.77, 3.4, 3.96, 3.96, 4.19, 4.19, 4.78, 4.78, 5.54, 6.35]

NPS10 = [0.125, 0.25, 0.375, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30, 32, 34, 36]
S10i = [7.82, 10.4, 13.8, 17.08, 22.48, 27.86, 36.66, 42.76, 54.76, 66.9, 82.8, 95.5, 108.2, 134.5, 161.5, 211.58, 264.62, 314.66, 342.9, 393.7, 444.3, 495.3, 546.3, 597.3, 644.16, 695.16, 746.16, 797.16, 848.16, 898.16]
S10o = [10.3, 13.7, 17.1, 21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 101.6, 114.3, 141.3, 168.3, 219.1, 273, 323.8, 355.6, 406.4, 457, 508, 559, 610, 660, 711, 762, 813, 864, 914]
S10t = [1.24, 1.65, 1.65, 2.11, 2.11, 2.77, 2.77, 2.77, 2.77, 3.05, 3.05, 3.05, 3.05, 3.4, 3.4, 3.76, 4.19, 4.57, 6.35, 6.35, 6.35, 6.35, 6.35, 6.35, 7.92, 7.92, 7.92, 7.92, 7.92, 7.92]

NPS20 = [8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30, 32, 34, 36]
S20i = [206.4, 260.3, 311.1, 339.76, 390.56, 441.16, 488.94, 539.94, 590.94, 634.6, 685.6, 736.6, 787.6, 838.6, 888.6]
S20o = [219.1, 273, 323.8, 355.6, 406.4, 457, 508, 559, 610, 660, 711, 762, 813, 864, 914]
S20t = [6.35, 6.35, 6.35, 7.92, 7.92, 7.92, 9.53, 9.53, 9.53, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7]

NPS30 = [0.125, 0.25, 0.375, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 8, 10, 12, 14, 16, 18, 20, 22, 24, 28, 30, 32, 34, 36]
S30i = [7.4, 10, 13.4, 16.48, 21.88, 27.6, 36.26, 41.94, 53.94, 63.44, 79.34, 92.04, 104.74, 205.02, 257.4, 307.04, 336.54, 387.34, 434.74, 482.6, 533.6, 581.46, 679.24, 730.24, 781.24, 832.24, 882.24]
S30o = [10.3, 13.7, 17.1, 21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 101.6, 114.3, 219.1, 273, 323.8, 355.6, 406.4, 457, 508, 559, 610, 711, 762, 813, 864, 914]
S30t = [1.45, 1.85, 1.85, 2.41, 2.41, 2.9, 2.97, 3.18, 3.18, 4.78, 4.78, 4.78, 4.78, 7.04, 7.8, 8.38, 9.53, 9.53, 11.13, 12.7, 12.7, 14.27, 15.88, 15.88, 15.88, 15.88, 15.88]

NPS40 = [0.125, 0.25, 0.375, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 24, 32, 34, 36]
S40i = [6.84, 9.22, 12.48, 15.76, 20.96, 26.64, 35.08, 40.94, 52.48, 62.68, 77.92, 90.12, 102.26, 128.2, 154.08, 202.74, 254.46, 303.18, 333.34, 381, 428.46, 477.82, 575.04, 778.04, 829.04, 875.9]
S40o = [10.3, 13.7, 17.1, 21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 101.6, 114.3, 141.3, 168.3, 219.1, 273, 323.8, 355.6, 406.4, 457, 508, 610, 813, 864, 914]
S40t = [1.73, 2.24, 2.31, 2.77, 2.87, 3.38, 3.56, 3.68, 3.91, 5.16, 5.49, 5.74, 6.02, 6.55, 7.11, 8.18, 9.27, 10.31, 11.13, 12.7, 14.27, 15.09, 17.48, 17.48, 17.48, 19.05]

NPS60 = [8, 10, 12, 14, 16, 18, 20, 22, 24]
S60i = [198.48, 247.6, 295.26, 325.42, 373.08, 418.9, 466.76, 514.54, 560.78]
S60o = [219.1, 273, 323.8, 355.6, 406.4, 457, 508, 559, 610]
S60t = [10.31, 12.7, 14.27, 15.09, 16.66, 19.05, 20.62, 22.23, 24.61]

NPS80 = [0.125, 0.25, 0.375, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24]
S80i = [5.48, 7.66, 10.7, 13.84, 18.88, 24.3, 32.5, 38.14, 49.22, 58.98, 73.66, 85.44, 97.18, 122.24, 146.36, 193.7, 242.82, 288.84, 317.5, 363.52, 409.34, 455.62, 501.84, 548.08]
S80o = [10.3, 13.7, 17.1, 21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 101.6, 114.3, 141.3, 168.3, 219.1, 273, 323.8, 355.6, 406.4, 457, 508, 559, 610]
S80t = [2.41, 3.02, 3.2, 3.73, 3.91, 4.55, 4.85, 5.08, 5.54, 7.01, 7.62, 8.08, 8.56, 9.53, 10.97, 12.7, 15.09, 17.48, 19.05, 21.44, 23.83, 26.19, 28.58, 30.96]

NPS100 = [8, 10, 12, 14, 16, 18, 20, 22, 24]
S100i = [188.92, 236.48, 280.92, 307.94, 354.02, 398.28, 442.92, 489.14, 532.22]
S100o = [219.1, 273, 323.8, 355.6, 406.4, 457, 508, 559, 610]
S100t = [15.09, 18.26, 21.44, 23.83, 26.19, 29.36, 32.54, 34.93, 38.89]

NPS120 = [4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24]
S120i = [92.04, 115.9, 139.76, 182.58, 230.12, 273, 300.02, 344.48, 387.14, 431.8, 476.44, 517.96]
S120o = [114.3, 141.3, 168.3, 219.1, 273, 323.8, 355.6, 406.4, 457, 508, 559, 610]
S120t = [11.13, 12.7, 14.27, 18.26, 21.44, 25.4, 27.79, 30.96, 34.93, 38.1, 41.28, 46.02]

NPS140 = [8, 10, 12, 14, 16, 18, 20, 22, 24]
S140i = [177.86, 222.2, 266.64, 292.1, 333.34, 377.66, 419.1, 463.74, 505.26]
S140o = [219.1, 273, 323.8, 355.6, 406.4, 457, 508, 559, 610]
S140t = [20.62, 25.4, 28.58, 31.75, 36.53, 39.67, 44.45, 47.63, 52.37]

NPS160 = [0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24]
S160i = [11.74, 15.58, 20.7, 29.5, 34.02, 42.82, 53.94, 66.64, 87.32, 109.54, 131.78, 173.08, 215.84, 257.16, 284.18, 325.42, 366.52, 407.98, 451.04, 490.92]
S160o = [21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 114.3, 141.3, 168.3, 219.1, 273, 323.8, 355.6, 406.4, 457, 508, 559, 610]
S160t = [4.78, 5.56, 6.35, 6.35, 7.14, 8.74, 9.53, 11.13, 13.49, 15.88, 18.26, 23.01, 28.58, 33.32, 35.71, 40.49, 45.24, 50.01, 53.98, 59.54]

# Schedules designated STD, XS, and XXS
NPSSTD = [0.125, 0.25, 0.375, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30, 32, 34, 36, 38, 40, 42, 44, 46, 48]
STDi = [6.84, 9.22, 12.48, 15.76, 20.96, 26.64, 35.08, 40.94, 52.48, 62.68, 77.92, 90.12, 102.26, 128.2, 154.08, 202.74, 254.46, 304.74, 336.54, 387.34, 437.94, 488.94, 539.94, 590.94, 640.94, 691.94, 742.94, 793.94, 844.94, 894.94, 945.94, 996.94, 1047.94, 1098.94, 1148.94, 1199.94]
STDo = [10.3, 13.7, 17.1, 21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 101.6, 114.3, 141.3, 168.3, 219.1, 273, 323.8, 355.6, 406.4, 457, 508, 559, 610, 660, 711, 762, 813, 864, 914, 965, 1016, 1067, 1118, 1168, 1219]
STDt = [1.73, 2.24, 2.31, 2.77, 2.87, 3.38, 3.56, 3.68, 3.91, 5.16, 5.49, 5.74, 6.02, 6.55, 7.11, 8.18, 9.27, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53]

NPSXS = [0.125, 0.25, 0.375, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30, 32, 34, 36, 38, 40, 42, 44, 46, 48]
XSi = [5.48, 7.66, 10.7, 13.84, 18.88, 24.3, 32.5, 38.14, 49.22, 58.98, 73.66, 85.44, 97.18, 122.24, 146.36, 193.7, 247.6, 298.4, 330.2, 381, 431.6, 482.6, 533.6, 584.6, 634.6, 685.6, 736.6, 787.6, 838.6, 888.6, 939.6, 990.6, 1041.6, 1092.6, 1142.6, 1193.6]
XSo = [10.3, 13.7, 17.1, 21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 101.6, 114.3, 141.3, 168.3, 219.1, 273, 323.8, 355.6, 406.4, 457, 508, 559, 610, 660, 711, 762, 813, 864, 914, 965, 1016, 1067, 1118, 1168, 1219]
XSt = [2.41, 3.02, 3.2, 3.73, 3.91, 4.55, 4.85, 5.08, 5.54, 7.01, 7.62, 8.08, 8.56, 9.53, 10.97, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7]

NPSXXS = [0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 4, 5, 6, 8, 10, 12]
XXSi = [6.36, 11.06, 15.22, 22.8, 28, 38.16, 44.96, 58.42, 80.06, 103.2, 124.4, 174.64, 222.2, 273]
XXSo = [21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 114.3, 141.3, 168.3, 219.1, 273, 323.8]
XXSt = [7.47, 7.82, 9.09, 9.7, 10.15, 11.07, 14.02, 15.24, 17.12, 19.05, 21.95, 22.23, 25.4, 25.4]

NPSS5 = [0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 30]
SS5DN = [15, 20, 25, 32, 40, 50, 65, 80, 90, 100, 125, 150, 200, 250, 300, 350, 400, 450, 500, 550, 600, 750]
SS5i = [18, 23.4, 30.1, 38.9, 45, 57, 68.78, 84.68, 97.38, 110.08, 135.76, 162.76, 213.56, 266.3, 315.98, 347.68, 398.02, 448.62, 498.44, 549.44, 598.92, 749.3]
SS5o = [21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 101.6, 114.3, 141.3, 168.3, 219.1, 273.1, 323.9, 355.6, 406.4, 457, 508, 559, 610, 762]
SS5t = [1.65, 1.65, 1.65, 1.65, 1.65, 1.65, 2.11, 2.11, 2.11, 2.11, 2.77, 2.77, 2.77, 3.4, 3.96, 3.96, 4.19, 4.19, 4.78, 4.78, 5.54, 6.35]

# Schedules 10, 40 and 80 from ASME B36.19M - Stainless Steel Pipe
NPSS10 = [0.125, 0.25, 0.375, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 30]
SS10DN = [6, 8, 10, 15, 20, 25, 32, 40, 50, 65, 80, 90, 100, 125, 150, 200, 250, 300, 350, 400, 450, 500, 550, 600, 750]
SS10i = [7.82, 10.4, 13.8, 17.08, 22.48, 27.86, 36.66, 42.76, 54.76, 66.9, 82.8, 95.5, 108.2, 134.5, 161.5, 211.58, 264.72, 314.76, 346.04, 396.84, 447.44, 496.92, 547.92, 597.3, 746.16]
SS10o = [10.3, 13.7, 17.1, 21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 101.6, 114.3, 141.3, 168.3, 219.1, 273.1, 323.9, 355.6, 406.4, 457, 508, 559, 610, 762]
SS10t = [1.24, 1.65, 1.65, 2.11, 2.11, 2.77, 2.77, 2.77, 2.77, 3.05, 3.05, 3.05, 3.05, 3.4, 3.4, 3.76, 4.19, 4.57, 4.78, 4.78, 4.78, 5.54, 5.54, 6.35, 7.92]

NPSS40 = [0.125, 0.25, 0.375, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 24]
SS40DN = [6, 8, 10, 15, 20, 25, 32, 40, 50, 65, 80, 90, 100, 125, 150, 200, 250, 300, 350, 400, 450, 500, 600]
SS40i = [6.84, 9.22, 12.48, 15.76, 20.96, 26.64, 35.08, 40.94, 52.48, 62.68, 77.92, 90.12, 102.26, 128.2, 154.08, 202.74, 254.56, 304.84, 336.54, 387.34, 437.94, 488.94, 590.94]
SS40o = [10.3, 13.7, 17.1, 21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 101.6, 114.3, 141.3, 168.3, 219.1, 273.1, 323.9, 355.6, 406.4, 457, 508, 610]
SS40t = [1.73, 2.24, 2.31, 2.77, 2.87, 3.38, 3.56, 3.68, 3.91, 5.16, 5.49, 5.74, 6.02, 6.55, 7.11, 8.18, 9.27, 9.53, 9.53, 9.53, 9.53, 9.53, 9.53]

NPSS80 = [0.125, 0.25, 0.375, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 24]
SS80DN = [6, 8, 10, 15, 20, 25, 32, 40, 50, 65, 80, 90, 100, 125, 150, 200, 250, 300, 350, 400, 450, 500, 600]
SS80i = [5.48, 7.66, 10.7, 13.84, 18.88, 24.3, 32.5, 38.14, 49.22, 58.98, 73.66, 85.44, 97.18, 122.24, 146.36, 193.7, 247.7, 298.5, 330.2, 381, 431.6, 482.6, 584.6]
SS80o = [10.3, 13.7, 17.1, 21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 101.6, 114.3, 141.3, 168.3, 219.1, 273.1, 323.9, 355.6, 406.4, 457, 508, 610]
SS80t = [2.41, 3.02, 3.2, 3.73, 3.91, 4.55, 4.85, 5.08, 5.54, 7.01, 7.62, 8.08, 8.56, 9.53, 10.97, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7, 12.7]

# ABS; note the IPS units are the standard. These are converted from IPS.
NPS_D1527 = [0.125, 0.25, 0.375, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12]
S40i_D1527 = [6.8326, 9.2456, 12.5222, 15.7988, 20.9296, 26.6446, 33.528, 40.894, 52.5018, 62.7126, 77.9272, 90.1192, 102.2604, 128.1938, 154.051, 202.7174, 254.508, 303.2252]
S40o_D1527 = S80o_D1527 = [10.287, 13.716, 17.145, 21.336, 26.67, 33.401, 40.64, 48.26, 60.325, 73.025, 88.9, 101.6, 114.3, 141.3002, 168.275, 219.075, 273.05, 323.85]
S40t_D1527 = [1.7272, 2.2352, 2.3114, 2.7686, 2.8702, 3.3782, 3.556, 3.683, 3.9116, 5.1562, 5.4864, 5.7404, 6.0198, 6.5532, 7.112, 8.1788, 9.271, 10.3124]

S80i_D1527 = [5.461, 7.6708, 10.7442, 13.8684, 18.8468, 24.3078, 30.9372, 38.1, 49.2506, 59.0042, 73.66, 85.4456, 97.1804, 122.2502, 146.3294, 193.675, 242.9256, 288.9504]
S80t_D1527 = [2.413, 3.0226, 3.2004, 3.7338, 3.9116, 4.5466, 4.8514, 5.08, 5.5372, 7.0104, 7.62, 8.0772, 8.5598, 9.525, 10.9728, 12.7, 15.0622, 17.4498]

# D2680 ABS and PVC; note the IPS units are the standard. These are converted from IPS.
NPS_D2680 = [8, 10, 12, 15]
SPVCo_D2680 = SABSo_D2680 = [239.014, 298.45, 357.378, 447.548]
SPVCi_D2680 = SABSi_D2680 = [196.85, 247.65, 298.45, 374.65]
SPVCt_D2680 = SABSt_D2680 = [21.082, 25.4, 29.464, 36.449]


# AWWA C900 PVC
# Standard does not specify if IPS or SI units are authoritative
# SI units are used here.
NPS_C900 = [4, 6, 8, 10, 12]
SDR14o_C900 = SDR18o_C900 = SDR25o_C900 = [121.9, 175.3, 229.9, 281.9, 335.3]
SDR25t_C900 = [4.88, 7.01, 9.19, 11.28, 13.41]
SDR25i_C900 = [112.14, 161.28, 211.52, 259.34, 308.48]

SDR18t_C900 = [6.78, 9.73, 12.78, 15.67, 18.62]
SDR18i_C900 = [108.34, 155.84, 204.34, 250.56, 298.06]

SDR14t_C900 = [8.71, 12.52, 16.41, 20.14, 23.95]
SDR14i_C900 = [104.48, 150.26, 197.08, 241.62, 287.4]

# AWWA C905
# Standard does not specify if IPS or SI units are authoritative
# SI units are used here.
NPS_C905 = [14, 16, 18, 20, 24, 30, 36, 42, 48]

NPSCIDR14_C905 = [14, 16, 18]
SCIDR14o_C905 = [388.6, 442.0, 495.3]
SCIDR14i_C905 = [333.08, 378.86, 424.54]
SCIDR14t_C905 = [27.76, 31.57, 35.38]

NPSCIDR18_C905 = [14, 16, 18, 20, 24, 30]
SCIDR18o_C905 = [388.6, 442.0, 495.3, 548.6, 655.3, 812.8]
SCIDR18i_C905 = [345.42, 392.88, 440.28, 487.64, 582.5, 722.48]
SCIDR18t_C905 = [21.59, 24.56, 27.51, 30.48, 36.4, 45.16]

NPSCIDR21_C905 = [14, 16, 18, 20, 24, 30, 36]
SCIDR21o_C905 = [388.6, 442.0, 495.3, 548.6, 655.3, 812.8, 972.8]
SCIDR21i_C905 = [351.58, 399.9, 448.1, 496.32, 592.86, 735.38, 880.14]
SCIDR21t_C905 = [18.51, 21.05, 23.6, 26.14, 31.22, 38.71, 46.33]

NPSCIDR25_C905 = NPS_C905
SCIDR25o_C905 = [388.6, 442.0, 495.3, 548.6, 655.3, 812.8, 972.8, 1130.3, 1290.3]
SCIDR25i_C905 = [357.52, 406.64, 455.68, 504.7, 602.88, 747.78, 894.98, 1039.88, 1187.08]
SCIDR25t_C905 = [15.54, 17.68, 19.81, 21.95, 26.21, 32.51, 38.91, 45.21, 51.61]

NPSCIDR325_C905 = NPS_C905
SCIDR325o_C905 = [388.6, 442.0, 495.3, 548.6, 655.3, 812.8, 972.8, 1130.3, 1290.3]
SCIDR325i_C905 = [364.68, 414.82, 464.82, 514.82, 614.96, 762.76, 912.96, 1060.76, 1210.9]
SCIDR325t_C905 = [11.96, 13.59, 15.24, 16.89, 20.17, 25.02, 29.92, 34.77, 39.7]

NPSCIDR41_C905 = NPS_C905
SCIDR41o_C905 = [388.6, 442.0, 495.3, 548.6, 655.3, 812.8, 972.8, 1130.3, 1290.3]
SCIDR41i_C905 = [369.66, 420.46, 471.12, 521.82, 623.34, 773.18, 925.36, 1075.18, 1227.36]
SCIDR41t_C905 = [9.47, 10.77, 12.09, 13.39, 15.98, 19.81, 23.72, 27.56, 31.47]

NPSCIDR51_C905 = [18, 20, 24, 30, 36, 42, 48]
SCIDR51o_C905 = [495.3, 548.6, 655.3, 812.8, 972.8, 1130.3, 1290.3]
SCIDR51i_C905 = [475.9, 527.06, 629.6, 780.94, 934.64, 1086.0, 1239.7]
SCIDR51t_C905 = [9.7, 10.77, 12.85, 15.93, 19.08, 22.15, 25.3]

NPSIPSDR21_C905 = NPSIPSDR26_C905 = NPSIPSDR325_C905 = NPSIPSDR41_C905 = [14, 16, 18, 20, 24, 30, 36]
SIPSDR21o_C905 = SIPSDR26o_C905 = SIPSDR325o_C905 = SIPSDR41o_C905 = [355.6, 406.4, 457.2, 508.0, 609.6, 762.0, 914.4]

SIPSDR21i_C905 = [321.76, 367.7, 413.66, 459.64, 551.54, 689.46, 827.32]
SIPSDR21t_C905 = [16.92, 19.35, 21.77, 24.18, 29.03, 36.27, 43.54]

SIPSDR26i_C905 = [328.26, 375.16, 422.04, 468.94, 562.72, 703.38, 844.04]
SIPSDR26t_C905 = [13.67, 15.62, 17.58, 19.53, 23.44, 29.31, 35.18]

SIPSDR325i_C905 = [333.76, 381.4, 429.06, 476.76, 572.1, 715.12, 858.12]
SIPSDR325t_C905 = [10.92, 12.5, 14.07, 15.62, 18.75, 23.44, 28.14]

SIPSDR41i_C905 = [338.28, 386.58, 434.9, 483.2, 579.88, 724.82, 869.8]
SIPSDR41t_C905 = [8.66, 9.91, 11.15, 12.4, 14.86, 18.59, 22.3]

# ASTM F679 ; note the IPS units are the standard. These are converted from IPS.
NPS_F679 = [18, 21, 24, 27, 30, 36, 42, 48, 54, 60]
SPS46o_F679 = SPS75o_F679 = SPS115o_F679 = [475.0054, 559.9938, 629.9962, 710.0062, 812.8, 972.82, 1130.3, 1290.32, 1462.024, 1564.894]

SPS46t_F679 = [12.6746, 14.9352, 16.7894, 18.923, 21.6662, 25.9334, 30.1498, 34.417, 38.989, 41.7322]
SPS46i_F679 = [449.6562, 530.1234, 596.4174, 672.1602, 769.4676, 920.9532, 1070.0004, 1221.486, 1384.046, 1481.4296]

SPS75t_F679 = [14.8336, 17.5006, 19.685, 22.1996, 25.4, 30.4038, 35.3314, 40.3352, 45.6946, 48.9204]
SPS75i_F679 = [445.3382, 524.9926, 590.6262, 665.607, 762.0, 912.0124, 1059.6372, 1209.6496, 1370.6348, 1467.0532]

SPS115t_F679 = [17.0434, 20.0914, 22.5806, 25.4508, 29.1592, 34.8742, 40.5384, 46.2788, 52.4256, 56.134]
SPS115i_F679 = [440.9186, 519.811, 584.835, 659.1046, 754.4816, 903.0716, 1049.2232, 1197.7624, 1357.1728, 1452.626]

# ASTM D2665 PVC drain and vent pipe; note the IPS units are the standard. These are converted from IPS.
NPS_D2665 = [1.25, 1.5, 2, 3, 4, 6, 8, 10, 12, 14, 16, 18, 20, 24]
SPVCo_D2665 = [42.164, 48.26, 60.325, 88.9, 114.3, 168.275, 219.075, 273.05, 323.85, 355.6, 406.4, 457.2, 508.0, 609.6]
SPVCt_D2665 = [3.556, 3.683, 3.9116, 5.4864, 6.0198, 7.112, 8.1788, 9.271, 10.3124, 10.922, 12.7, 14.2748, 15.0622, 17.4498]
SPVCi_D2665 = [35.052, 40.894, 52.5018, 77.9272, 102.2604, 154.051, 202.7174, 254.508, 303.2252, 333.756, 381.0, 428.6504, 477.8756, 574.7004]

# ASTM D1785 PVC schedules; note the IPS units are the standard. These are converted from IPS.
NPS_D1785 = [0.125, 0.25, 0.375, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 24]
S40o_D1785 = S80o_D1785 = [10.287, 13.716, 17.145, 21.336, 26.67, 33.401, 42.164, 48.26, 60.325, 73.025, 88.9, 101.6, 114.3, 141.3002, 168.275, 219.075, 273.05, 323.85, 355.6, 406.4, 457.2, 508.0, 609.6]

S40i_D1785 = [6.8326, 9.2456, 12.5222, 15.7988, 20.9296, 26.6446, 35.052, 40.894, 52.5018, 62.7126, 77.9272, 90.1192, 102.2604, 128.1938, 154.051, 202.7174, 254.508, 303.2252, 333.4004, 381.0, 428.6504, 477.8756, 574.7004]
S40t_D1785 = [1.7272, 2.2352, 2.3114, 2.7686, 2.8702, 3.3782, 3.556, 3.683, 3.9116, 5.1562, 5.4864, 5.7404, 6.0198, 6.5532, 7.112, 8.1788, 9.271, 10.3124, 11.0998, 12.7, 14.2748, 15.0622, 17.4498]

S80i_D1785 = [5.461, 7.6708, 10.7442, 13.8684, 18.8468, 24.3078, 32.4612, 38.1, 49.2506, 59.0042, 73.66, 85.4456, 97.1804, 122.2502, 146.3294, 193.675, 242.9256, 288.9504, 317.5, 363.5756, 409.6004, 455.6252, 547.7256]
S80t_D1785 = [2.413, 3.0226, 3.2004, 3.7338, 3.9116, 4.5466, 4.8514, 5.08, 5.5372, 7.0104, 7.62, 8.0772, 8.5598, 9.525, 10.9728, 12.7, 15.0622, 17.4498, 19.05, 21.4122, 23.7998, 26.1874, 30.9372]

NPS120_D1785 = NPS_D1785[3:18]
S120i_D1785 = [12.7, 18.034, 23.241, 31.242, 36.83, 47.625, 57.785, 71.12, 83.82, 92.1004, 115.9002, 139.7254, 182.6006, 230.2256, 273.05]
S120t_D1785 = [4.318, 4.318, 5.08, 5.461, 5.715, 6.35, 7.62, 8.89, 8.89, 11.0998, 12.7, 14.2748, 18.2372, 21.4122, 25.4]
S120o_D1785 = S80o_D1785[3:18]

# ASTM D2241

NPS_D2241 = [0.125, 0.25, 0.375, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 24, 30, 36]
o_D2241_complete = [10.287, 13.716, 17.145, 21.336, 26.67, 33.401, 42.164, 48.26, 60.325, 73.025, 88.9, 101.6, 114.3, 141.3002, 168.275, 219.075, 273.05, 323.85, 355.6, 406.4, 457.2, 508.0, 609.6, 762.0, 914.4]

NPSDR64_D2241 = NPS_D2241[12:18]
SDR64o_D2241 = o_D2241_complete[12:18]
SDR64t_D2241 = [1.778, 2.2098, 2.6416, 3.429, 4.2672, 5.0546]
SDR64i_D2241 = [110.744, 136.8806, 162.9918, 212.217, 264.5156, 313.7408]

NPSDR41_D2241 = NPS_D2241[10:]
SDR41o_D2241 = o_D2241_complete[10:]
SDR41t_D2241 = [2.159, 2.4892, 2.794, 3.4544, 4.1148, 5.334, 6.6548, 7.8994, 8.6614, 9.906, 11.1506, 12.3952, 14.859, 18.5928, 22.3012]
SDR41i_D2241 = [84.582, 96.6216, 108.712, 134.3914, 160.0454, 208.407, 259.7404, 308.0512, 338.2772, 386.588, 434.8988, 483.2096, 579.882, 724.8144, 869.7976]

NPSDR325_D2241 = NPS_D2241[6:]
SDR325o_D2241 = o_D2241_complete[6:]
SDR325t_D2241 = [1.524, 1.524, 1.8542, 2.2352, 2.7432, 3.1242, 3.5052, 4.3434, 5.1816, 6.731, 8.4074, 9.9568, 10.922, 12.4968, 14.0716, 15.621, 18.7452, 23.4442, 28.1432]
SDR325i_D2241 = [39.116, 45.212, 56.6166, 68.5546, 83.4136, 95.3516, 107.2896, 132.6134, 157.9118, 205.613, 256.2352, 303.9364, 333.756, 381.4064, 429.0568, 476.758, 572.1096, 715.1116, 858.1136]

NPSDR26_D2241 = NPS_D2241[5:]
SDR26o_D2241 = o_D2241_complete[5:]
SDR26t_D2241 = [1.524, 1.6256, 1.8542, 2.3114, 2.794, 3.429, 3.9116, 4.3942, 5.4356, 6.477, 8.4328, 10.4902, 12.446, 13.6652, 15.621, 17.5768, 19.5326, 23.4442, 29.3116, 35.179]
SDR26i_D2241 = [30.353, 38.9128, 44.5516, 55.7022, 67.437, 82.042, 93.7768, 105.5116, 130.429, 155.321, 202.2094, 252.0696, 298.958, 328.2696, 375.158, 422.0464, 468.9348, 562.7116, 703.3768, 844.042]

NPSDR21_D2241 = NPS_D2241[4:]
SDR21o_D2241 = o_D2241_complete[4:]
SDR21t_D2241 = [1.524, 1.6002, 2.0066, 2.286, 2.8702, 3.4798, 4.2418, 4.826, 5.4356, 6.731, 8.0264, 10.414, 12.9794, 15.3924, 16.9164, 19.3548, 21.7678, 24.1808, 29.0322, 36.2712, 43.5356]
SDR21i_D2241 = [23.622, 30.2006, 38.1508, 43.688, 54.5846, 66.0654, 80.4164, 91.948, 103.4288, 127.8382, 152.2222, 198.247, 247.0912, 293.0652, 321.7672, 367.6904, 413.6644, 459.6384, 551.5356, 689.4576, 827.3288]

NPSDR17_D2241 = NPS_D2241[4:]
SDR17o_D2241 = o_D2241_complete[4:]
SDR17t_D2241 = [1.5748, 1.9558, 2.4892, 2.8448, 3.556, 4.2926, 5.2324, 5.969, 6.731, 8.3058, 9.906, 12.9032, 16.0528, 19.05, 20.9042, 23.9014, 26.8986, 29.8704, 35.8648, 44.831, 53.7972]
SDR17i_D2241 = [23.5204, 29.4894, 37.1856, 42.5704, 53.213, 64.4398, 78.4352, 89.662, 100.838, 124.6886, 148.463, 193.2686, 240.9444, 285.75, 313.7916, 358.5972, 403.4028, 448.2592, 537.8704, 672.338, 806.8056]

NPSDR135_D2241 = NPS_D2241[:15]
SDR135o_D2241 = o_D2241_complete[:15]
SDR135t_D2241 = [1.524, 1.524, 1.524, 1.5748, 1.9812, 2.4638, 3.1242, 3.5814, 4.4704, 5.4102, 6.5786, 7.5184, 8.4582, 10.4648, 12.4714]
SDR135i_D2241 = [7.239, 10.668, 14.097, 18.1864, 22.7076, 28.4734, 35.9156, 41.0972, 51.3842, 62.2046, 75.7428, 86.5632, 97.3836, 120.3706, 143.3322]

# D2241 in Copper tube sizes
NPS_CTS_D2241 = [0.5, 0.75, 1, 1.25, 1.5, 2]
o_CTS_D2241 = [15.875, 22.225, 28.575, 34.925, 41.275, 53.975]

NPSDR21_D2241CTS = NPS_CTS_D2241[2:]
SDR21o_D2241CTS = o_CTS_D2241[2:]
SDR21t_D2241CTS = [1.524, 1.651, 1.9558, 2.5654]
SDR21i_D2241CTS = [25.527, 31.623, 37.3634, 48.8442]

NPSDR17_D2241CTS = NPS_CTS_D2241[1:]
SDR17o_D2241CTS = o_CTS_D2241[1:]
SDR17t_D2241CTS = [1.524, 1.6764, 2.0574, 2.4384, 3.175]
SDR17i_D2241CTS = [19.177, 25.2222, 30.8102, 36.3982, 47.625]

NPSDR135_D2241CTS = NPS_CTS_D2241
SDR135o_D2241CTS = o_CTS_D2241
SDR135t_D2241CTS = [1.524, 1.651, 2.1082, 2.5908, 3.048, 3.9878]
SDR135i_D2241CTS = [12.827, 18.923, 24.3586, 29.7434, 35.179, 45.9994]

NPSDR11_D2241CTS = NPS_CTS_D2241
SDR11o_D2241CTS = o_CTS_D2241
SDR11t_D2241CTS = [1.524, 2.032, 2.5908, 3.175, 3.7592, 4.9022]
SDR11i_D2241CTS = [12.827, 18.161, 23.3934, 28.575, 33.7566, 44.1706]

NPS_D2241PIP = [6, 8, 10, 12, 15, 18, 21, 24, 27]
o_D2241PIP = [155.956, 207.264, 259.08, 310.896, 388.62, 475.0054, 559.9938, 629.9962, 710.0062]

NPSDR81_D2241PIP = NPS_D2241PIP[:5]
SDR81o_D2241PIP = o_D2241PIP[:5]
SDR81t_D2241PIP = [1.9304, 2.5654, 3.2004, 3.8354, 4.8006]
SDR81i_D2241PIP = [152.0952, 202.1332, 252.6792, 303.2252, 379.0188]

NPSDR51_D2241PIP = NPS_D2241PIP
SDR51o_D2241PIP = o_D2241PIP
SDR51t_D2241PIP = [3.048, 4.064, 5.08, 6.096, 7.62, 9.2964, 10.9728, 12.3444, 13.9192]
SDR51i_D2241PIP = [149.86, 199.136, 248.92, 298.704, 373.38, 456.4126, 538.0482, 605.3074, 682.1678]

NPSDR41_D2241PIP = NPS_D2241PIP
SDR41o_D2241PIP = o_D2241PIP
SDR41t_D2241PIP = [3.81, 5.0546, 6.3246, 7.5946, 9.4742, 11.5824, 13.6652, 15.367, 17.3228]
SDR41i_D2241PIP = [148.336, 197.1548, 246.4308, 295.7068, 369.6716, 451.8406, 532.6634, 599.2622, 675.3606]

NPSDR35_D2241PIP = NPS_D2241PIP[4:]
SDR35o_D2241PIP = o_D2241PIP[4:]
SDR35t_D2241PIP = [11.0998, 13.5636, 16.002, 18.0086, 20.2946]
SDR35i_D2241PIP = [366.4204, 447.8782, 527.9898, 593.979, 669.417]

NPSDR325_D2241PIP = NPS_D2241PIP
SDR325o_D2241PIP = o_D2241PIP
SDR325t_D2241PIP = [4.8006, 6.3754, 7.9756, 9.5758, 11.9634, 14.605, 17.2212, 19.3802, 21.844]
SDR325i_D2241PIP = [146.3548, 194.5132, 243.1288, 291.7444, 364.6932, 445.7954, 525.5514, 591.2358, 666.3182]

NPSDR26_D2241PIP = NPS_D2241PIP[4:]
SDR26o_D2241PIP = o_D2241PIP[4:]
SDR26t_D2241PIP = [14.9352, 18.2626, 21.5392, 24.2316, 27.305]
SDR26i_D2241PIP = [358.7496, 438.4802, 516.9154, 581.533, 655.3962]

NPSDR21_D2241PIP = NPS_D2241PIP[4:5]
SDR21o_D2241PIP = o_D2241PIP[4:5]
SDR21t_D2241PIP = [18.4912]
SDR21i_D2241PIP = [351.6376]

# ASTM F441 IPS - has an independent SI and IPS data set

NPS_F441 = [0.25, 0.375, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 3.5, 4, 5, 6, 8, 10, 12, 14, 16]
DN_F441 = [8, 10, 15, 20, 25, 32, 40, 50, 65, 80, 90, 100, 125, 150, 200, 250, 300, 350, 400]

S80o_F441IPS = S40o_F441IPS = [13.716, 17.145, 21.336, 26.67, 33.401, 42.164, 48.26, 60.325, 73.025, 88.9, 101.6, 114.3, 141.3002, 168.275, 219.075, 273.05, 323.85, 355.6, 406.4]
S80o_F441SI = S40o_F441SI = [13.7, 17.1, 21.3, 26.7, 33.4, 42.2, 48.3, 60.3, 73, 88.9, 101.6, 114.3, 141.3, 168.3, 219.1, 273.1, 323.9, 355.6, 406.4]

S40t_F441IPS = [2.2352, 2.3114, 2.7686, 2.8702, 3.3782, 3.556, 3.683, 3.9116, 5.1562, 5.4864, 5.7404, 6.0198, 6.5532, 7.112, 8.1788, 9.271, 10.3124, 11.0998, 12.7]
S40t_F441SI = [2.24, 2.31, 2.77, 2.87, 3.38, 3.56, 3.68, 3.91, 5.16, 5.49, 5.74, 6.02, 6.55, 7.11, 8.18, 9.27, 10.31, 11.1, 12.7]
S40i_F441IPS = [9.2456, 12.5222, 15.7988, 20.9296, 26.6446, 35.052, 40.894, 52.5018, 62.7126, 77.9272, 90.1192, 102.2604, 128.1938, 154.051, 202.7174, 254.508, 303.2252, 333.4004, 381.0]
S40i_F441SI = [9.22, 12.48, 15.76, 20.96, 26.64, 35.08, 40.94, 52.48, 62.68, 77.92, 90.12, 102.26, 128.2, 154.08, 202.74, 254.56, 303.28, 333.4, 381.0]

S80t_F441IPS = [3.0226, 3.2004, 3.7338, 3.9116, 4.5466, 4.8514, 5.08, 5.5372, 7.0104, 7.62, 8.0772, 8.5598, 9.525, 10.9728, 12.7, 15.0622, 17.4498, 19.05, 21.4122]
S80t_F441SI = [3.02, 3.2, 3.73, 3.91, 4.55, 4.85, 5.08, 5.54, 7.01, 7.62, 8.08, 8.56, 9.52, 10.97, 12.7, 15.06, 17.45, 19.05, 21.14]
S80i_F441IPS = [7.6708, 10.7442, 13.8684, 18.8468, 24.3078, 32.4612, 38.1, 49.2506, 59.0042, 73.66, 85.4456, 97.1804, 122.2502, 146.3294, 193.675, 242.9256, 288.9504, 317.5, 363.5756]
S80i_F441SI = [7.66, 10.7, 13.84, 18.88, 24.3, 32.5, 38.14, 49.22, 58.98, 73.66, 85.44, 97.18, 122.26, 146.36, 193.7, 242.98, 289.0, 317.5, 364.12]

# ASTM F2619
# Note size 26" is missing in the table but the size it should be is obvious.
NPS_F2619 = [0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 3, 4, 5, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30, 32, 34, 36, 42, 48, 54, 63, 65]
o_F2619IPS = [21.336, 26.67, 33.401, 42.164, 48.26, 60.325, 73.025, 88.9, 114.3, 141.3002, 168.275, 219.075, 273.05, 323.85, 355.6, 406.4, 457.2, 508.0, 558.8, 609.6, 660.4, 711.2, 762.0, 812.8, 863.6, 914.4, 1066.8, 1219.2, 1371.6, 1600.2, 1651.0]
o_F2619SI = [21.34, 26.67, 33.4, 42.16, 48.26, 60.33, 73.03, 88.9, 114.3, 141.3, 168.28, 219.08, 273.05, 323.85, 355.6, 406.4, 457.2, 508.0, 558.8, 609.6, 660.4, 711.2, 762.0, 812.8, 863.6, 914.4, 1066.8, 1219.2, 1371.6, 1600.2, 1651.0]

NPSDR7_F2619 = NPS_F2619[:20]
SDR7o_F2619IPS = o_F2619IPS[:20]
SDR7o_F2619SI = o_F2619SI[:20]
SDR7t_F2619IPS = [3.048, 3.81, 4.7752, 6.0198, 6.8834, 8.6106, 10.4394, 12.7, 16.3322, 20.193, 24.0284, 31.2928, 39.0144, 46.2534, 50.8, 58.0644, 65.3034, 72.5678, 79.8322, 87.0966]
SDR7t_F2619SI = [3.05, 3.81, 4.78, 6.02, 6.88, 8.61, 10.43, 12.7, 16.33, 20.19, 24.03, 31.29, 39.01, 46.25, 50.8, 58.06, 65.3, 72.57, 79.83, 87.1]
SDR7i_F2619IPS = [15.24, 19.05, 23.8506, 30.1244, 34.4932, 43.1038, 52.1462, 63.5, 81.6356, 100.9142, 120.2182, 156.4894, 195.0212, 231.3432, 254.0, 290.2712, 326.5932, 362.8644, 399.1356, 435.4068]
SDR7i_F2619SI = [15.24, 19.05, 23.84, 30.12, 34.5, 43.11, 52.17, 63.5, 81.64, 100.92, 120.22, 156.5, 195.03, 231.35, 254.0, 290.28, 326.6, 362.86, 399.14, 435.4]

NPSDR73_F2619 = NPS_F2619[:20]
SDR73o_F2619IPS = o_F2619IPS[:20]
SDR73o_F2619SI = o_F2619SI[:20]
SDR73t_F2619IPS = [2.921, 3.6576, 4.572, 5.7658, 6.604, 8.255, 10.0076, 12.1666, 15.6464, 19.3548, 23.0632, 30.0228, 37.4142, 44.3738, 48.7172, 55.6768, 62.6364, 69.596, 76.5556, 83.5152]
SDR73t_F2619SI = [2.92, 3.66, 4.57, 5.77, 6.6, 8.26, 10.0, 12.17, 15.65, 19.36, 23.06, 30.02, 37.41, 44.37, 48.72, 55.68, 62.64, 69.6, 76.56, 83.52]
SDR73i_F2619IPS = [15.494, 19.3548, 24.257, 30.6324, 35.052, 43.815, 53.0098, 64.5668, 83.0072, 102.5906, 122.1486, 159.0294, 198.2216, 235.1024, 258.1656, 295.0464, 331.9272, 368.808, 405.6888, 442.5696]
SDR73i_F2619SI = [15.5, 19.35, 24.26, 30.62, 35.06, 43.81, 53.03, 64.56, 83.0, 102.58, 122.16, 159.04, 198.23, 235.11, 258.16, 295.04, 331.92, 368.8, 405.68, 442.56]

NPSDR9_F2619 = NPS_F2619[:23]
SDR9o_F2619IPS = o_F2619IPS[:23]
SDR9o_F2619SI = o_F2619SI[:23]
SDR9t_F2619IPS = [2.3622, 2.9718, 3.7084, 4.6736, 5.3594, 6.7056, 8.1026, 9.8806, 12.7, 15.6972, 18.6944, 24.3332, 30.3276, 35.9918, 39.5224, 45.1612, 50.8, 56.4388, 62.0776, 67.7418, 73.3806, 79.0194, 84.6582]
SDR9t_F2619SI = [2.36, 2.97, 3.71, 4.67, 5.36, 6.71, 8.1, 9.88, 12.7, 15.7, 18.69, 24.33, 30.33, 35.99, 39.52, 45.16, 50.8, 56.44, 62.08, 67.74, 73.38, 79.02, 84.66]
SDR9i_F2619IPS = [16.6116, 20.7264, 25.9842, 32.8168, 37.5412, 46.9138, 56.8198, 69.1388, 88.9, 109.9058, 130.8862, 170.4086, 212.3948, 251.8664, 276.5552, 316.0776, 355.6, 395.1224, 434.6448, 474.1164, 513.6388, 553.1612, 592.6836]
SDR9i_F2619SI = [16.62, 20.73, 25.98, 32.82, 37.54, 46.91, 56.83, 69.14, 88.9, 109.9, 130.9, 170.42, 212.39, 251.87, 276.56, 316.08, 355.6, 395.12, 434.64, 474.12, 513.64, 553.16, 592.68]

NPSDR11_F2619 = NPS_F2619[:26]
SDR11o_F2619IPS = o_F2619IPS[:26]
SDR11o_F2619SI = o_F2619SI[:26]
SDR11t_F2619IPS = [1.9304, 2.413, 3.048, 3.8354, 4.3942, 5.4864, 6.6294, 8.0772, 10.3886, 12.8524, 15.2908, 19.9136, 24.8158, 29.4386, 32.3342, 36.957, 41.5544, 46.1772, 50.8, 55.4228, 60.0456, 64.643, 69.2658, 73.8886, 78.5114, 83.1342]
SDR11t_F2619SI = [1.93, 2.41, 3.05, 3.84, 4.39, 5.49, 6.63, 8.08, 10.39, 12.85, 15.29, 19.91, 24.82, 29.44, 32.33, 36.96, 41.55, 46.18, 50.8, 55.42, 60.05, 62.23, 69.27, 73.89, 78.51, 83.13]
SDR11i_F2619IPS = [17.4752, 21.844, 27.305, 34.4932, 39.4716, 49.3522, 59.7662, 72.7456, 93.5228, 115.5954, 137.6934, 179.2478, 223.4184, 264.9728, 290.9316, 332.486, 374.0912, 415.6456, 457.2, 498.7544, 540.3088, 581.914, 623.4684, 665.0228, 706.5772, 748.1316]
SDR11i_F2619SI = [17.48, 21.85, 27.3, 34.48, 39.48, 49.35, 59.77, 72.74, 93.52, 115.6, 137.7, 179.26, 223.41, 264.97, 290.94, 332.48, 374.1, 415.64, 457.2, 498.76, 540.3, 586.74, 623.46, 665.02, 706.58, 748.14]

NPSDR135_F2619 = NPS_F2619[:27]
SDR135o_F2619IPS = o_F2619IPS[:27]
SDR135o_F2619SI = o_F2619SI[:27]
SDR135t_F2619IPS = [1.5748, 1.9812, 2.4638, 3.1242, 3.5814, 4.4704, 5.4102, 6.5786, 8.4582, 10.4648, 12.4714, 16.2306, 20.2184, 23.9776, 26.3398, 30.099, 33.8582, 37.6174, 41.402, 45.1612, 48.9204, 52.6796, 56.4388, 60.198, 63.9826, 67.7418, 79.0194]
SDR135t_F2619SI = [1.58, 1.98, 2.46, 3.12, 3.58, 4.47, 5.41, 6.58, 8.46, 10.47, 12.47, 16.23, 20.22, 23.98, 26.34, 30.1, 33.86, 37.62, 41.4, 45.16, 48.92, 52.68, 56.44, 60.2, 63.98, 67.74, 79.02]
SDR135i_F2619IPS = [18.1864, 22.7076, 28.4734, 35.9156, 41.0972, 51.3842, 62.2046, 75.7428, 97.3836, 120.3706, 143.3322, 186.6138, 232.6132, 275.8948, 302.9204, 346.202, 389.4836, 432.7652, 475.996, 519.2776, 562.5592, 605.8408, 649.1224, 692.404, 735.6348, 778.9164, 908.7612]
SDR135i_F2619SI = [18.18, 22.71, 28.48, 35.92, 41.1, 51.39, 62.21, 75.74, 97.38, 120.36, 143.34, 186.62, 232.61, 275.89, 302.92, 346.2, 389.48, 432.76, 476.0, 519.28, 562.56, 605.84, 649.12, 692.4, 735.64, 778.92, 908.76]

NPSDR17_F2619 = NPS_F2619[5:29]
SDR17o_F2619IPS = o_F2619IPS[5:29]
SDR17o_F2619SI = o_F2619SI[5:29]
SDR17t_F2619IPS = [3.556, 4.2926, 5.2324, 6.731, 8.3058, 9.906, 12.8778, 16.0528, 19.05, 20.9296, 23.9014, 26.8986, 29.8704, 32.8676, 35.8648, 38.8366, 41.8338, 44.831, 47.8028, 50.8, 53.7972, 62.7634, 71.7296, 80.6704]
SDR17t_F2619SI = [3.56, 4.29, 5.23, 6.73, 8.31, 9.91, 12.88, 16.05, 19.05, 20.93, 23.9, 26.9, 29.87, 32.87, 35.86, 38.84, 41.83, 44.83, 47.8, 50.8, 53.8, 62.76, 71.73, 80.67]
SDR17i_F2619IPS = [53.213, 64.4398, 78.4352, 100.838, 124.6886, 148.463, 193.3194, 240.9444, 285.75, 313.7408, 358.5972, 403.4028, 448.2592, 493.0648, 537.8704, 582.7268, 627.5324, 672.338, 717.1944, 762.0, 806.8056, 941.2732, 1075.7408, 1210.2592]
SDR17i_F2619SI = [53.21, 64.45, 78.44, 100.84, 124.68, 148.46, 193.32, 240.95, 285.75, 313.74, 358.6, 403.4, 448.26, 493.06, 537.88, 582.72, 627.54, 672.34, 717.2, 762.0, 806.8, 941.28, 1075.74, 1210.26]

NPSDR21_F2619 = NPS_F2619[6:]
SDR21o_F2619IPS = o_F2619IPS[6:]
SDR21o_F2619SI = o_F2619SI[6:]
SDR21t_F2619IPS = [3.4798, 4.2418, 5.4356, 6.731, 8.001, 10.4394, 13.0048, 15.4178, 16.9418, 19.3548, 21.7678, 24.1808, 26.6192, 29.0322, 31.4452, 33.8582, 36.2966, 38.7096, 41.1226, 43.5356, 50.8, 58.0644, 65.3034, 76.2, 78.613]
SDR21t_F2619SI = [3.48, 4.24, 5.44, 6.73, 8, 10.44, 13, 15.42, 16.94, 19.35, 21.77, 24.18, 26.62, 29.03, 31.45, 33.86, 36.3, 38.71, 41.12, 43.54, 50.8, 58.06, 65.3, 76.2, 78.61]
SDR21i_F2619IPS = [66.0654, 80.4164, 103.4288, 127.8382, 152.273, 198.1962, 247.0404, 293.0144, 321.7164, 367.6904, 413.6644, 459.6384, 505.5616, 551.5356, 597.5096, 643.4836, 689.4068, 735.3808, 781.3548, 827.3288, 965.2, 1103.0712, 1240.9932, 1447.8, 1493.774]
SDR21i_F2619SI = [66.07, 80.42, 103.42, 127.84, 152.28, 198.2, 247.05, 293.01, 321.72, 367.7, 413.66, 459.64, 505.56, 551.54, 597.5, 643.48, 689.4, 735.38, 781.36, 827.32, 965.2, 1103.08, 1241.0, 1447.8, 1493.78]

NPSDR26_F2619 = NPS_F2619[8:]
SDR26o_F2619IPS = o_F2619IPS[8:]
SDR26o_F2619SI = o_F2619SI[8:]
SDR26t_F2619IPS = [4.3942, 5.4356, 6.477, 8.4328, 10.4902, 12.446, 13.6652, 15.621, 17.5768, 19.5326, 21.4884, 23.4442, 25.4, 27.3558, 29.3116, 31.2674, 33.2232, 35.179, 41.021, 46.8884, 52.7558, 61.5442, 63.5]
SDR26t_F2619SI = [4.39, 5.43, 6.48, 8.43, 10.49, 12.45, 13.67, 15.62, 17.58, 20.22, 21.49, 23.44, 25.4, 27.36, 29.31, 31.27, 33.22, 35.18, 41.02, 46.89, 52.76, 61.55, 63.5]
SDR26i_F2619IPS = [105.5116, 130.429, 155.321, 202.2094, 252.0696, 298.958, 328.2696, 375.158, 422.0464, 468.9348, 515.8232, 562.7116, 609.6, 656.4884, 703.3768, 750.2652, 797.1536, 844.042, 984.758, 1125.4232, 1266.0884, 1477.1116, 1524.0]
SDR26i_F2619SI = [105.52, 130.44, 155.32, 202.22, 252.07, 298.95, 328.26, 375.16, 422.04, 467.56, 515.82, 562.72, 609.6, 656.48, 703.38, 750.26, 797.16, 844.04, 984.76, 1125.42, 1266.08, 1477.1, 1524.0]

NPSDR325_F2619 = NPS_F2619[8:]
SDR325o_F2619IPS = o_F2619IPS[8:]
SDR325o_F2619SI = o_F2619SI[8:]
SDR325t_F2619IPS = [3.5052, 4.3434, 5.1816, 6.731, 8.4074, 9.9568, 10.9474, 12.4968, 14.0716, 15.621, 17.1958, 18.7452, 20.32, 21.8948, 23.4442, 25.019, 26.5684, 28.1432, 32.8168, 37.5158, 42.2148, 49.2252, 50.8]
SDR325t_F2619SI = [3.51, 4.35, 5.18, 6.73, 8.41, 9.96, 10.95, 12.5, 14.07, 15.62, 17.2, 18.75, 20.32, 21.89, 23.44, 25.02, 26.57, 28.14, 32.82, 37.52, 42.21, 49.22, 50.8]
SDR325i_F2619IPS = [107.2896, 132.6134, 157.9118, 205.613, 256.2352, 303.9364, 333.7052, 381.4064, 429.0568, 476.758, 524.4084, 572.1096, 619.76, 667.4104, 715.1116, 762.762, 810.4632, 858.1136, 1001.1664, 1144.1684, 1287.1704, 1501.7496, 1549.4]
SDR325i_F2619SI = [107.28, 132.6, 157.92, 205.62, 256.23, 303.93, 333.7, 381.4, 429.06, 476.76, 524.4, 572.1, 619.76, 667.42, 715.12, 762.76, 810.46, 858.12, 1001.16, 1144.16, 1287.18, 1501.76, 1549.4]

schedule_lookup = { '40': (NPS40, S40i, S40o, S40t),
                    '5': (NPS5, S5i, S5o, S5t),
                    '10': (NPS10, S10i, S10o, S10t),
                    '20': (NPS20, S20i, S20o, S20t),
                    '30': (NPS30, S30i, S30o, S30t),
                    '60': (NPS60, S60i, S60o, S60t),
                    '80': (NPS80, S80i, S80o, S80t),
                    '100': (NPS100, S100i, S100o, S100t),
                    '120': (NPS120, S120i, S120o, S120t),
                    '140': (NPS140, S140i, S140o, S140t),
                    '160': (NPS160, S160i, S160o, S160t),
                    'STD': (NPSSTD, STDi, STDo, STDt),
                    'XS': (NPSXS, XSi, XSo, XSt),
                    'XXS': (NPSXXS, XXSi, XXSo, XXSt),

                    '5S': (NPSS5, SS5i, SS5o, SS5t),
                    '10S': (NPSS10, SS10i, SS10o, SS10t),
                    '40S': (NPSS40, SS40i, SS40o, SS40t),
                    '80S': (NPSS80, SS80i, SS80o, SS80t),

                    '40D1527': (NPS_D1527, S40i_D1527, S40o_D1527, S40t_D1527),
                    '80D1527': (NPS_D1527, S80i_D1527, S80o_D1527, S80t_D1527),

                    'ABSD2680': (NPS_D2680, SABSi_D2680, SABSo_D2680, SABSt_D2680),
                    'PVCD2680': (NPS_D2680, SPVCi_D2680, SPVCo_D2680, SPVCt_D2680),
                    'DR25C900': (NPS_C900, SDR25i_C900, SDR25o_C900, SDR25t_C900),
                    'DR18C900': (NPS_C900, SDR18i_C900, SDR18o_C900, SDR18t_C900),
                    'DR14C900': (NPS_C900, SDR14i_C900, SDR14o_C900, SDR14t_C900),

                    'CIDR51C905': (NPSCIDR51_C905, SCIDR51i_C905, SCIDR51o_C905, SCIDR51t_C905),
                    'CIDR41C905': (NPSCIDR41_C905, SCIDR41i_C905, SCIDR41o_C905, SCIDR41t_C905),
                    'CIDR325C905': (NPSCIDR325_C905, SCIDR325i_C905, SCIDR325o_C905, SCIDR325t_C905),
                    'CIDR25C905': (NPSCIDR25_C905, SCIDR25i_C905, SCIDR25o_C905, SCIDR25t_C905),
                    'CIDR21C905': (NPSCIDR21_C905, SCIDR21i_C905, SCIDR21o_C905, SCIDR21t_C905),
                    'CIDR18C905': (NPSCIDR18_C905, SCIDR18i_C905, SCIDR18o_C905, SCIDR18t_C905),
                    'CIDR14C905': (NPSCIDR14_C905, SCIDR14i_C905, SCIDR14o_C905, SCIDR14t_C905),

                    'IPSDR21': (NPSIPSDR21_C905, SIPSDR21i_C905, SIPSDR21o_C905, SIPSDR21t_C905),
                    'IPSDR26': (NPSIPSDR26_C905, SIPSDR26i_C905, SIPSDR26o_C905, SIPSDR26t_C905),
                    'IPSDR325': (NPSIPSDR325_C905, SIPSDR325i_C905, SIPSDR325o_C905, SIPSDR325t_C905),
                    'IPSDR41': (NPSIPSDR41_C905, SIPSDR41i_C905, SIPSDR41o_C905, SIPSDR41t_C905),

                    'PS115F679': (NPS_F679, SPS115i_F679, SPS115o_F679, SPS115t_F679),
                    'PS75F679': (NPS_F679, SPS75i_F679, SPS75o_F679, SPS75t_F679),
                    'PS46F679': (NPS_F679, SPS46i_F679, SPS46o_F679, SPS46t_F679),

                    'PVCD2665': (NPS_D2665, SPVCi_D2665, SPVCo_D2665, SPVCt_D2665),

                    '40D1785': (NPS_D1785, S40i_D1785, S40o_D1785, S40t_D1785),
                    '80D1785': (NPS_D1785, S80i_D1785, S80o_D1785, S80t_D1785),
                    '120D1785': (NPS120_D1785, S120i_D1785, S120o_D1785, S120t_D1785),

                    'DR135D2241': (NPSDR135_D2241, SDR135i_D2241, SDR135o_D2241, SDR135t_D2241),
                    'DR17D2241': (NPSDR17_D2241, SDR17i_D2241, SDR17o_D2241, SDR17t_D2241),
                    'DR21D2241': (NPSDR21_D2241, SDR21i_D2241, SDR21o_D2241, SDR21t_D2241),
                    'DR26D2241': (NPSDR26_D2241, SDR26i_D2241, SDR26o_D2241, SDR26t_D2241),
                    'DR325D2241': (NPSDR325_D2241, SDR325i_D2241, SDR325o_D2241, SDR325t_D2241),
                    'DR41D2241': (NPSDR41_D2241, SDR41i_D2241, SDR41o_D2241, SDR41t_D2241),
                    'DR64D2241': (NPSDR64_D2241, SDR64i_D2241, SDR64o_D2241, SDR64t_D2241),

                    'DR21D2241CTS': (NPSDR21_D2241CTS, SDR21i_D2241CTS, SDR21o_D2241CTS, SDR21t_D2241CTS),
                    'DR17D2241CTS': (NPSDR17_D2241CTS, SDR17i_D2241CTS, SDR17o_D2241CTS, SDR17t_D2241CTS),
                    'DR135D2241CTS': (NPSDR135_D2241CTS, SDR135i_D2241CTS, SDR135o_D2241CTS, SDR135t_D2241CTS),
                    'DR11D2241CTS': (NPSDR11_D2241CTS, SDR11i_D2241CTS, SDR11o_D2241CTS, SDR11t_D2241CTS),

                    'DR21D2241PIP': (NPSDR21_D2241PIP, SDR21i_D2241PIP, SDR21o_D2241PIP, SDR21t_D2241PIP),
                    'DR26D2241PIP': (NPSDR26_D2241PIP, SDR26i_D2241PIP, SDR26o_D2241PIP, SDR26t_D2241PIP),
                    'DR325D2241PIP': (NPSDR325_D2241PIP, SDR325i_D2241PIP, SDR325o_D2241PIP, SDR325t_D2241PIP),
                    'DR35D2241PIP': (NPSDR35_D2241PIP, SDR35i_D2241PIP, SDR35o_D2241PIP, SDR35t_D2241PIP),
                    'DR41D2241PIP': (NPSDR41_D2241PIP, SDR41i_D2241PIP, SDR41o_D2241PIP, SDR41t_D2241PIP),
                    'DR51D2241PIP': (NPSDR51_D2241PIP, SDR51i_D2241PIP, SDR51o_D2241PIP, SDR51t_D2241PIP),
                    'DR81D2241PIP': (NPSDR81_D2241PIP, SDR81i_D2241PIP, SDR81o_D2241PIP, SDR81t_D2241PIP),

                    'S40F441IPS': (NPS_F441, S40i_F441IPS, S40o_F441IPS, S40t_F441IPS),
                    'S80F441IPS': (NPS_F441, S80i_F441IPS, S80o_F441IPS, S80t_F441IPS),
                    'S40F441SI': (DN_F441, S40i_F441SI, S40o_F441SI, S40t_F441SI),
                    'S80F441SI': (DN_F441, S80i_F441SI, S80o_F441SI, S80t_F441SI),

                    'DR325F2619SI': (NPSDR325_F2619, SDR325i_F2619SI, SDR325o_F2619SI, SDR325t_F2619SI),
                    'DR26F2619SI': (NPSDR26_F2619, SDR26i_F2619SI, SDR26o_F2619SI, SDR26t_F2619SI),
                    'DR21F2619SI': (NPSDR21_F2619, SDR21i_F2619SI, SDR21o_F2619SI, SDR21t_F2619SI),
                    'DR17F2619SI': (NPSDR17_F2619, SDR17i_F2619SI, SDR17o_F2619SI, SDR17t_F2619SI),
                    'DR135F2619SI': (NPSDR135_F2619, SDR135i_F2619SI, SDR135o_F2619SI, SDR135t_F2619SI),
                    'DR11F2619SI': (NPSDR11_F2619, SDR11i_F2619SI, SDR11o_F2619SI, SDR11t_F2619SI),
                    'DR9F2619SI': (NPSDR9_F2619, SDR9i_F2619SI, SDR9o_F2619SI, SDR9t_F2619SI),
                    'DR73F2619SI': (NPSDR73_F2619, SDR73i_F2619SI, SDR73o_F2619SI, SDR73t_F2619SI),
                    'DR7F2619SI': (NPSDR7_F2619, SDR7i_F2619SI, SDR7o_F2619SI, SDR7t_F2619SI),

                    'DR325F2619IPS': (NPSDR325_F2619, SDR325i_F2619IPS, SDR325o_F2619IPS, SDR325t_F2619IPS),
                    'DR26F2619IPS': (NPSDR26_F2619, SDR26i_F2619IPS, SDR26o_F2619IPS, SDR26t_F2619IPS),
                    'DR21F2619IPS': (NPSDR21_F2619, SDR21i_F2619IPS, SDR21o_F2619IPS, SDR21t_F2619IPS),
                    'DR17F2619IPS': (NPSDR17_F2619, SDR17i_F2619IPS, SDR17o_F2619IPS, SDR17t_F2619IPS),
                    'DR135F2619IPS': (NPSDR135_F2619, SDR135i_F2619IPS, SDR135o_F2619IPS, SDR135t_F2619IPS),
                    'DR11F2619IPS': (NPSDR11_F2619, SDR11i_F2619IPS, SDR11o_F2619IPS, SDR11t_F2619IPS),
                    'DR9F2619IPS': (NPSDR9_F2619, SDR9i_F2619IPS, SDR9o_F2619IPS, SDR9t_F2619IPS),
                    'DR73F2619IPS': (NPSDR73_F2619, SDR73i_F2619IPS, SDR73o_F2619IPS, SDR73t_F2619IPS),
                    'DR7F2619IPS': (NPSDR7_F2619, SDR7i_F2619IPS, SDR7o_F2619IPS, SDR7t_F2619IPS),
                    }


def nearest_pipe(Do=None, Di=None, NPS=None, schedule='40'):
    r'''Searches for and finds the nearest standard pipe size to a given
    specification. Acceptable inputs are:

    - Nominal pipe size
    - Nominal pipe size and schedule
    - Outer diameter `Do`
    - Outer diameter `Do` and schedule
    - Inner diameter `Di`
    - Inner diameter `Di` and schedule

    Acceptable schedules are: '5', '10', '20', '30', '40', '60', '80', '100',
    '120', '140', '160', 'STD', 'XS', 'XXS', '5S', '10S', '40S', '80S',
    '40D1527', '80D1527',
    'ABSD2680', 'PVCD2680',
    'DR25C900', 'DR18C900', 'DR14C900',
    'CIDR51C905', 'CIDR41C905', 'CIDR325C905', 'CIDR25C905', 'CIDR21C905', 'CIDR18C905', 'CIDR14C905',
    'IPSDR21', 'IPSDR26', 'IPSDR325', 'IPSDR41',
    'PS115F679', 'PS75F679', 'PS46F679',
    'PVCD2665',
    '40D1785', '80D1785', '120D1785',
    'DR135D2241', 'DR17D2241', 'DR21D2241', 'DR26D2241', 'DR325D2241', 'DR41D2241', 'DR64D2241',
    'DR21D2241CTS', 'DR17D2241CTS', 'DR135D2241CTS', 'DR11D2241CTS',
    'DR21D2241PIP', 'DR26D2241PIP', 'DR325D2241PIP', 'DR35D2241PIP', 'DR41D2241PIP', 'DR51D2241PIP', 'DR81D2241PIP',
    'S40F441IPS', 'S80F441IPS', 'S40F441SI', 'S80F441SI'
    'DR325F2619SI', 'DR26F2619SI', 'DR21F2619SI', 'DR17F2619SI', 'DR135F2619SI', 'DR11F2619SI', 'DR9F2619SI', 'DR73F2619SI', 'DR7F2619SI',
    'DR325F2619IPS', 'DR26F2619IPS', 'DR21F2619IPS', 'DR17F2619IPS', 'DR135F2619IPS', 'DR11F2619IPS', 'DR9F2619IPS', 'DR73F2619IPS', 'DR7F2619IPS',

    Parameters
    ----------
    Do : float
        Pipe outer diameter, [m]
    Di : float
        Pipe inner diameter, [m]
    NPS : float
        Nominal pipe size, [-]
    schedule : str
        String representing schedule size

    Returns
    -------
    NPS : float
        Nominal pipe size, [-]
    Di : float
        Pipe inner diameter, [m]
    Do : float
        Pipe outer diameter, [m]
    t : float
        Pipe wall thickness, [m]

    Notes
    -----
    Internal units within this function are mm.
    The imperial schedules are not quite identical to these value, but
    all rounding differences happen in the sub-0.1 mm level.

    Examples
    --------
    >>> nearest_pipe(Di=0.021)
    (1, 0.02664, 0.0334, 0.0033799999999999998)
    >>> nearest_pipe(Do=.273, schedule='5S')
    (10, 0.26630000000000004, 0.2731, 0.0034)

    References
    ----------
    .. [1] American National Standards Institute, and American Society of
       Mechanical Engineers. B36.10M-2004: Welded and Seamless Wrought Steel
       Pipe. New York: American Society of Mechanical Engineers, 2004.
    .. [2] American National Standards Institute, and American Society of
       Mechanical Engineers. B36-19M-2004: Stainless Steel Pipe.
       New York, N.Y.: American Society of Mechanical Engineers, 2004.
    .. [3] F17 Committee. "Specification for Acrylonitrile-Butadiene-Styrene
       (ABS) Plastic Pipe, Schedules 40 and 80." ASTM International.
       https://doi.org/10.1520/D1527-99R05.
    .. [4] F17 Committee. "Specification for Acrylonitrile-Butadiene-Styrene
       (ABS) and Poly(Vinyl Chloride) (PVC) Composite Sewer Piping." ASTM
       International. https://doi.org/10.1520/D2680-01R14.
    .. [5] AWWA-American Water Works Association. "AWWA C900-07 Polyvinyl
       Chloride (PVC) Pressure Pipe and Fabricated Fittings, 4 In. Through 12
       In. (100 Mm Through 300 Mm), for Water Transmission and Distribution."
    .. [6] AWWA-American Water Works Association. "AWWA C905-97 Polyvinyl
       Chloride (PVC) Pressure Pipe and Fabricated Fittings, 14 in. Through
       48 in. (350 Mm through 1,200 Mm), for Water Transmission and
       Distribution."
    .. [7] F17 Committee. "Specification for Poly(Vinyl Chloride) (PVC)
       Large-Diameter Plastic Gravity Sewer Pipe and Fittings." ASTM
       International. https://doi.org/10.1520/F0679-16.
    .. [8] F17 Committee. "Specification for Poly(Vinyl Chloride) (PVC) Plastic
       Drain, Waste, and Vent Pipe and Fittings." ASTM International.
       https://doi.org/10.1520/D2665-14.
    .. [9] F17 Committee. "Specification for Poly(Vinyl Chloride) (PVC) Plastic
       Pipe, Schedules 40, 80, and 120." ASTM International.
       https://doi.org/10.1520/D1785-15E01.
    .. [10] F17 Committee. "Specification for Chlorinated Poly(Vinyl Chloride)
       (CPVC) Plastic Pipe, Schedules 40 and 80." ASTM International.
       https://doi.org/10.1520/F0441_F0441M-15.
    .. [11] F17 Committee. "Specification for High-Density Polyethylene (PE)
       Line Pipe." ASTM International. https://doi.org/10.1520/F2619_F2619M-20.

    '''
    if Di:
        Di *= 1E3
    if Do:
        Do *= 1E3
    if NPS:
        NPS = float(NPS)

    def Di_lookup(Di, NPSes, Dis, Dos, ts):
        for i in range(len(Dis)): # Go up ascending list; once larger than specified, return
            if Dis[-1] < Di:
                return None
            if Dis[i] >= Di:
                _nps, _di, _do, _t = NPSes[i], Dis[i], Dos[i], ts[i]
                return (_nps, _di, _do, _t)
        raise ValueError('Di lookup failed')

    def Do_lookup(Do, NPSes, Dis, Dos, ts):
        for i in range(len(Dos)): # Go up ascending list; once larger than specified, return
            if Dos[-1] < Do:
                return None
            if Dos[i] >= Do:
                _nps, _di, _do, _t = NPSes[i], Dis[i], Dos[i], ts[i]
                return (_nps, _di, _do, _t)
        raise ValueError('Do lookup failed')

    def NPS_lookup(NPS, NPSes, Dis, Dos, ts):
        for i in range(len(NPSes)): # Go up ascending list; once larger than specified, return
            if NPSes[i] == NPS:
                _nps, _di, _do, _t = NPSes[i], Dis[i], Dos[i], ts[i]
                return (_nps, _di, _do, _t)
        raise ValueError('NPS not in list')


    # If accidentally given an numerical schedule, convert it to a string
    schedule_type = type(schedule)
    if schedule_type in (int, float):
        schedule = str(int(schedule))

    if schedule not in schedule_lookup:
        raise ValueError('Schedule not recognized')
    else:
        NPSes, Dis, Dos, ts = schedule_lookup[schedule]

    # Handle the three cases of different inputs
    if Di:
        nums = Di_lookup(Di, NPSes, Dis, Dos, ts)
    elif Do:
        nums = Do_lookup(Do, NPSes, Dis, Dos, ts)
    elif NPS:
        nums = NPS_lookup(NPS, NPSes, Dis, Dos, ts)

    if nums is None:
        raise ValueError('Pipe input is larger than max of selected schedule')
    _nps, _di, _do, _t = nums
    return _nps, _di/1E3, _do/1E3, _t/1E3


### Wire gauge schedules

# Stub's Steel Wire Gage
SSWG_integers = list(range(1, 81))
SSWG_inch = [0.227, 0.219, 0.212, 0.207, 0.204, 0.201, 0.199, 0.197, 0.194,
             0.191, 0.188, 0.185, 0.182, 0.18, 0.178, 0.175, 0.172, 0.168,
             0.164, 0.161, 0.157, 0.155, 0.153, 0.151, 0.148, 0.146, 0.143,
             0.139, 0.134, 0.127, 0.12, 0.115, 0.112, 0.11, 0.108, 0.106,
             0.103, 0.101, 0.099, 0.097, 0.095, 0.092, 0.088, 0.085, 0.081,
             0.079, 0.077, 0.075, 0.072, 0.069, 0.066, 0.063, 0.058, 0.055,
             0.05, 0.045, 0.042, 0.041, 0.04, 0.039, 0.038, 0.037, 0.036,
             0.035, 0.033, 0.032, 0.031, 0.03, 0.029, 0.027, 0.026, 0.024,
             0.023, 0.022, 0.02, 0.018, 0.016, 0.015, 0.014, 0.013]
#SSWG_SI = [round(i*inch, 7) for i in SSWG_inch] # 7 decimals for equal conversion
SSWG_SI = [0.0057658, 0.0055626, 0.0053848, 0.0052578, 0.0051816, 0.0051054,
           0.0050546, 0.0050038, 0.0049276, 0.0048514, 0.0047752, 0.004699,
           0.0046228, 0.004572, 0.0045212, 0.004445, 0.0043688, 0.0042672,
           0.0041656, 0.0040894, 0.0039878, 0.003937, 0.0038862, 0.0038354,
           0.0037592, 0.0037084, 0.0036322, 0.0035306, 0.0034036, 0.0032258,
           0.003048, 0.002921, 0.0028448, 0.002794, 0.0027432, 0.0026924,
           0.0026162, 0.0025654, 0.0025146, 0.0024638, 0.002413, 0.0023368,
           0.0022352, 0.002159, 0.0020574, 0.0020066, 0.0019558, 0.001905,
           0.0018288, 0.0017526, 0.0016764, 0.0016002, 0.0014732, 0.001397,
           0.00127, 0.001143, 0.0010668, 0.0010414, 0.001016, 0.0009906,
           0.0009652, 0.0009398, 0.0009144, 0.000889, 0.0008382, 0.0008128,
           0.0007874, 0.000762, 0.0007366, 0.0006858, 0.0006604, 0.0006096,
           0.0005842, 0.0005588, 0.000508, 0.0004572, 0.0004064, 0.000381,
           0.0003556, 0.0003302]


# British Standard Wire Gage (Imperial Wire Gage)
BSWG_integers = [0.143, .167, 0.2, 0.25, 0.33, 0.5] + list(range(51))
BSWG_inch = [0.5, 0.464, 0.432, 0.4, 0.372, 0.348, 0.324, 0.3, 0.276, 0.252, 0.232,
        0.212, 0.192, 0.176, 0.16, 0.144, 0.128, 0.116, 0.104, 0.092, 0.08,
        0.072, 0.064, 0.056, 0.048, 0.04, 0.036, 0.032, 0.028, 0.024, 0.022,
        0.02, 0.018, 0.0164, 0.0149, 0.0136, 0.0124, 0.0116, 0.0108, 0.01,
        0.0092, 0.0084, 0.0076, 0.0068, 0.006, 0.0052, 0.0048, 0.0044, 0.004,
        0.0036, 0.0032, 0.0028, 0.0024, 0.002, 0.0016, 0.0012, 0.001]
#BSWG_SI = [round(i*inch,8) for i in BSWG_inch] # 8 decimals for equal conversion
BSWG_SI = [0.0127, 0.0117856, 0.0109728, 0.01016, 0.0094488, 0.0088392,
           0.0082296, 0.00762, 0.0070104, 0.0064008, 0.0058928, 0.0053848,
           0.0048768, 0.0044704, 0.004064, 0.0036576, 0.0032512, 0.0029464,
           0.0026416, 0.0023368, 0.002032, 0.0018288, 0.0016256, 0.0014224,
           0.0012192, 0.001016, 0.0009144, 0.0008128, 0.0007112, 0.0006096,
           0.0005588, 0.000508, 0.0004572, 0.00041656, 0.00037846, 0.00034544,
           0.00031496, 0.00029464, 0.00027432, 0.000254, 0.00023368,
           0.00021336, 0.00019304, 0.00017272, 0.0001524, 0.00013208,
           0.00012192, 0.00011176, 0.0001016, 9.144e-05, 8.128e-05, 7.112e-05,
           6.096e-05, 5.08e-05, 4.064e-05, 3.048e-05, 2.54e-05]


# Music Wire Gauge
MWG_integers = [45, 44, 43, 42, 41, 40, 39, 38, 37, 36, 35, 34, 33, 32, 31, 30,
                29, 28, 27, 26, 25, 24, 23, 22, 21, 20, 19, 18, 17, 16, 15, 14,
                13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0, 0.5, 0.33, 0.25,
                0.2, 0.167]
#MWG_integers = [.167, 0.2, 0.25, 0.33, 0.5] + list(range(46))
#MWG_inch = [0.004, 0.005, 0.006, 0.007, 0.008, 0.009, 0.01, 0.011, 0.012,
#            0.013, 0.014, 0.016, 0.018, 0.02, 0.022, 0.024, 0.026, 0.029,
#            0.031, 0.033, 0.035, 0.037, 0.039, 0.041, 0.043, 0.045, 0.047,
#            0.049, 0.051, 0.055, 0.059, 0.063, 0.067, 0.071, 0.075, 0.08,
#            0.085, 0.09, 0.095, 0.1, 0.106, 0.112, 0.118, 0.124, 0.13, 0.138,
#            0.146, 0.154, 0.162, 0.17, 0.18]
#MWG_SI = [round(i*inch,7) for i in MWG_inch] # 7 decimals for equal conversion
# Scale gets bigger instead of smaller; reverse for convenience
#MWG_integers.reverse()
#MWG_inch.reverse()

MWG_inch = [0.18, 0.17, 0.162, 0.154, 0.146, 0.138, 0.13, 0.124, 0.118, 0.112,
            0.106, 0.1, 0.095, 0.09, 0.085, 0.08, 0.075, 0.071, 0.067, 0.063,
            0.059, 0.055, 0.051, 0.049, 0.047, 0.045, 0.043, 0.041, 0.039,
            0.037, 0.035, 0.033, 0.031, 0.029, 0.026, 0.024, 0.022, 0.02,
            0.018, 0.016, 0.014, 0.013, 0.012, 0.011, 0.01, 0.009, 0.008,
            0.007, 0.006, 0.005, 0.004]
#MWG_SI.reverse()
MWG_SI = [0.004572, 0.004318, 0.0041148, 0.0039116, 0.0037084, 0.0035052,
          0.003302, 0.0031496, 0.0029972, 0.0028448, 0.0026924, 0.00254,
          0.002413, 0.002286, 0.002159, 0.002032, 0.001905, 0.0018034,
          0.0017018, 0.0016002, 0.0014986, 0.001397, 0.0012954, 0.0012446,
          0.0011938, 0.001143, 0.0010922, 0.0010414, 0.0009906, 0.0009398,
          0.000889, 0.0008382, 0.0007874, 0.0007366, 0.0006604, 0.0006096,
          0.0005588, 0.000508, 0.0004572, 0.0004064, 0.0003556, 0.0003302,
          0.0003048, 0.0002794, 0.000254, 0.0002286, 0.0002032, 0.0001778,
          0.0001524, 0.000127, 0.0001016]

# Steel Wire Gage -Also Washburn & Moen gage, American Steel gage;
# Wire Co.gage;  Roebling Wire Gages.
SWG_integers = [0.143, .167, 0.2, 0.25, 0.33, 0.5] + list(range(51))
SWG_inch = [0.49, 0.4615, 0.4305, 0.3938, 0.3625, 0.331, 0.3065, 0.283, 0.2625,
            0.2437, 0.2253, 0.207, 0.192, 0.177, 0.162, 0.1483, 0.135, 0.1205,
            0.1055, 0.0915, 0.08, 0.072, 0.0625, 0.054, 0.0475, 0.041, 0.0348,
            0.0318, 0.0286, 0.0258, 0.023, 0.0204, 0.0181, 0.0173, 0.0162,
            0.015, 0.014, 0.0132, 0.0128, 0.0118, 0.0104, 0.0095, 0.009,
            0.0085, 0.008, 0.0075, 0.007, 0.0066, 0.0062, 0.006, 0.0058,
            0.0055, 0.0052, 0.005, 0.0048, 0.0046, 0.0044]
#SWG_SI = [round(i*inch,8) for i in SWG_inch] # 8 decimals for equal conversion
SWG_SI = [0.012446, 0.0117221, 0.0109347, 0.01000252, 0.0092075, 0.0084074,
          0.0077851, 0.0071882, 0.0066675, 0.00618998, 0.00572262, 0.0052578,
          0.0048768, 0.0044958, 0.0041148, 0.00376682, 0.003429, 0.0030607,
          0.0026797, 0.0023241, 0.002032, 0.0018288, 0.0015875, 0.0013716,
          0.0012065, 0.0010414, 0.00088392, 0.00080772, 0.00072644, 0.00065532,
          0.0005842, 0.00051816, 0.00045974, 0.00043942, 0.00041148, 0.000381,
          0.0003556, 0.00033528, 0.00032512, 0.00029972, 0.00026416, 0.0002413,
          0.0002286, 0.0002159, 0.0002032, 0.0001905, 0.0001778, 0.00016764,
          0.00015748, 0.0001524, 0.00014732, 0.0001397, 0.00013208, 0.000127,
          0.00012192, 0.00011684, 0.00011176]

# American Wire or Brown & Sharpe Gage
AWG_integers = [.167, 0.2, 0.25, 0.33, 0.5] + list(range(51))
AWG_inch = [0.58, 0.5165, 0.46, 0.4096, 0.3648, 0.3249, 0.2893, 0.2576, 0.2294,
            0.2043, 0.1819, 0.162, 0.1443, 0.1285, 0.1144, 0.1019, 0.0907,
            0.0808, 0.072, 0.0641, 0.0571, 0.0508, 0.0453, 0.0403, 0.0359,
            0.032, 0.0285, 0.0253, 0.0226, 0.0201, 0.0179, 0.0159, 0.0142,
            0.0126, 0.0113, 0.01, 0.00893, 0.00795, 0.00708, 0.0063, 0.00561,
            0.005, 0.00445, 0.00396, 0.00353, 0.00314, 0.0028, 0.00249,
            0.00222, 0.00198, 0.00176, 0.00157, 0.0014, 0.00124, 0.00111,
            0.00099]
#AWG_SI = [round(i*inch,9) for i in AWG_inch] # 9 decimals for equal conversion
AWG_SI = [0.014732, 0.0131191, 0.011684, 0.01040384, 0.00926592, 0.00825246,
          0.00734822, 0.00654304, 0.00582676, 0.00518922, 0.00462026,
          0.0041148, 0.00366522, 0.0032639, 0.00290576, 0.00258826, 0.00230378,
          0.00205232, 0.0018288, 0.00162814, 0.00145034, 0.00129032,
          0.00115062, 0.00102362, 0.00091186, 0.0008128, 0.0007239, 0.00064262,
          0.00057404, 0.00051054, 0.00045466, 0.00040386, 0.00036068,
          0.00032004, 0.00028702, 0.000254, 0.000226822, 0.00020193,
          0.000179832, 0.00016002, 0.000142494, 0.000127, 0.00011303,
          0.000100584, 8.9662e-05, 7.9756e-05, 7.112e-05, 6.3246e-05,
          5.6388e-05, 5.0292e-05, 4.4704e-05, 3.9878e-05, 3.556e-05,
          3.1496e-05, 2.8194e-05, 2.5146e-05]

# Birmingham or Stub's Iron Wire Gage
BWG_integers = [0.2, 0.25, 0.33, 0.5] + list(range(37))
BWG_inch = [0.5, 0.454, 0.425, 0.38, 0.34, 0.3, 0.284, 0.259, 0.238, 0.22,
            0.203, 0.18, 0.165, 0.148, 0.134, 0.12, 0.109, 0.095, 0.083,
            0.072, 0.065, 0.058, 0.049, 0.042, 0.035, 0.032, 0.028, 0.025,
            0.022, 0.02, 0.018, 0.016, 0.014, 0.013, 0.012, 0.01, 0.009,
            0.008, 0.007, 0.005, 0.004]
#BWG_SI = [round(i*inch,6) for i in BWG_inch]
BWG_SI = [0.0127, 0.011532, 0.010795, 0.009652, 0.008636, 0.00762, 0.007214,
          0.006579, 0.006045, 0.005588, 0.005156, 0.004572, 0.004191, 0.003759,
          0.003404, 0.003048, 0.002769, 0.002413, 0.002108, 0.001829, 0.001651,
          0.001473, 0.001245, 0.001067, 0.000889, 0.000813, 0.000711, 0.000635,
          0.000559, 0.000508, 0.000457, 0.000406, 0.000356, 0.00033, 0.000305,
          0.000254, 0.000229, 0.000203, 0.000178, 0.000127, 0.000102]

wire_schedules = {'BWG': (BWG_integers, BWG_inch, BWG_SI, True),
                 'AWG': (AWG_integers, AWG_inch, AWG_SI, True),
                 'SWG': (SWG_integers, SWG_inch, SWG_SI, True),
                 'MWG': (MWG_integers, MWG_inch, MWG_SI, False),
                 'BSWG': (BSWG_integers, BSWG_inch, BSWG_SI, True),
                 'SSWG': (SSWG_integers, SSWG_inch, SSWG_SI, True)}


def gauge_from_t(t, SI=True, schedule='BWG'):
    r'''Looks up the gauge of a given wire thickness of given schedule.
    Values are all non-linear, and tabulated internally.

    Parameters
    ----------
    t : float
        Thickness, [m]
    SI : bool, optional
        If False, requires that the thickness is given in inches not meters
    schedule : str
        Gauge schedule, one of 'BWG', 'AWG', 'SWG', 'MWG', 'BSWG', or 'SSWG'

    Returns
    -------
    gauge : float-like
        Wire Gauge, [-]

    Notes
    -----
    An internal variable, tol, is used in the selection of the wire gauge. If
    the next smaller wire gauge is within 10% of the difference between it and
    the previous wire gauge, the smaller wire gauge is selected. Accordingly,
    this function can return a gauge with a thickness smaller than desired
    in some circumstances.

    * Birmingham Wire Gauge (BWG) ranges from 0.2 (0.5 inch) to 36 (0.004 inch).
    * American Wire Gauge (AWG) ranges from 0.167 (0.58 inch) to 51 (0.00099
      inch). These are used for electrical wires.
    * Steel Wire Gauge (SWG) ranges from 0.143 (0.49 inch) to 51 (0.0044 inch).
      Also called Washburn & Moen wire gauge, American Steel gauge, Wire Co.
      gauge, and Roebling wire gauge.
    * Music Wire Gauge (MWG) ranges from 0.167 (0.004 inch) to 46 (0.18
      inch). Also called Piano Wire Gauge.
    * British Standard Wire Gage (BSWG) ranges from 0.143 (0.5 inch) to
      51 (0.001 inch). Also called Imperial Wire Gage (IWG).
    * Stub's Steel Wire Gage (SSWG) ranges from 1 (0.227 inch) to 80 (0.013 inch)

    Examples
    --------
    >>> gauge_from_t(.5, SI=False, schedule='BWG')
    0.2

    References
    ----------
    .. [1] Oberg, Erik, Franklin D. Jones, and Henry H. Ryffel. Machinery's
       Handbook. Industrial Press, Incorporated, 2012.
    '''
    tol = 0.1
    # Handle units
    if SI:
        t_inch = round(t/inch, 9) # all schedules are in inches
    else:
        t_inch = t

    # Get the schedule
    try:
        sch_integers, sch_inch, sch_SI, decreasing = wire_schedules[schedule]
    except:
        raise ValueError('Wire gauge schedule not found')

    # Check if outside limits
    sch_max, sch_min = sch_inch[0], sch_inch[-1]
    if t_inch > sch_max:
        raise ValueError('Input thickness is above the largest in the selected schedule')


    # If given thickness is exactly in the index, be happy
    if t_inch in sch_inch:
        gauge = sch_integers[sch_inch.index(t_inch)]

    else:
        for i in range(len(sch_inch)):
            if sch_inch[i] >= t_inch:
                larger = sch_inch[i]
            else:
                break
        if larger == sch_min:
            gauge = sch_min # If t is under the lowest schedule, be happy
        else:
            smaller = sch_inch[i]
            if (t_inch - smaller) <= tol*(larger - smaller):
                gauge = sch_integers[i]
            else:
                gauge = sch_integers[i-1]
    return gauge


def t_from_gauge(gauge, SI=True, schedule='BWG'):
    r'''Looks up the thickness of a given wire gauge of given schedule.
    Values are all non-linear, and tabulated internally.

    Parameters
    ----------
    gauge : float-like
        Wire Gauge, []
    SI : bool, optional
        If False, will return a thickness in inches not meters
    schedule : str
        Gauge schedule, one of 'BWG', 'AWG', 'SWG', 'MWG', 'BSWG', or 'SSWG'

    Returns
    -------
    t : float
        Thickness, [m]

    Notes
    -----

    * Birmingham Wire Gauge (BWG) ranges from 0.2 (0.5 inch) to 36 (0.004 inch).
    * American Wire Gauge (AWG) ranges from 0.167 (0.58 inch) to 51 (0.00099
      inch). These are used for electrical wires.
    * Steel Wire Gauge (SWG) ranges from 0.143 (0.49 inch) to 51 (0.0044 inch).
      Also called Washburn & Moen wire gauge, American Steel gauge, Wire Co.
      gauge, and Roebling wire gauge.
    * Music Wire Gauge (MWG) ranges from 0.167 (0.004 inch) to 46 (0.18
      inch). Also called Piano Wire Gauge.
    * British Standard Wire Gage (BSWG) ranges from 0.143 (0.5 inch) to
      51 (0.001 inch). Also called Imperial Wire Gage (IWG).
    * Stub's Steel Wire Gage (SSWG) ranges from 1 (0.227 inch) to 80 (0.013 inch)

    Examples
    --------
    >>> t_from_gauge(.2, False, 'BWG')
    0.5

    References
    ----------
    .. [1] Oberg, Erik, Franklin D. Jones, and Henry H. Ryffel. Machinery's
       Handbook. Industrial Press, Incorporated, 2012.
    '''
    try:
        sch_integers, sch_inch, sch_SI, decreasing = wire_schedules[schedule]
    except:
        raise ValueError("Wire gauge schedule not found; supported gauges are \
'BWG', 'AWG', 'SWG', 'MWG', 'BSWG', and 'SSWG'.")

    try:
        i = sch_integers.index(gauge)
    except:
        raise ValueError('Input gauge not found in selected schedule')
    if SI:
        return sch_SI[i] # returns thickness in m
    else:
        return sch_inch[i] # returns thickness in inch

