# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/step.ipynb (unless otherwise specified).

__all__ = ['StepBase', 'Step']

# Cell
import torch
import numpy as np

from abc import ABCMeta
from abc import abstractmethod
from typing import Callable
from tqdm import tqdm

# Cell
class StepBase:
    """Defines the interface that all step models here expose."""
    __metaclass__ = ABCMeta

    @abstractmethod
    def batch_fit(self, data_loader: torch.utils.data.DataLoader, epochs: int):
        """Trains the model on a batch of user-item interactions."""
        pass

    @abstractmethod
    def step(self, user: torch.tensor, item: torch.tensor,
             rating: torch.tensor, preference: torch.tensor):
        """Trains the model incrementally."""
        pass

    @abstractmethod
    def predict(self, user: torch.tensor, k: int):
        """Recommends the top-k items to a specific user."""
        pass

    @abstractmethod
    def save(self, path: str):
        """Saves the model parameters to the given path."""
        pass

    @abstractmethod
    def load(self, path: str):
        """Loads the model parameters from a given path."""
        pass

# Cell
class Step(StepBase):
    """Incremental and batch training of recommender systems."""
    def __init__(self, model: torch.nn.Module, objective: Callable,
                 optimizer: Callable, conf_func: Callable = lambda x: torch.tensor(1),
                 device: str = 'cpu'):
        self.model = model.to(device)
        self.objective = objective
        self.optimizer = optimizer
        self.conf_func = conf_func
        self.device = device

        # check if the user has provided user and item embeddings
        assert self.model.user_embeddings, 'User embedding matrix could not be found.'
        assert self.model.item_embeddings, 'Item embedding matrix could not be found.'

    @property
    def user_embeddings(self):
        return self.model.user_embeddings

    @property
    def item_embeddings(self):
        return self.model.item_embeddings

    def batch_fit(self, data_loader: torch.utils.data.DataLoader, epochs: int = 1):
        """Trains the model on a batch of user-item interactions."""
        self.model.train()
        for epoch in range(epochs):
            with tqdm(total=len(data_loader)) as pbar:
                for _, (features, preferences) in enumerate(data_loader):
                    users = features[:, 0].to(self.device)
                    items = features[:, 1].to(self.device)
                    rtngs = features[:, 2].to(self.device)
                    prefs = preferences.to(self.device)

                    preds = self.model(users, items)
                    confs = self.conf_func(rtngs)

                    if hasattr(self.objective, 'weight'):
                        self.objective.weight = confs

                    loss = self.objective(preds, prefs).mean()
                    loss.backward()

                    self.optimizer.step()
                    self.optimizer.zero_grad()

                    pbar.update(1)

    def step(self, user: torch.tensor, item: torch.tensor,
             rating: torch.tensor = None, preference: torch.tensor = None):
        """Trains the model incrementally."""
        self.model.train()

        user = user.to(self.device)
        item = item.to(self.device)
        rtng = rating.to(self.device)
        pref = preference.to(self.device)

        pred = self.model(user, item)
        conf = self.conf_func(rtng)

        if hasattr(self.objective, 'weight'):
            self.objective.weight = conf

        loss = self.objective(pred, pref)
        loss.backward()

        self.optimizer.step()
        self.optimizer.zero_grad()

    def predict(self, user: torch.tensor, k:int = 10) -> torch.tensor:
        """Recommends the top-k items to a specific user."""
        self.model.eval()
        user = user.to(self.device)
        user_embedding = self.user_embeddings(user)
        item_embeddings = self.item_embeddings.weight
        score = item_embeddings @ user_embedding.transpose(0, 1)
        predictions = score.squeeze().argsort()[-k:]
        return predictions.cpu()

    def save(self, path: str):
        """Saves the model parameters to the given path."""
        torch.save(self.model.state_dict(), path)

    def load(self, path: str):
        """Loads the model parameters from a given path."""
        self.model.load_state_dict(torch.load(path))