"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Calendar = exports.CalendarSourceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const calendar_setup_function_1 = require("./calendar-setup-function");
/**
 * The source types for the calendar file.
 */
var CalendarSourceType;
(function (CalendarSourceType) {
    /**
     * The calendar source is an S3 Bucket.
     */
    CalendarSourceType["S3_OBJECT"] = "s3Object";
    /**
     * The calendar source is a local path.
     */
    CalendarSourceType["PATH"] = "path";
})(CalendarSourceType = exports.CalendarSourceType || (exports.CalendarSourceType = {}));
/**
 * The calendar for determining if pipeline stage should be open or closed.
 */
class Calendar {
    constructor() { }
    /**
     * Creates a calendar from a local file.
     */
    static path(options) {
        return new (class extends Calendar {
            _bind(scope) {
                const calendarBody = fs.readFileSync(path.join(options.calendarPath, options.calendarName), { encoding: 'utf-8' });
                const calendar = new CustomResourceCalendar(scope, {
                    sourceType: CalendarSourceType.PATH,
                    calendarBody,
                    calendarName: options.calendarName,
                });
                this.calendarArn = calendar.calendarArn;
                this.calendarName = calendar.calendarName;
                return calendar;
            }
        })();
    }
    /**
     * Creates a calendar from a S3 bucket.
     */
    static s3Location(options) {
        return new (class extends Calendar {
            _bind(scope) {
                const calendar = new CustomResourceCalendar(scope, {
                    sourceType: CalendarSourceType.S3_OBJECT,
                    bucketName: options.bucket.bucketName,
                    calendarName: options.calendarName,
                    roleArn: options.role?.roleArn,
                });
                this.calendarArn = calendar.calendarArn;
                this.calendarName = calendar.calendarName;
                return calendar;
            }
        })();
    }
}
exports.Calendar = Calendar;
_a = JSII_RTTI_SYMBOL_1;
Calendar[_a] = { fqn: "@cdklabs/cdk-codepipeline-extensions.Calendar", version: "0.0.43" };
/**
 * The custom resource for getting the calendar and uploading it to SSM.
 */
class CustomResourceCalendar extends Calendar {
    constructor(scope, options) {
        super();
        this.calendarName = options.calendarName;
        this.calendarArn = aws_cdk_lib_1.Arn.format({
            service: 'ssm',
            resource: 'document',
            resourceName: options.calendarName,
        }, aws_cdk_lib_1.Stack.of(scope));
        const onEvent = new calendar_setup_function_1.CalendarSetupFunction(scope, 'OnEventHandler');
        onEvent.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ssm:CreateDocument',
                'ssm:UpdateDocument',
                'ssm:DeleteDocument',
            ],
            resources: [this.calendarArn],
        }));
        const provider = new custom_resources_1.Provider(scope, 'Provider', {
            onEventHandler: onEvent,
        });
        new aws_cdk_lib_1.CustomResource(scope, 'SSMCalendarCustomResource', {
            serviceToken: provider.serviceToken,
            properties: {
                sourceType: options.sourceType,
                calendarBody: options.calendarBody,
                bucketName: options.bucketName,
                calendarName: options.calendarName,
                roleArn: options.roleArn,
            },
        });
    }
    _bind() { }
}
//# sourceMappingURL=data:application/json;base64,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