# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pydantic_settings']

package_data = \
{'': ['*']}

install_requires = \
['Django>=1.11', 'pydantic[email]>=1.8,<2.0']

extras_require = \
{':python_version < "2.8"': ['typing-extensions>=3.7.4,<5.0.0'],
 'sentry': ['sentry-sdk']}

setup_kwargs = {
    'name': 'django-pydantic-settings',
    'version': '0.6.3',
    'description': 'Manage Django settings with Pydantic.',
    'long_description': '# django-pydantic-settings\n\n## Use pydantic settings management to simplify configuration of Django settings.\n\nVery much a work in progress, but reads the standard DJANGO_SETTINGS_MODULE environment variable (defaulting to pydantic_settings.settings.PydanticSettings) to load a sub-class of pydantic_settings.Settings. All settings (that have been defined in pydantic_settings.Settings) can be overridden with environment variables. A special DatabaseSettings class is used to allow multiple databases to be configured simply with DSNs.\n\nAs of django-pydantic-settings 0.6.0, Django 4.0 is now supported, but, as a result, support for Python 3.6 and 3.7 has been dropped. Python <3.8 can still be used with versions 0.5.0 and lower, and Django 3.2.x and lower; Python 3.6.0 requires django-pydantic-settings <0.4.0. Currently, django-pydantic-settings is tested on Python 3.8, 3.9, and 3.10, and Django 2.2, 3.0, 3.1, 3.2, and 4.0.\n\n## Installation & Setup\n\nInstall django-pydantic-settings:\n\n```\npip install django-pydantic-settings\n```\n\nModify your Django project\'s `manage.py` file to use django-pydantic-settings, it should look something like this:\n\n```python\n#!/usr/bin/env python\n"""Django\'s command-line utility for administrative tasks."""\nimport sys\n\nfrom pydantic_settings import SetUp\n\n\ndef main():\n    """Run administrative tasks."""\n    SetUp().configure()\n\n    try:\n        from django.core.management import execute_from_command_line\n    except ImportError as exc:\n        raise ImportError(\n            "Couldn\'t import Django. Are you sure it\'s installed and "\n            "available on your PYTHONPATH environment variable? Did you "\n            "forget to activate a virtual environment?"\n        ) from exc\n    execute_from_command_line(sys.argv)\n\n\nif __name__ == "__main__":\n    main()\n```\n\nYour `wsgi.py` and/or `asgi.py` files will need to be modified similarly, and look something like this:\n\n```python\nfrom django.core.wsgi import get_wsgi_application\n\nfrom pydantic_settings import SetUp\n\nSetUp().configure()\napplication = get_wsgi_application()\n```\n\nThe `SetUp` class will automatically look for the standard `DJANGO_SETTINGS_MODULE` environment variable, read it, confirm that it points to an existing Python module, and load that module. Your `DJANGO_SETTINGS_MODULE` variable should point to a `pydantic_settings.settings.PydanticSettings` sub-class (though technically any Python class that defines a `dict()` method which returns a Python dictionary of key/value pairs matching the required Django settings will work). Calling the `configure()` method will then use the specified module to configure your project\'s Django settings.\n\nIf your project uses a package to specify multiple different settings classes, simply set `DJANGO_SETTINGS_MODULE` to be the full path to the desired settings class. For example, given the following directory structure:\n\n```\nmy_project/\n├─ settings/\n│  ├─ __init__.py\n│  ├─ base.py\n│  ├─ local.py\n│  ├─ production.py\n├─ my_app/\n\n```\n\nTo use a settings class called `MyLocal` in `local.py` you would set your `DJANGO_SETTINGS_MODULE` to `my_project.settings.local.MyLocal`.\n\n## Required settings\n\nThere are no settings that must be configured in order to use Django with django-pydantic-settings. All of the possible settings defined by Django ([Settings Reference](https://docs.djangoproject.com/en/3.1/ref/settings/)) are configured in the `pydantic_settings.settings.PydanticSettings` class, using their normal default values provided by Django, or a reasonable calculated value. Settings worth thinking about are `ROOT_URLCONF` and `WSGI_APPLICATION`, which, unless otherwise specified, are calculated based on your `DJANGO_SETTINGS_MODULE` assuming that you\'re using the default Django project layout a provided by `django-admin.py startproject`. So, for example, if your `DJANGO_SETINGS_MODULE` is set to `my_awesome_project.settings.PydanticSettingsSubclass`, then `ROOT_URLCONF` and `WSGI_APPLICATION` will be set to `my_awesome_project.urls` and `my_awesome_project.wsgi` respectively. This default behavior can be overridden by simply specifying `ROOT_URLCONF:str = \'the_actual_urlconf\'` and `WSGI_APPLICATION:str = \'the_actual_wsgi_file.application\'` in your `PydanticSettings` sub-class. Alternatively, rather than individually settings the `ROOT_URLCONF` and `WSGI_APPLICATION` settings, you can set `BASE_DIR`, and that will be used instead of `DJANGO_SETTINGS_MODULE`.\n\nThe other setting worth thinking about is `SECRET_KEY`. By default, `SECRET_KEY` is automatically generated using Django\'s own `get_random_secret_key()` function. This will work just fine, though as it will be re-calculated every time your `PydanticSettings` sub-class is instantiated, you should set this to somethign static if you\'re using Django\'s authentication and don\'t want to lose your session every time the server is restarted.\n\n## Database configuration\n\nBy defining multiple `DatabaseDsn` attributes of the `DatabaseSettings` sub-class, you can easily configure one or more database connections with environment variables. Google Cloud SQL database connections from within Google Cloud Run are supported; the DatabaseDsn type will detect and automatically escape DSN strings of the form `postgres://username:password@/cloudsql/project:region:instance/database` so that they can be properly handled. The below example is taken from the test project in this repository, and shows a working multi-database configuration file. In this example, the value for `DJANGO_SETTINGS_MODULE` should be set, as below, to `settings_test.database_settings.TestSettings`.\n\n```python\nclass Databases(DatabaseSettings):\n    DEFAULT: DatabaseDsn = Field(env="DATABASE_URL")\n    SECONDARY: DatabaseDsn = Field(env="SECONDARY_DATABASE_URL")\n\n\nclass TestSettings(PydanticSettings):\n    DATABASES: Databases = Field({})\n```\n\n```python\n❯ DJANGO_SETTINGS_MODULE=settings_test.database_settings.TestSettings DATABASE_URL=postgres://username:password@/cloudsql/project:region:instance/database SECONDARY_DATABASE_URL=sqlite:///foo poetry run python manage.py shell\nPython 3.10.2 (main, Feb  2 2022, 06:19:27) [Clang 13.0.0 (clang-1300.0.29.3)] on darwin\nType "help", "copyright", "credits" or "license" for more information.\n(InteractiveConsole)\n>>> from rich import print\n>>> from django.conf import settings\n>>> print(settings.DATABASES)\n{\n    \'default\': {\n        \'NAME\': \'database\',\n        \'USER\': \'username\',\n        \'PASSWORD\': \'password\',\n        \'HOST\': \'/cloudsql/project:region:instance\',\n        \'PORT\': \'\',\n        \'CONN_MAX_AGE\': 0,\n        \'ENGINE\': \'django.db.backends.postgresql\'\n    },\n    \'secondary\': {\'NAME\': \'foo\', \'USER\': \'\', \'PASSWORD\': \'\', \'HOST\': \'\', \'PORT\': \'\', \'CONN_MAX_AGE\': 0, \'ENGINE\': \'django.db.backends.sqlite3\'}\n}\n>>>\n```\n\n## Sentry configuration\n\ndjango-pydantic-settings provides built-in functionality for configuring your Django project to use [Sentry](https://sentry.io/). The simplest way to use this is to inherit from `pydantic_settings.sentry.SentrySettings` rather than `pydantic_settings.settings.PydanticSettings`. This adds the setting `SENTRY_DSN`, which uses the `pydantic_settings.sentry.SentryDsn` type. This will automatically be set according to the `DJANGO_SENTRY_DSN` environment variable, and expects a Sentry DSN (obviously). It validates that the provided DSN is a valid URL, and then automatically initializes the Sentry SDK using the built-in DjangoIntegration. Using this functionality required `sentry-sdk` to be installed, which will be included automatically if you install `django-pydantic-settings[sentry]`.\n',
    'author': 'Josh Ourisman',
    'author_email': 'me@josho.io',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/joshourisman/django-pydantic-settings',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
